<?php

use App\Http\Controllers\AccountController;
use App\Http\Controllers\AccountReportsController;
use App\Http\Controllers\AccountTypeController;
use App\Http\Controllers\Api\ReportController as ApiReportController;
use App\Http\Controllers\AuthentificationController;
use App\Http\Controllers\BackUpController;
use App\Http\Controllers\BarcodeController;
use App\Http\Controllers\BrandController;
use App\Http\Controllers\BusinessController;
use App\Http\Controllers\BusinessLocationController;
use App\Http\Controllers\ContactController;
use App\Http\Controllers\CustomerGroupController;
use App\Http\Controllers\CashRegisterController;
use App\Http\Controllers\CombinedPurchaseReturnController;
use App\Http\Controllers\ExpenseCategoryController;
use App\Http\Controllers\ExpenseController;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\ImportOpeningStockController;
use App\Http\Controllers\ImportProductsController;
use App\Http\Controllers\ImportSalesController;
use App\Http\Controllers\InventoryController;
use App\Http\Controllers\InvoiceLayoutController;
use App\Http\Controllers\InvoiceSchemeController;
use App\Http\Controllers\LabelsController;
use App\Http\Controllers\LocationSettingsController;
use App\Http\Controllers\ManageUserController;
use App\Http\Controllers\NotificationController;
use App\Http\Controllers\NotificationTemplateController;
use App\Http\Controllers\OpeningStockController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\PurchaseController;
use App\Http\Controllers\ReportController;
use App\Http\Controllers\RoleController;
use App\Http\Controllers\SalesCommissionAgentController;
use App\Http\Controllers\SellController;
use App\Http\Controllers\SellingPriceGroupController;
use App\Http\Controllers\SellPosController;
use App\Http\Controllers\SellReturnController;
use App\Http\Controllers\StockAdjustmentController;
use App\Http\Controllers\StockTransferController;
use App\Http\Controllers\TaxonomyController;
use App\Http\Controllers\TransactionPaymentController;
use App\Http\Controllers\TypesOfServiceController;
use App\Http\Controllers\UnitController;
use App\Http\Controllers\UserController;
use App\Http\Controllers\VariationTemplateController;
use App\Http\Controllers\WarrantyController;
use App\Models\InvoiceLayout;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Validator;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/


Route::get('/business/register', [BusinessController::class, 'getRegister'])->name('business.getRegister');
Route::post('/business/register', [BusinessController::class, 'postRegister'])->name('business.postRegister');
Route::post('/business/register/check-username', [BusinessController::class, 'postCheckUsername'])->name('business.postCheckUsername');
Route::post('/business/register/check-email', [BusinessController::class, 'postCheckEmail'])->name('business.postCheckEmail');
Route::get('/invoice/{token}', [SellPosController::class, 'showInvoice'])
    ->name('show_invoice');


// Routes publiques
Route::post('/login', [AuthentificationController::class, 'login']);
Route::post('/login-flexible', [AuthentificationController::class, 'loginWithEmailOrUsername']);

// Routes protégées par Sanctum
Route::middleware('auth:sanctum')->group(function () {
    Route::post('/logout', [AuthentificationController::class, 'logout']);
    Route::post('/logout-all', [AuthentificationController::class, 'logoutAll']);
    Route::get('/me', [AuthentificationController::class, 'me']);

    // Vos autres routes protégées...
    Route::get('/business/settings', [AuthentificationController::class, 'getBusinessSettings']);

    // Gestion des rôles et permissions
    Route::apiResource('roles',RoleController::class);
    Route::get('roles/permissions', [ManageUserController::class, 'getPermissions']);
    Route::post('/users/update-password', [UserController::class, 'updatePassword']);
    // Récupérer le profil de l'utilisateur connecté
    Route::get('/users/profil', [UserController::class, 'getProfile']);

    // Mettre à jour le profil de l'utilisateur
    Route::put('/users/profil/update', [UserController::class, 'updateProfile']);
    // Gestion des Utilisateurs et Agent de Commission
    Route::apiResource('users', ManageUserController::class);
    Route::get('users/dropdown', [ManageUserController::class, 'getUsersDropdown']);

    Route::apiResource('sales-commission-agents', SalesCommissionAgentController::class);
    Route::get('sales-commission-agents-datatable', [SalesCommissionAgentController::class, 'datatable']);
    Route::get('sales-commission-agents-dropdown', [SalesCommissionAgentController::class, 'dropdown']);

    // Gestion des Patients et fournisseurs
    Route::apiResource('contacts', ContactController::class);
    Route::get('contacts-customers', [ContactController::class, 'getCustomers']);
    Route::patch('contacts/{id}/status', [ContactController::class, 'updateStatus']);
    Route::get('contacts/{id}/ledger', [ContactController::class, 'getLedger']);
    Route::post('contacts/check-id', [ContactController::class, 'checkContactId']);
    // Gestion des Groupe de Clients
    Route::apiResource('customer-groups', CustomerGroupController::class);
    Route::get('customer-groups-dropdown', [CustomerGroupController::class, 'dropdown']);
    Route::get('customer-groups-statistics', [CustomerGroupController::class, 'statistics']);
    Route::get('customer-groups-datatable', [CustomerGroupController::class, 'datatable']);

    // Product API Routes
    Route::prefix('products')->group(function () {
        Route::get('/', [ProductController::class, 'index']);
        Route::post('/', [ProductController::class, 'store']);
        Route::get('/{id}', [ProductController::class, 'show']);
        Route::put('/{id}', [ProductController::class, 'update']);
        Route::patch('/{id}', [ProductController::class, 'update']);
        Route::delete('/{id}', [ProductController::class, 'destroy']);

        // Routes utilitaires pour les produits
        Route::get('/search/products', [ProductController::class, 'getProducts']);
        Route::get('/subcategories/get', [ProductController::class, 'getSubCategories']);
        Route::get('/sku/check', [ProductController::class, 'checkSku']);
        Route::get('/types/list', [ProductController::class, 'getProductTypes']);
        Route::get('/form/data', [ProductController::class, 'getFormData']);
    });

    Route::prefix('api/import/products')->group(function () {
        Route::get('/', [ImportProductsController::class, 'index']);
        Route::post('/', [ImportProductsController::class, 'store']);
        Route::get('/template', [ImportProductsController::class, 'getTemplate']);
        Route::post('/validate', [ImportProductsController::class, 'validateImportFile']);
    }); // Import Products API Routes

    // Category API Routes
    Route::prefix('categories')->group(function () {
        Route::get('/', [TaxonomyController::class, 'index']);
        Route::post('/', [TaxonomyController::class, 'store']);
        Route::get('/{id}', [TaxonomyController::class, 'show']);
        Route::put('/{id}', [TaxonomyController::class, 'update']);
        Route::patch('/{id}', [TaxonomyController::class, 'update']);
        Route::delete('/{id}', [TaxonomyController::class, 'destroy']);

        // Routes utilitaires pour les catégories
        Route::get('/dropdown/list', [TaxonomyController::class, 'getForDropdown']);
        Route::get('/subcategories/list', [TaxonomyController::class, 'getSubcategories']);
        Route::get('/tree/structure', [TaxonomyController::class, 'getCategoryTree']);
    });

    // Brand API Routes
    Route::prefix('brands')->group(function () {
        Route::get('/', [BrandController::class, 'index']);
        Route::post('/', [BrandController::class, 'store']);
        Route::get('/{id}', [BrandController::class, 'show']);
        Route::put('/{id}', [BrandController::class, 'update']);
        Route::patch('/{id}', [BrandController::class, 'update']);
        Route::delete('/{id}', [BrandController::class, 'destroy']);

        // Routes utilitaires pour les marques
        Route::get('/dropdown/list', [BrandController::class, 'getForDropdown']);
        Route::get('/all/list', [BrandController::class, 'getAllBrands']);
    });

    // Unit API Routes
    Route::prefix('units')->group(function () {
        Route::get('/', [UnitController::class, 'index']);
        Route::post('/', [UnitController::class, 'store']);
        Route::get('/{id}', [UnitController::class, 'show']);
        Route::put('/{id}', [UnitController::class, 'update']);
        Route::patch('/{id}', [UnitController::class, 'update']);
        Route::delete('/{id}', [UnitController::class, 'destroy']);

        // Routes utilitaires pour les unités
        Route::get('/dropdown/list', [UnitController::class, 'getForDropdown']);
        Route::get('/sub-units/list', [UnitController::class, 'getSubUnits']);
        Route::get('/all/list', [UnitController::class, 'getAllUnits']);
    });

    // Variation Template API Routes
    Route::prefix('variation-templates')->group(function () {
        Route::get('/', [VariationTemplateController::class, 'index']);
        Route::post('/', [VariationTemplateController::class, 'store']);
        Route::get('/{id}', [VariationTemplateController::class, 'show']);
        Route::put('/{id}', [VariationTemplateController::class, 'update']);
        Route::patch('/{id}', [VariationTemplateController::class, 'update']);
        Route::delete('/{id}', [VariationTemplateController::class, 'destroy']);

        // Routes utilitaires pour les templates de variation
        Route::get('/dropdown/list', [VariationTemplateController::class, 'getForDropdown']);
        Route::get('/with-values/get', [VariationTemplateController::class, 'getTemplateWithValues']);
        Route::get('/all/list', [VariationTemplateController::class, 'getAllVariationTemplates']);
        Route::post('/{id}/values', [VariationTemplateController::class, 'addVariationValue']);
        Route::delete('/{templateId}/values/{valueId}', [VariationTemplateController::class, 'removeVariationValue']);
    });

    // Warranties
    Route::apiResource('warranties', WarrantyController::class);

    //Gestion des purchases
    Route::apiResource('purchases', PurchaseController::class);
    // Additional purchase endpoints
    Route::get('/suppliers/list', [PurchaseController::class, 'getSuppliers'])->name('api.purchases.suppliers');
    Route::get('/products/list', [PurchaseController::class, 'getProducts'])->name('api.purchases.products');
    Route::post('/check-ref-number', [PurchaseController::class, 'checkRefNumber'])->name('api.purchases.check-ref');
    Route::patch('/{id}/status', [PurchaseController::class, 'updateStatus'])->name('api.purchases.update-status');

    // Stock Transfer API Routes
    Route::prefix('stock-transfers')->group(function () {
        Route::get('/', [StockTransferController::class, 'index'])->name('api.stock-transfers.index');
        Route::post('/', [StockTransferController::class, 'store'])->name('api.stock-transfers.store');
        Route::get('/{id}', [StockTransferController::class, 'show'])->name('api.stock-transfers.show');
        Route::put('/{id}', [StockTransferController::class, 'update'])->name('api.stock-transfers.update');
        Route::delete('/{id}', [StockTransferController::class, 'destroy'])->name('api.stock-transfers.destroy');
        Route::patch('/{id}/status', [StockTransferController::class, 'updateStatus'])->name('api.stock-transfers.update-status');
    });

    // Stock Adjustment API Routes

    Route::apiResource('stock-adjustments', StockAdjustmentController::class);
    Route::post('stock-adjustments/get-product-row', [StockAdjustmentController::class, 'getProductRow']);
    Route::post('stock-adjustments/remove-expired-stock/{purchase_line_id}', [StockAdjustmentController::class, 'removeExpiredStock']);

    // Gestion des depenses
    // Expenses
    Route::apiResource('expenses', ExpenseController::class);

    // Expense Categories  
    Route::apiResource('expense-categories', ExpenseCategoryController::class);

    // Account API Routes
    Route::group(['prefix' => 'accounts', 'as' => 'accounts.'], function () {
        Route::get('/', [AccountController::class, 'index'])->name('index');
        Route::get('/create', [AccountController::class, 'create'])->name('create');
        Route::post('/', [AccountController::class, 'store'])->name('store');
        Route::get('/{id}', [AccountController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [AccountController::class, 'edit'])->name('edit');
        Route::put('/{id}', [AccountController::class, 'update'])->name('update');
        Route::patch('/{id}', [AccountController::class, 'update'])->name('patch');

        // Account specific actions
        Route::post('/{id}/close', [AccountController::class, 'close'])->name('close');
        Route::post('/{id}/activate', [AccountController::class, 'activate'])->name('activate');
        Route::get('/{id}/balance', [AccountController::class, 'getAccountBalance'])->name('balance');

        // Fund Transfer
        Route::get('/{id}/fund-transfer', [AccountController::class, 'getFundTransfer'])->name('fund-transfer.get');
        Route::post('/fund-transfer', [AccountController::class, 'postFundTransfer'])->name('fund-transfer.post');

        // Deposit
        Route::get('/{id}/deposit', [AccountController::class, 'getDeposit'])->name('deposit.get');
        Route::post('/deposit', [AccountController::class, 'postDeposit'])->name('deposit.post');

        // Account Transactions
        Route::delete('/transactions/{id}', [AccountController::class, 'destroyAccountTransaction'])->name('transactions.destroy');

        // Cash Flow
        Route::get('/reports/cash-flow', [AccountController::class, 'cashFlow'])->name('cash-flow');
    });

    // Account Reports API Routes  
    Route::group(['prefix' => 'account-reports', 'as' => 'account-reports.'], function () {
        Route::get('/balance-sheet', [AccountReportsController::class, 'balanceSheet'])->name('balance-sheet');
        Route::get('/trial-balance', [AccountReportsController::class, 'trialBalance'])->name('trial-balance');
        Route::get('/payment-account-report', [AccountReportsController::class, 'paymentAccountReport'])->name('payment-account-report');

        // Link Account with Payment
        Route::get('/link-account/{id}', [AccountReportsController::class, 'getLinkAccount'])->name('link-account.get');
        Route::post('/link-account', [AccountReportsController::class, 'postLinkAccount'])->name('link-account.post');
    });

    // Account Types API Routes
    Route::group(['prefix' => 'account-types', 'as' => 'account-types.'], function () {
        Route::get('/', [AccountTypeController::class, 'index'])->name('index');
        Route::get('/create', [AccountTypeController::class, 'create'])->name('create');
        Route::post('/', [AccountTypeController::class, 'store'])->name('store');
        Route::get('/{id}', [AccountTypeController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [AccountTypeController::class, 'edit'])->name('edit');
        Route::put('/{id}', [AccountTypeController::class, 'update'])->name('update');
        Route::patch('/{id}', [AccountTypeController::class, 'update'])->name('patch');
        Route::delete('/{id}', [AccountTypeController::class, 'destroy'])->name('destroy');
    });

    // Routes pour BackUp
    Route::prefix('backups')->name('backups.')->group(function () {
        Route::get('/', [BackUpController::class, 'index'])->name('index');
        Route::get('/create', [BackUpController::class, 'create'])->name('create');
        Route::post('/', [BackUpController::class, 'store'])->name('store');
        Route::get('/{filename}', [BackUpController::class, 'show'])->name('show');
        Route::get('/{filename}/edit', [BackUpController::class, 'edit'])->name('edit');
        Route::put('/{filename}', [BackUpController::class, 'update'])->name('update');
        Route::patch('/{filename}', [BackUpController::class, 'update'])->name('patch');
        Route::delete('/{filename}', [BackUpController::class, 'destroy'])->name('destroy');
        Route::get('/{filename}/download', [BackUpController::class, 'download'])->name('download');
    });

    // Routes pour Barcode
    Route::prefix('barcodes')->name('barcodes.')->group(function () {
        Route::get('/', [BarcodeController::class, 'index'])->name('index');
        Route::get('/create', [BarcodeController::class, 'create'])->name('create');
        Route::post('/', [BarcodeController::class, 'store'])->name('store');
        Route::get('/{id}', [BarcodeController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [BarcodeController::class, 'edit'])->name('edit');
        Route::put('/{id}', [BarcodeController::class, 'update'])->name('update');
        Route::patch('/{id}', [BarcodeController::class, 'update'])->name('patch');
        Route::delete('/{id}', [BarcodeController::class, 'destroy'])->name('destroy');
        Route::post('/{id}/set-default', [BarcodeController::class, 'setDefault'])->name('set-default');
    });

    // Routes pour Business Location
    Route::prefix('business-locations')->name('business-locations.')->group(function () {
        Route::get('/', [BusinessLocationController::class, 'index'])->name('index');
        Route::get('/create', [BusinessLocationController::class, 'create'])->name('create');
        Route::post('/', [BusinessLocationController::class, 'store'])->name('store');
        Route::get('/{id}', [BusinessLocationController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [BusinessLocationController::class, 'edit'])->name('edit');
        Route::put('/{id}', [BusinessLocationController::class, 'update'])->name('update');
        Route::patch('/{id}', [BusinessLocationController::class, 'update'])->name('patch');
        Route::delete('/{id}', [BusinessLocationController::class, 'destroy'])->name('destroy');
        Route::post('/check-location-id', [BusinessLocationController::class, 'checkLocationId'])->name('check-location-id');
        Route::post('/{id}/toggle-status', [BusinessLocationController::class, 'activateDeactivateLocation'])->name('toggle-status');
    });


    // Routes pour Cash Register
    Route::prefix('cash-registers')->name('cash-registers.')->group(function () {

        // Routes CRUD standard
        Route::get('/', [CashRegisterController::class, 'index'])->name('index');
        Route::get('/create', [CashRegisterController::class, 'create'])->name('create');
        Route::post('/', [CashRegisterController::class, 'store'])->name('store');
        Route::get('/{id}', [CashRegisterController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [CashRegisterController::class, 'edit'])->name('edit');
        Route::put('/{id}', [CashRegisterController::class, 'update'])->name('update');
        Route::patch('/{id}', [CashRegisterController::class, 'update'])->name('patch');
        Route::delete('/{id}', [CashRegisterController::class, 'destroy'])->name('destroy');

        // Routes spéciales pour la gestion des caisses
        Route::get('/current/details', [CashRegisterController::class, 'getCurrentRegisterDetails'])->name('current.details');
        Route::get('/close/form', [CashRegisterController::class, 'getCloseRegisterForm'])->name('close.form');
        Route::post('/close', [CashRegisterController::class, 'closeRegister'])->name('close');
        Route::get('/status/check', [CashRegisterController::class, 'getRegisterStatus'])->name('status.check');
        Route::get('/statistics/overview', [CashRegisterController::class, 'getStatistics'])->name('statistics.overview');
    });

    // Routes pour Purchase Returns
    Route::prefix('purchase-returns')->name('purchase-returns.')->group(function () {

        // Routes CRUD standard
        Route::get('/', [CombinedPurchaseReturnController::class, 'index'])->name('index');
        Route::get('/create', [CombinedPurchaseReturnController::class, 'create'])->name('create');
        Route::post('/', [CombinedPurchaseReturnController::class, 'store'])->name('store');
        Route::get('/{id}', [CombinedPurchaseReturnController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [CombinedPurchaseReturnController::class, 'edit'])->name('edit');
        Route::put('/{id}', [CombinedPurchaseReturnController::class, 'update'])->name('update');
        Route::patch('/{id}', [CombinedPurchaseReturnController::class, 'update'])->name('patch');
        Route::delete('/{id}', [CombinedPurchaseReturnController::class, 'destroy'])->name('destroy');

        // Routes spéciales
        Route::post('/get-product-row', [CombinedPurchaseReturnController::class, 'getProductRow'])->name('get-product-row');
        Route::get('/products/{location_id}', [CombinedPurchaseReturnController::class, 'getProductsByLocation'])->name('products.by-location');
        Route::get('/statistics/overview', [CombinedPurchaseReturnController::class, 'getStatistics'])->name('statistics.overview');
    });

    // Dashboard Routes
    Route::prefix('dashboard')->group(function () {
        Route::get('/', [HomeController::class, 'index']);
        Route::get('/due-contacts', [HomeController::class, 'getDueContacts']);
        Route::get('/totals', [HomeController::class, 'getTotals']);
        Route::get('/stock-alerts', [HomeController::class, 'getProductStockAlert']);
        Route::get('/purchase-payment-dues', [HomeController::class, 'getPurchasePaymentDues']);
        Route::get('/sales-payment-dues', [HomeController::class, 'getSalesPaymentDues']);
        Route::get('/notifications', [HomeController::class, 'loadMoreNotifications']);
        Route::get('/unread-notifications-count', [HomeController::class, 'getTotalUnreadNotifications']);
    });

    // Import Opening Stock Routes
    Route::apiResource('import-opening-stocks', ImportOpeningStockController::class);

    // Import Sales Routes
    Route::prefix('import-sales')->group(function () {
        Route::get('/', [ImportSalesController::class, 'index']);
        Route::post('/preview', [ImportSalesController::class, 'preview']);
        Route::post('/', [ImportSalesController::class, 'store']);
        Route::delete('/revert/{batch}', [ImportSalesController::class, 'revertSaleImport']);
    });

    // Sales Management Routes
    Route::apiResource('sales', ImportSalesController::class, [
        'except' => ['index', 'store'] // These are handled by import-sales routes
    ]);

    // =====================================================
    // INVENTORY API ROUTES
    // =====================================================
    Route::prefix('inventories')->name('api.inventories.')->group(function () {

        // Routes RESTful de base
        Route::get('/', [InventoryController::class, 'index'])->name('index');
        Route::get('/create', [InventoryController::class, 'create'])->name('create');
        Route::post('/{location_id}', [InventoryController::class, 'store'])->name('store');
        Route::get('/{id}', [InventoryController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [InventoryController::class, 'edit'])->name('edit');
        Route::put('/{id}', [InventoryController::class, 'update'])->name('update');
        Route::patch('/{id}', [InventoryController::class, 'update'])->name('patch');
        Route::delete('/{id}', [InventoryController::class, 'destroy'])->name('destroy');

        // Routes spécifiques
        Route::get('/location/{location_id}/products', [InventoryController::class, 'createprod'])->name('createprod');
        Route::post('/products/by-location', [InventoryController::class, 'products'])->name('products');
        Route::post('/{location_id}/import', [InventoryController::class, 'import'])->name('import');
        Route::get('/{location_id}/export', [InventoryController::class, 'export'])->name('export');
    });

    // =====================================================
    // ROUTES CRUD DE BASE POUR INVOICE LAYOUTS
    // =====================================================
    Route::prefix('invoice-layouts')->name('api.invoice-layouts.')->group(function () {

        // Routes RESTful standard
        Route::get('/', [InvoiceLayoutController::class, 'index'])->name('index');
        Route::get('/create', [InvoiceLayoutController::class, 'create'])->name('create');
        Route::post('/', [InvoiceLayoutController::class, 'store'])->name('store');
        Route::get('/{id}', [InvoiceLayoutController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [InvoiceLayoutController::class, 'edit'])->name('edit');
        Route::put('/{id}', [InvoiceLayoutController::class, 'update'])->name('update');
        Route::patch('/{id}', [InvoiceLayoutController::class, 'update'])->name('patch');
        Route::delete('/{id}', [InvoiceLayoutController::class, 'destroy'])->name('destroy');

        // =====================================================
        // ROUTES SPÉCIFIQUES POUR LA GESTION DES LAYOUTS
        // =====================================================

        // Gestion du layout par défaut
        Route::post('/{id}/set-default', [InvoiceLayoutController::class, 'setDefault'])->name('set-default');
        Route::get('/default/current', function () {
            $business_id = auth()->user()->business_id;
            $defaultLayout = InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();

            if (!$defaultLayout) {
                return response()->json([
                    'success' => false,
                    'message' => 'Aucun layout par défaut défini.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $defaultLayout,
                'message' => 'Layout par défaut récupéré avec succès.'
            ]);
        })->name('get-default');

        // Duplication de layout
        Route::post('/{id}/duplicate', [InvoiceLayoutController::class, 'duplicate'])->name('duplicate');

        // Prévisualisation
        Route::get('/{id}/preview', [InvoiceLayoutController::class, 'preview'])->name('preview');
        Route::post('/{id}/preview', [InvoiceLayoutController::class, 'preview'])->name('preview-post');

        // Import/Export de configuration
        Route::get('/{id}/export-config', [InvoiceLayoutController::class, 'exportConfig'])->name('export-config');
        Route::post('/import-config', [InvoiceLayoutController::class, 'importConfig'])->name('import-config');

        // Validation de configuration
        Route::post('/validate-config', [InvoiceLayoutController::class, 'validateConfig'])->name('validate-config');

        // Réinitialisation aux valeurs par défaut
        Route::post('/{id}/reset-to-default', [InvoiceLayoutController::class, 'resetToDefault'])->name('reset-to-default');

        // Statistiques
        Route::get('/statistics/overview', [InvoiceLayoutController::class, 'statistics'])->name('statistics');

        // =====================================================
        // ROUTES POUR LES DESIGNS ET CONFIGURATIONS
        // =====================================================

        // Designs disponibles
        Route::get('/designs/available', [InvoiceLayoutController::class, 'getDesigns'])->name('designs');

        // Configuration par design
        Route::get('/designs/{design}/default-config', function ($design) {
            $controller = new InvoiceLayoutController(new \App\Utils\Util());
            $reflection = new \ReflectionClass($controller);
            $method = $reflection->getMethod('getDefaultConfigForDesign');
            $method->setAccessible(true);

            try {
                $config = $method->invoke($controller, $design);
                return response()->json([
                    'success' => true,
                    'data' => $config,
                    'message' => 'Configuration par défaut pour ' . $design . ' récupérée.'
                ]);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Design non supporté.'
                ], 400);
            }
        })->name('design-default-config');

        // =====================================================
        // ROUTES POUR LA GESTION DES CHAMPS PERSONNALISÉS
        // =====================================================

        // Gérer les champs personnalisés produits
        Route::get('/{id}/custom-fields/products', function ($id) {
            $business_id = auth()->user()->business_id;
            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return response()->json(['success' => false, 'message' => 'Layout non trouvé.'], 404);
            }

            $customFields = json_decode($layout->product_custom_fields, true) ?? [];

            return response()->json([
                'success' => true,
                'data' => $customFields,
                'message' => 'Champs personnalisés produits récupérés.'
            ]);
        })->name('custom-fields-products');

        Route::put('/{id}/custom-fields/products', function (\Illuminate\Http\Request $request, $id) {
            $business_id = auth()->user()->business_id;
            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return response()->json(['success' => false, 'message' => 'Layout non trouvé.'], 404);
            }

            $layout->update([
                'product_custom_fields' => json_encode($request->custom_fields ?? [])
            ]);

            return response()->json([
                'success' => true,
                'data' => $layout,
                'message' => 'Champs personnalisés produits mis à jour.'
            ]);
        })->name('update-custom-fields-products');

        // Gérer les champs personnalisés contacts
        Route::get('/{id}/custom-fields/contacts', function ($id) {
            $business_id = auth()->user()->business_id;
            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return response()->json(['success' => false, 'message' => 'Layout non trouvé.'], 404);
            }

            $customFields = json_decode($layout->contact_custom_fields, true) ?? [];

            return response()->json([
                'success' => true,
                'data' => $customFields,
                'message' => 'Champs personnalisés contacts récupérés.'
            ]);
        })->name('custom-fields-contacts');

        Route::put('/{id}/custom-fields/contacts', function (\Illuminate\Http\Request $request, $id) {
            $business_id = auth()->user()->business_id;
            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return response()->json(['success' => false, 'message' => 'Layout non trouvé.'], 404);
            }

            $layout->update([
                'contact_custom_fields' => json_encode($request->custom_fields ?? [])
            ]);

            return response()->json([
                'success' => true,
                'data' => $layout,
                'message' => 'Champs personnalisés contacts mis à jour.'
            ]);
        })->name('update-custom-fields-contacts');

        // Gérer les champs personnalisés locations
        Route::get('/{id}/custom-fields/locations', function ($id) {
            $business_id = auth()->user()->business_id;
            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return response()->json(['success' => false, 'message' => 'Layout non trouvé.'], 404);
            }

            $customFields = json_decode($layout->location_custom_fields, true) ?? [];

            return response()->json([
                'success' => true,
                'data' => $customFields,
                'message' => 'Champs personnalisés locations récupérés.'
            ]);
        })->name('custom-fields-locations');

        Route::put('/{id}/custom-fields/locations', function (\Illuminate\Http\Request $request, $id) {
            $business_id = auth()->user()->business_id;
            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return response()->json(['success' => false, 'message' => 'Layout non trouvé.'], 404);
            }

            $layout->update([
                'location_custom_fields' => json_encode($request->custom_fields ?? [])
            ]);

            return response()->json([
                'success' => true,
                'data' => $layout,
                'message' => 'Champs personnalisés locations mis à jour.'
            ]);
        })->name('update-custom-fields-locations');

        // =====================================================
        // ROUTES POUR LA GESTION DES EN-TÊTES DE TAXES
        // =====================================================

        Route::get('/{id}/tax-headings', function ($id) {
            $business_id = auth()->user()->business_id;
            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return response()->json(['success' => false, 'message' => 'Layout non trouvé.'], 404);
            }

            $taxHeadings = json_decode($layout->table_tax_headings, true) ?? ['', '', '', ''];

            return response()->json([
                'success' => true,
                'data' => $taxHeadings,
                'message' => 'En-têtes de taxes récupérés.'
            ]);
        })->name('tax-headings');

        Route::put('/{id}/tax-headings', function (\Illuminate\Http\Request $request, $id) {
            $validator = Validator::make($request->all(), [
                'tax_headings' => 'required|array|max:4',
                'tax_headings.*' => 'string|max:50'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Données de validation incorrectes',
                    'data' => $validator->errors()
                ], 422);
            }

            $business_id = auth()->user()->business_id;
            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return response()->json(['success' => false, 'message' => 'Layout non trouvé.'], 404);
            }

            $layout->update([
                'table_tax_headings' => json_encode($request->tax_headings)
            ]);

            return response()->json([
                'success' => true,
                'data' => $layout,
                'message' => 'En-têtes de taxes mis à jour.'
            ]);
        })->name('update-tax-headings');

        // =====================================================
        // ROUTES POUR LA GESTION DES LOGOS
        // =====================================================

        Route::post('/{id}/upload-logo', function (\Illuminate\Http\Request $request, $id) {
            $validator = Validator::make($request->all(), [
                'logo' => 'required|image|mimes:jpeg,png,gif|max:1000'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Fichier logo invalide',
                    'data' => $validator->errors()
                ], 422);
            }

            $business_id = auth()->user()->business_id;
            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return response()->json(['success' => false, 'message' => 'Layout non trouvé.'], 404);
            }

            try {
                $commonUtil = new \App\Utils\Util();
                $logo_name = $commonUtil->uploadFile($request, 'logo', 'invoice_logos', 'image');

                if (!empty($logo_name)) {
                    // Delete old logo if exists
                    if (!empty($layout->logo)) {
                        $oldLogoPath = public_path('storage/invoice_logos/' . $layout->logo);
                        if (file_exists($oldLogoPath)) {
                            unlink($oldLogoPath);
                        }
                    }

                    $layout->update(['logo' => $logo_name]);

                    return response()->json([
                        'success' => true,
                        'data' => [
                            'logo' => $logo_name,
                            'logo_url' => asset('storage/invoice_logos/' . $logo_name)
                        ],
                        'message' => 'Logo uploadé avec succès.'
                    ]);
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'Erreur lors de l\'upload du logo.'
                    ], 500);
                }
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erreur lors de l\'upload',
                    'data' => [$e->getMessage()]
                ], 500);
            }
        })->name('upload-logo');

        Route::delete('/{id}/remove-logo', function ($id) {
            $business_id = auth()->user()->business_id;
            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return response()->json(['success' => false, 'message' => 'Layout non trouvé.'], 404);
            }

            if (!empty($layout->logo)) {
                $logoPath = public_path('storage/invoice_logos/' . $layout->logo);
                if (file_exists($logoPath)) {
                    unlink($logoPath);
                }

                $layout->update(['logo' => null]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Logo supprimé avec succès.'
            ]);
        })->name('remove-logo');

        // =====================================================
        // ROUTES POUR LES ACTIONS EN MASSE
        // =====================================================

        Route::post('/bulk-actions', function (\Illuminate\Http\Request $request) {
            $validator = Validator::make($request->all(), [
                'action' => 'required|string|in:delete,duplicate,export',
                'layout_ids' => 'required|array|min:1',
                'layout_ids.*' => 'integer|exists:invoice_layouts,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Données de validation incorrectes',
                    'data' => $validator->errors()
                ], 422);
            }

            $business_id = auth()->user()->business_id;
            $action = $request->action;
            $layoutIds = $request->layout_ids;

            $layouts = InvoiceLayout::where('business_id', $business_id)
                ->whereIn('id', $layoutIds)
                ->get();

            if ($layouts->count() !== count($layoutIds)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Certains layouts n\'ont pas été trouvés.'
                ], 404);
            }

            try {
                $results = [];

                switch ($action) {
                    case 'delete':
                        foreach ($layouts as $layout) {
                            if ($layout->is_default) {
                                $results[] = [
                                    'id' => $layout->id,
                                    'name' => $layout->name,
                                    'success' => false,
                                    'message' => 'Layout par défaut non supprimable'
                                ];
                                continue;
                            }

                            $layout->delete();
                            $results[] = [
                                'id' => $layout->id,
                                'name' => $layout->name,
                                'success' => true,
                                'message' => 'Supprimé avec succès'
                            ];
                        }
                        break;

                    case 'duplicate':
                        foreach ($layouts as $layout) {
                            $duplicate = $layout->replicate();
                            $duplicate->name = $layout->name . ' - Copie';
                            $duplicate->is_default = 0;
                            $duplicate->save();

                            $results[] = [
                                'id' => $layout->id,
                                'name' => $layout->name,
                                'duplicate_id' => $duplicate->id,
                                'success' => true,
                                'message' => 'Dupliqué avec succès'
                            ];
                        }
                        break;

                    case 'export':
                        $exportData = [];
                        foreach ($layouts as $layout) {
                            $layoutData = $layout->toArray();
                            unset(
                                $layoutData['id'],
                                $layoutData['business_id'],
                                $layoutData['created_at'],
                                $layoutData['updated_at']
                            );
                            $exportData[] = $layoutData;

                            $results[] = [
                                'id' => $layout->id,
                                'name' => $layout->name,
                                'success' => true,
                                'message' => 'Ajouté à l\'export'
                            ];
                        }

                        return response()->json($exportData)
                            ->header('Content-Type', 'application/json')
                            ->header('Content-Disposition', 'attachment; filename="invoice_layouts_export_' . date('Y-m-d_H-i-s') . '.json"');
                }

                return response()->json([
                    'success' => true,
                    'data' => $results,
                    'message' => 'Action en masse terminée.'
                ]);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erreur lors de l\'action en masse',
                    'data' => [$e->getMessage()]
                ], 500);
            }
        })->name('bulk-actions');

        // =====================================================
        // ROUTES POUR LES TEMPLATES ET MODÈLES
        // =====================================================

        Route::get('/templates/available', function () {
            $templates = [
                'basic' => [
                    'name' => 'Template de base',
                    'description' => 'Template simple pour factures standards',
                    'design' => 'classic',
                    'features' => ['Logo', 'Informations client', 'Produits', 'Totaux']
                ],
                'professional' => [
                    'name' => 'Template professionnel',
                    'description' => 'Template élégant pour entreprises',
                    'design' => 'elegant',
                    'features' => ['Logo', 'En-tête personnalisé', 'Couleurs', 'Pied de page']
                ],
                'detailed' => [
                    'name' => 'Template détaillé',
                    'description' => 'Template avec toutes les informations produits',
                    'design' => 'detailed',
                    'features' => ['SKU', 'Marques', 'Dates d\'expiration', 'Lots']
                ],
                'thermal' => [
                    'name' => 'Template thermique',
                    'description' => 'Optimisé pour imprimantes thermiques',
                    'design' => 'slim',
                    'features' => ['Compact', 'Sans logo', 'Informations essentielles']
                ]
            ];

            return response()->json([
                'success' => true,
                'data' => $templates,
                'message' => 'Templates disponibles récupérés.'
            ]);
        })->name('templates');

        Route::post('/create-from-template', function (\Illuminate\Http\Request $request) {
            $validator = Validator::make($request->all(), [
                'template' => 'required|string|in:basic,professional,detailed,thermal',
                'name' => 'required|string|max:255'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Données de validation incorrectes',
                    'data' => $validator->errors()
                ], 422);
            }

            $business_id = auth()->user()->business_id;
            $template = $request->template;
            $name = $request->name;

            // Configuration par template
            $templateConfigs = [
                'basic' => [
                    'design' => 'classic',
                    'show_logo' => 1,
                    'show_business_name' => 1,
                    'show_customer' => 1,
                    'show_payments' => 1,
                    'show_sku' => 1,
                ],
                'professional' => [
                    'design' => 'elegant',
                    'show_logo' => 1,
                    'show_business_name' => 1,
                    'show_customer' => 1,
                    'show_payments' => 1,
                    'highlight_color' => '#2c3e50',
                    'header_text' => 'FACTURE PROFESSIONNELLE',
                ],
                'detailed' => [
                    'design' => 'detailed',
                    'show_logo' => 1,
                    'show_business_name' => 1,
                    'show_customer' => 1,
                    'show_brand' => 1,
                    'show_sku' => 1,
                    'show_expiry' => 1,
                    'show_lot' => 1,
                ],
                'thermal' => [
                    'design' => 'slim',
                    'show_logo' => 0,
                    'show_business_name' => 1,
                    'show_customer' => 1,
                    'show_barcode' => 0,
                    'show_image' => 0,
                ]
            ];

            $config = array_merge($templateConfigs[$template], [
                'name' => $name,
                'business_id' => $business_id,
                'is_default' => 0,
                'invoice_heading' => 'FACTURE',
                'footer_text' => 'Merci pour votre confiance',
            ]);

            try {
                $layout = InvoiceLayout::create($config);

                return response()->json([
                    'success' => true,
                    'data' => $layout,
                    'message' => 'Layout créé à partir du template avec succès.'
                ]);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erreur lors de la création du layout',
                    'data' => [$e->getMessage()]
                ], 500);
            }
        })->name('create-from-template');

        // =====================================================
        // ROUTES POUR LES RAPPORTS ET ANALYTICS
        // =====================================================

        Route::get('/analytics/usage', function () {
            $business_id = auth()->user()->business_id;

            // Simuler des données d'usage (vous devrez implémenter la logique réelle)
            $analytics = [
                'most_used_layout' => InvoiceLayout::where('business_id', $business_id)
                    ->where('is_default', 1)
                    ->first(['id', 'name']),
                'layouts_usage' => [
                    // Ces données devraient venir d'une table de logs d'utilisation
                    'classic' => rand(10, 100),
                    'elegant' => rand(5, 80),
                    'detailed' => rand(2, 50),
                    'slim' => rand(1, 30)
                ],
                'creation_timeline' => InvoiceLayout::where('business_id', $business_id)
                    ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
                    ->groupBy('date')
                    ->orderBy('date', 'desc')
                    ->limit(30)
                    ->get(),
                'feature_adoption' => [
                    'logo_usage' => InvoiceLayout::where('business_id', $business_id)->where('show_logo', 1)->count(),
                    'custom_colors' => InvoiceLayout::where('business_id', $business_id)->whereNotNull('highlight_color')->count(),
                    'custom_fields' => InvoiceLayout::where('business_id', $business_id)
                        ->where(function ($q) {
                            $q->whereNotNull('product_custom_fields')
                                ->orWhereNotNull('contact_custom_fields')
                                ->orWhereNotNull('location_custom_fields');
                        })->count()
                ]
            ];

            return response()->json([
                'success' => true,
                'data' => $analytics,
                'message' => 'Analytics d\'usage récupérés.'
            ]);
        })->name('analytics-usage');

        // =====================================================
        // ROUTES POUR LES SAUVEGARDES ET RESTAURATIONS
        // =====================================================

        Route::post('/backup/create', function () {
            $business_id = auth()->user()->business_id;

            $layouts = InvoiceLayout::where('business_id', $business_id)->get();

            $backup = [
                'backup_info' => [
                    'created_at' => now()->toISOString(),
                    'created_by' => auth()->user()->name ?? auth()->user()->email,
                    'business_id' => $business_id,
                    'total_layouts' => $layouts->count()
                ],
                'layouts' => $layouts->map(function ($layout) {
                    $data = $layout->toArray();
                    unset($data['id'], $data['business_id'], $data['created_at'], $data['updated_at']);
                    return $data;
                })
            ];

            $fileName = 'invoice_layouts_backup_' . date('Y-m-d_H-i-s') . '.json';

            return response()->json($backup)
                ->header('Content-Type', 'application/json')
                ->header('Content-Disposition', 'attachment; filename="' . $fileName . '"');
        })->name('backup-create');

        Route::post('/backup/restore', function (\Illuminate\Http\Request $request) {
            $validator = Validator::make($request->all(), [
                'backup_file' => 'required|file|mimes:json',
                'restore_mode' => 'required|string|in:replace,merge'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Données de validation incorrectes',
                    'data' => $validator->errors()
                ], 422);
            }

            try {
                $business_id = auth()->user()->business_id;
                $restoreMode = $request->restore_mode;

                $fileContent = file_get_contents($request->file('backup_file')->path());
                $backupData = json_decode($fileContent, true);

                if (!$backupData || !isset($backupData['layouts'])) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Fichier de sauvegarde invalide.'
                    ], 422);
                }

                if ($restoreMode === 'replace') {
                    // Supprimer tous les layouts existants (sauf défaut)
                    InvoiceLayout::where('business_id', $business_id)
                        ->where('is_default', 0)
                        ->delete();
                }

                $restored = 0;
                foreach ($backupData['layouts'] as $layoutData) {
                    $layoutData['business_id'] = $business_id;
                    $layoutData['is_default'] = 0; // Jamais restaurer comme défaut

                    // Assurer l'unicité du nom
                    $baseName = $layoutData['name'];
                    $counter = 1;
                    while (InvoiceLayout::where('business_id', $business_id)->where('name', $layoutData['name'])->exists()) {
                        $layoutData['name'] = $baseName . ' (' . $counter . ')';
                        $counter++;
                    }

                    InvoiceLayout::create($layoutData);
                    $restored++;
                }

                return response()->json([
                    'success' => true,
                    'data' => [
                        'restored_count' => $restored,
                        'restore_mode' => $restoreMode
                    ],
                    'message' => $restored . ' layouts restaurés avec succès.'
                ]);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erreur lors de la restauration',
                    'data' => [$e->getMessage()]
                ], 500);
            }
        })->name('backup-restore');
    });

    /*
    |--------------------------------------------------------------------------
    | Labels API Routes
    |--------------------------------------------------------------------------
    */

    // Labels Resource Routes
    Route::apiResource('labels', LabelsController::class);

    // Additional Labels Routes
    Route::prefix('labels')->group(function () {
        Route::post('add-product-row', [LabelsController::class, 'addProductRow'])
            ->name('labels.add-product-row');
        Route::post('preview', [LabelsController::class, 'preview'])
            ->name('labels.preview');
    });

    /*
    |--------------------------------------------------------------------------
    | Invoice Scheme API Routes
    |--------------------------------------------------------------------------
    */

    // Invoice Scheme Resource Routes
    Route::apiResource('invoice-schemes', InvoiceSchemeController::class);

    // Additional Invoice Scheme Routes
    Route::prefix('invoice-schemes')->group(function () {
        Route::patch('{id}/set-default', [InvoiceSchemeController::class, 'setDefault'])
            ->name('invoice-schemes.set-default');
        Route::get('default', [InvoiceSchemeController::class, 'getDefault'])
            ->name('invoice-schemes.get-default');
    });

    // Routes pour Location Settings
    Route::prefix('location-settings')->group(function () {
        Route::get('/', [LocationSettingsController::class, 'index']);
        Route::get('/{location_id}', [LocationSettingsController::class, 'show']);
        Route::post('/', [LocationSettingsController::class, 'store']);
        Route::put('/{location_id}', [LocationSettingsController::class, 'update']);
        Route::delete('/{location_id}', [LocationSettingsController::class, 'destroy']);
    });

    // Routes pour Notifications
    Route::prefix('notifications')->group(function () {
        Route::get('/', [NotificationController::class, 'index']);
        Route::get('/{id}/{template_for}', [NotificationController::class, 'show']);
        Route::post('/', [NotificationController::class, 'store']);
        Route::put('/{id}', [NotificationController::class, 'update']);
        Route::delete('/{id}', [NotificationController::class, 'destroy']);
        Route::post('/send', [NotificationController::class, 'send']);
    });

    // Routes pour Notification Templates
    Route::prefix('notification-templates')->group(function () {
        Route::get('/', [NotificationTemplateController::class, 'index']);
        Route::get('/{id}', [NotificationTemplateController::class, 'show']);
        Route::post('/', [NotificationTemplateController::class, 'store']);
        Route::put('/{id}', [NotificationTemplateController::class, 'update']);
        Route::delete('/{id}', [NotificationTemplateController::class, 'destroy']);
        Route::post('/bulk-update', [NotificationTemplateController::class, 'bulkUpdate']);
    });

    // Opening Stock Routes
    Route::prefix('opening-stock')->group(function () {
        
        // GET /api/opening-stock/add/{product_id} - Récupérer les données pour l'ajout/modification du stock d'ouverture
        Route::get('/add/{product_id}', [OpeningStockController::class, 'add']);
        
        // POST /api/opening-stock/save - Sauvegarder le stock d'ouverture
        Route::post('/save', [OpeningStockController::class, 'save']);
        
    });

    Route::group(['prefix' => 'reports'], function () {
        Route::get('/profit-loss', [ReportController::class, 'getProfitLoss']);
        Route::get('/purchase-sell', [ReportController::class, 'getPurchaseSell']);
        Route::get('/customer-suppliers', [ReportController::class, 'getCustomerSuppliers']);
        Route::get('/stock-report', [ReportController::class, 'getStockReport']);
        Route::get('/stock-details', [ReportController::class, 'getStockDetails']);
        Route::get('/tax-details', [ReportController::class, 'getTaxDetails']);
        Route::get('/tax-report', [ReportController::class, 'getTaxReport']);
        Route::get('/trending-products', [ReportController::class, 'getTrendingProducts']);
        Route::get('/expense-report', [ReportController::class, 'getExpenseReport']);
        Route::get('/stock-adjustment-report', [ReportController::class, 'getStockAdjustmentReport']);
        Route::get('/register-report', [ReportController::class, 'getRegisterReport']);
        Route::get('/sales-representative-report', [ReportController::class, 'getSalesRepresentativeReport']);
        Route::get('/sales-representative-total-expense', [ReportController::class, 'getSalesRepresentativeTotalExpense']);
        Route::get('/sales-representative-total-sell', [ReportController::class, 'getSalesRepresentativeTotalSell']);
        Route::get('/sales-representative-total-commission', [ReportController::class, 'getSalesRepresentativeTotalCommission']);
        Route::get('/stock-expiry-report', [ReportController::class, 'getStockExpiryReport']);
        Route::get('/stock-expiry-report-edit-modal/{purchase_line_id}', [ReportController::class, 'getStockExpiryReportEditModal']);
        Route::post('/update-stock-expiry-report', [ReportController::class, 'updateStockExpiryReport']);
        Route::get('/customer-group', [ReportController::class, 'getCustomerGroup']);
        Route::get('/product-purchase-report', [ReportController::class, 'getproductPurchaseReport']);
        Route::get('/product-sell-report', [ReportController::class, 'getproductSellReport']);
        Route::get('/product-sell-report-with-purchase', [ReportController::class, 'getproductSellReportWithPurchase']);
        Route::get('/lot-report', [ReportController::class, 'getLotReport']);
        Route::get('/purchase-payment-report', [ReportController::class, 'purchasePaymentReport']);
        Route::get('/sell-payment-report', [ReportController::class, 'sellPaymentReport']);
        Route::get('/table-report', [ReportController::class, 'getTableReport']);
        Route::get('/service-staff-report', [ReportController::class, 'getServiceStaffReport']);
        Route::get('/product-sell-grouped-report', [ReportController::class, 'getproductSellGroupedReport']);
        Route::get('/product-stock-details', [ReportController::class, 'productStockDetails']);
        Route::post('/adjust-product-stock', [ReportController::class, 'adjustProductStock']);
        Route::get('/service-staff-line-orders', [ReportController::class, 'serviceStaffLineOrders']);
        Route::get('/get-profit/{by?}', [ReportController::class, 'getProfit']);
        Route::get('/items-report', [ReportController::class, 'itemsReport']);
        Route::get('/purchase-report', [ReportController::class, 'purchaseReport']);
        Route::get('/sale-report', [ReportController::class, 'saleReport']);
        Route::get('/get-stock-value', [ReportController::class, 'getStockValue']);

        // Stock Adjustment Report
        Route::get('/stock-adjustment',  [ReportController::class, 'getStockAdjustmentReport']);

        // Register Report
        Route::get('/register',  [ReportController::class, 'getRegisterReport']);

        // Sales Representative Report
        Route::get('/sales-representative',  [ReportController::class, 'getSalesRepresentativeReport']);
        Route::get('/sales-representative-expense',  [ReportController::class, 'getSalesRepresentativeTotalExpense']);
        Route::get('/sales-representative-sell',  [ReportController::class, 'getSalesRepresentativeTotalSell']);
        Route::get('/sales-representative-commission',  [ReportController::class, 'getSalesRepresentativeTotalCommission']);

        // Stock Expiry Report
        Route::get('/stock-expiry',  [ReportController::class, 'getStockExpiryReport']);
        Route::get('/stock-expiry-edit-modal/{purchase_line_id}',  [ReportController::class, 'getStockExpiryReportEditModal']);
        Route::post('/stock-expiry-update',  [ReportController::class, 'updateStockExpiryReport']);

        // Customer Group Report
        Route::get('/customer-group',  [ReportController::class, 'getCustomerGroup']);

        // Product Purchase Report
        Route::get('/product-purchase',  [ReportController::class, 'getproductPurchaseReport']);

        // Product Sell Report
        Route::get('/product-sell',  [ReportController::class, 'getproductSellReport']);

        // Product Sell Report With Purchase
        Route::get('/product-sell-with-purchase',  [ReportController::class, 'getproductSellReportWithPurchase']);

        // Lot Report
        Route::get('/lot',  [ReportController::class, 'getLotReport']);

        // Purchase Payment Report
        Route::get('/purchase-payment',  [ReportController::class, 'purchasePaymentReport']);

        // Sell Payment Report
        Route::get('/sell-payment',  [ReportController::class, 'sellPaymentReport']);

        // Table Report
        Route::get('/table',  [ReportController::class, 'getTableReport']);

        // Service Staff Report
        Route::get('/service-staff',  [ReportController::class, 'getServiceStaffReport']);

        // Product Sell Grouped Report
        Route::get('/product-sell-grouped',  [ReportController::class, 'getproductSellGroupedReport']);

        // Product Stock Details
        Route::get('/product-stock-details',  [ReportController::class, 'productStockDetails']);

        // Adjust Product Stock
        Route::post('/adjust-product-stock',  [ReportController::class, 'adjustProductStock']);

        // Service Staff Line Orders
        Route::get('/service-staff-line-orders',  [ReportController::class, 'serviceStaffLineOrders']);

        // Profit Report
        Route::get('/profit/{by?}',  [ReportController::class, 'getProfit']);

        // Items Report
        Route::get('/items',  [ReportController::class, 'itemsReport']);

        // Purchase Report
        Route::get('/purchase',  [ReportController::class, 'purchaseReport']);

        // Sale Report
        Route::get('/sale',  [ReportController::class, 'saleReport']);

        // Stock Value
        Route::get('/stock-value',  [ReportController::class, 'getStockValue']);
    });

    Route::group(['prefix' => 'sells'], function () {
        // List Sells
        Route::get('/', [SellController::class, 'index'])->name('api.sells.index');

        // Get Sell Details
        Route::get('/{id}', [SellController::class, 'show'])->name('api.sells.show');

        // Update Shipping
        Route::put('/{id}/shipping', [SellController::class, 'updateShipping'])->name('api.sells.updateShipping');

        // Edit Shipping (Get Data for Modal)
        Route::get('/{id}/shipping/edit', [SellController::class, 'editShipping'])->name('api.sells.editShipping');

        // List Drafts
        Route::get('/drafts', [SellController::class, 'getDrafts'])->name('api.sells.getDrafts');

        // List Final Sells
        Route::get('/final', [SellController::class, 'getFinal'])->name('api.sells.getFinal');

        // List Quotations
        Route::get('/quotations', [SellController::class, 'getQuotations'])->name('api.sells.getQuotations');

        // Duplicate Sell
        Route::post('/{id}/duplicate', [SellController::class, 'duplicateSell'])->name('api.sells.duplicateSell');

        // Shipments (Get Data for Filters)
        Route::get('/shipments', [SellController::class, 'shipments'])->name('api.sells.shipments');
    });

    Route::group(['prefix' => 'selling-price-groups'], function () {
        // List Selling Price Groups
        Route::get('/', [SellingPriceGroupController::class, 'index'])->name('api.selling-price-groups.index');

        // Create Selling Price Group
        Route::post('/', [SellingPriceGroupController::class, 'store'])->name('api.selling-price-groups.store');

        // Show the form for creating a new resource. (Not typically needed for a pure API)
        //Route::get('/create', 'SellingPriceGroupController@create')->name('api.selling-price-groups.create');

        // Get Selling Price Group Details for Editing
        Route::get('/{id}/edit', [SellingPriceGroupController::class, 'edit'])->name('api.selling-price-groups.edit');

        // Update Selling Price Group
        Route::put('/{id}', [SellingPriceGroupController::class, 'update'])->name('api.selling-price-groups.update');

        // Delete Selling Price Group
        Route::delete('/{id}', [SellingPriceGroupController::class, 'destroy'])->name('api.selling-price-groups.destroy');

        // Activate/Deactivate Selling Price Group
        Route::post('/{id}/activate-deactivate', [SellingPriceGroupController::class, 'activateDeactivate'])->name('api.selling-price-groups.activateDeactivate');

        // Export Selling Price Group Prices (Consider if this is truly an API function)
        //Route::get('/export', [SellingPriceGroupController::class, 'export'])->name('api.selling-price-groups.export');

        // Import Selling Price Group Prices (Consider if this is truly an API function)
        //Route::post('/import', [SellingPriceGroupController::class, 'import'])->name('api.selling-price-groups.import');
    });

    // Main POS Routes
    Route::prefix('pos')->name('pos.')->group(function () {

        // List POS sells data (for dropdowns and initial data)
        Route::get('/', [SellPosController::class, 'index'])->name('index');

        // Get POS creation data
        Route::get('/create', [SellPosController::class, 'create'])->name('create');

        // Store new POS sale
        Route::post('/', [SellPosController::class, 'store'])->name('store');

        // Get POS edit data
        Route::get('/{id}/edit', [SellPosController::class, 'edit'])->name('edit');

        // Transaction recovery (recouvrement)
        Route::get('/{id}/recover', [SellPosController::class, 'recouvrementApi'])->name('recover');

        // Update recovered transaction
        Route::put('/{id}/recover', [SellPosController::class, 'updateRecApi'])->name('update-recover');

        // Delete POS sale
        Route::delete('/{id}', [SellPosController::class, 'destroyApi'])->name('destroy');

        // Cancel transaction
        Route::post('/{id}/cancel', [SellPosController::class, 'cancelledApi'])->name('cancel');

        // Print invoice
        Route::get('/{id}/print', [SellPosController::class, 'printInvoiceApi'])->name('print');

        // Show invoice URL
        Route::get('/{id}/invoice-url', [SellPosController::class, 'showInvoiceUrlApi'])->name('invoice-url');
    });

    // Product and Utility Routes
    Route::prefix('pos-utils')->name('pos.utils.')->group(function () {

        // Get product row for POS
        Route::get('/product-row/{variation_id}/{location_id}', [SellPosController::class, 'getProductRowApi'])->name('product-row');

        // Get payment row
        Route::get('/payment-row', [SellPosController::class, 'getPaymentRowApi'])->name('payment-row');

        // Get recent transactions
        Route::get('/recent-transactions', [SellPosController::class, 'getRecentTransactionsApi'])->name('recent-transactions');

        // Get product suggestions
        Route::get('/product-suggestions', [SellPosController::class, 'getProductSuggestionApi'])->name('product-suggestions');

        // Get reward details
        Route::get('/reward-details', [SellPosController::class, 'getRewardDetails'])->name('reward-details');

        // Get types of service details
        Route::get('/service-details', [SellPosController::class, 'getTypesOfServiceDetails'])->name('service-details');
    });

    // Subscription Management Routes
    Route::prefix('subscriptions')->name('subscriptions.')->group(function () {

        // List all subscriptions
        Route::get('/', [SellPosController::class, 'listSubscriptionsApi'])->name('index');

        // Toggle recurring invoice (start/stop)
        Route::post('/{id}/toggle', [SellPosController::class, 'toggleRecurringInvoices'])->name('toggle');
    });

    // Liste des retours de vente (index)
    Route::get('/sell-returns', [SellReturnController::class, 'index'])->name('api.sell_returns.index');

    // Formulaire pour créer un nouveau retour (add)
    // Note : en API on retourne souvent les données nécessaires au formulaire, ici on récupère un retour parent par ID
    Route::get('/sell-returns/add/{id}', [SellReturnController::class, 'add'])->name('api.sell_returns.add');

    // Stocker un nouveau retour de vente (store)
    Route::post('/sell-returns', [SellReturnController::class, 'store'])->name('api.sell_returns.store');

    // Afficher un retour de vente précis (show)
    Route::get('/sell-returns/{id}', [SellReturnController::class, 'show'])->name('api.sell_returns.show');

    // Imprimer la facture d’un retour de vente (printInvoice)
    Route::get('/sell-returns/{transaction_id}/print-invoice', [SellReturnController::class, 'printInvoice'])
        ->name('api.sell_returns.print_invoice');

    // Optionnel : version POST si besoin
    Route::post('/sell-returns/{transaction_id}/print-invoice', [SellReturnController::class, 'printInvoice'])
        ->name('api.sell_returns.print_invoice.post');

    Route::prefix('payments')->group(function () {
        Route::post('/', [TransactionPaymentController::class, 'store']);
        Route::get('/{id}', [TransactionPaymentController::class, 'show']);
        Route::get('/{id}/edit', [TransactionPaymentController::class, 'edit']);
        Route::put('/{id}', [TransactionPaymentController::class, 'update']);
        Route::delete('/{id}', [TransactionPaymentController::class, 'destroy']);

        Route::get('/transaction/{transaction_id}/add', [TransactionPaymentController::class, 'addPayment']);
        Route::get('/contact/{contact_id}/pay-due', [TransactionPaymentController::class, 'getPayContactDue']);
        Route::post('/contact/pay-due', [TransactionPaymentController::class, 'postPayContactDue']);
        Route::get('/view/{payment_id}', [TransactionPaymentController::class, 'viewPayment']);
        Route::get('/{payment_id}/children', [TransactionPaymentController::class, 'showChildPayments']);
        Route::get('/contact/{contact_id}/opening-balance', [TransactionPaymentController::class, 'getOpeningBalancePayments']);
    });

    Route::get('types-of-service', [TypesOfServiceController::class, 'index']);
    Route::post('types-of-service', [TypesOfServiceController::class, 'store']);
    Route::get('types-of-service/create', [TypesOfServiceController::class, 'create']);
    Route::get('types-of-service/{id}/edit', [TypesOfServiceController::class, 'edit']);
    Route::put('types-of-service/{id}', [TypesOfServiceController::class, 'update']);
    Route::delete('types-of-service/{id}', [TypesOfServiceController::class, 'destroy']);
});

/*
|--------------------------------------------------------------------------
| ROUTES PUBLIQUES (SANS AUTHENTIFICATION)
|--------------------------------------------------------------------------
*/

// Designs disponibles en public
Route::get('public/invoice-layouts/designs', [InvoiceLayoutController::class, 'getDesigns'])
    ->name('api.public.invoice-layouts.designs');


Route::middleware('auth:sanctum')->get('/user', function (Request $request) {
    return $request->user();
});
