<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;
use Illuminate\Foundation\Auth\User as Authenticatable;

class User extends Authenticatable
{
	use HasFactory, Notifiable, HasApiTokens, SoftDeletes, HasRoles;

	protected $table = 'users';

	protected $casts = [
		'business_id' => 'int',
		'max_sales_discount_percent' => 'float',
		'allow_login' => 'bool',
		'is_cmmsn_agnt' => 'bool',
		'cmmsn_percent' => 'float',
		'selected_contacts' => 'bool',
		'dob' => 'datetime'
	];

	protected $hidden = [
		'password',
		'remember_token'
	];

	protected $fillable = [
		'user_type',
		'surname',
		'first_name',
		'last_name',
		'username',
		'email',
		'password',
		'language',
		'contact_no',
		'address',
		'remember_token',
		'business_id',
		'max_sales_discount_percent',
		'allow_login',
		'status',
		'is_cmmsn_agnt',
		'cmmsn_percent',
		'selected_contacts',
		'dob',
		'gender',
		'marital_status',
		'blood_group',
		'contact_number',
		'fb_link',
		'twitter_link',
		'social_media_1',
		'social_media_2',
		'permanent_address',
		'current_address',
		'guardian_name',
		'custom_field_1',
		'custom_field_2',
		'custom_field_3',
		'custom_field_4',
		'bank_details',
		'id_proof_name',
		'id_proof_number'
	];



	/**
	 * Get the identifier that will be stored in the subject claim of the JWT.
	 *
	 * @return mixed
	 */
	public function getJWTIdentifier()
	{
		return $this->getKey();
	}

	/**
	 * Return a key value array, containing any custom claims to be added to the JWT.
	 *
	 * @return array
	 */
	public function getJWTCustomClaims()
	{
		return [];
	}


	/**
	 * Vérifier si l'utilisateur est actif
	 */
	public function isActive()
	{
		return $this->is_active;
	}

	/**
	 * Scope pour les utilisateurs actifs
	 */
	public function scopeActive($query)
	{
		return $query->where('is_active', true);
	}

	public function bookings()
	{
		return $this->hasMany(Booking::class, 'created_by');
	}

	public function brands()
	{
		return $this->hasMany(Brand::class, 'created_by');
	}

	public function business()
	{
		return $this->hasMany(Business::class, 'owner_id');
	}

	// Dans User.php
	public function busines()
	{
		return $this->belongsTo(Business::class, 'business_id');
	}

	public function cash_registers()
	{
		return $this->hasMany(CashRegister::class);
	}

	public function categories()
	{
		return $this->hasMany(Category::class, 'created_by');
	}

	public function contacts()
	{
		return $this->hasMany(Contact::class, 'created_by');
	}

	public function scopeUser($query){
		return $query->where('users.user_type', 'user');
	}

	/**
	 * The contactAccess that belong to the User
	 *
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
	 */
	public function contactAccess()
	{
		return $this->belongsToMany(Contact::class, 'user_contact_access', 'user_id', 'contact_id')
			->withPivot('created_by', 'updated_by')
			->withTimestamps();
	}

	/**
	 * Creates a new user based on the input provided.
	 *
	 * @return object
	 */
	public static function create_user($details)
	{
		// Préparer les données en filtrant les valeurs nulles pour les champs optionnels
		$userData = array_filter([
			'surname' => $details['surname'] ?? null,
			'first_name' => $details['first_name'] ?? null,
			'last_name' => $details['last_name'] ?? null,
			'username' => $details['username'],
			'email' => $details['email'],
			'password' => Hash::make($details['password']),
			'language' => !empty($details['language']) ? $details['language'] : 'fr'
		], function ($value) {
			return $value !== null;
		});

		$user = User::create($userData);
		return $user;
	}

	/**
	 * Gives locations permitted for the logged in user
	 *
	 * @return string or array
	 */
	public function permitted_locations()
	{
		$user = $this;

		if ($user->can('access_all_locations')) {
			return 'all';
		} else {
			$business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
			$permitted_locations = [];
			$all_locations = BusinessLocation::where('business_id', $business_id)->get();
			foreach ($all_locations as $location) {
				if ($user->can('location.' . $location->id)) {
					$permitted_locations[] = $location->id;
				}
			}

			return $permitted_locations;
		}
	}

	/**
	 * Returns if a user can access the input location
	 *
	 * @param: int $location_id
	 * @return boolean
	 */
	public static function can_access_this_location($location_id)
	{
		$permitted_locations = auth()->user()->permitted_locations();

		if ($permitted_locations == 'all' || in_array($location_id, $permitted_locations)) {
			return true;
		}

		return false;
	}

	public function scopeOnlyPermittedLocations($query)
	{
		$user = auth()->user();
		$permitted_locations = $user->permitted_locations();
		$is_admin = $user->hasAnyPermission('Admin#' . $user->business_id);
		if ($permitted_locations != 'all' && !$user->can('superadmin') && !$is_admin) {
			$permissions = ['access_all_locations'];
			foreach ($permitted_locations as $location_id) {
				$permissions[] = 'location.' . $location_id;
			}

			return $query->whereHas('permissions', function ($q) use ($permissions) {
				$q->whereIn('permissions.name', $permissions);
			});
		} else {
			return $query;
		}
	}

	/**
	 * Return list of users dropdown for a business
	 *
	 * @param $business_id int
	 * @param $prepend_none = true (boolean)
	 * @param $include_commission_agents = false (boolean)
	 *
	 * @return array users
	 */
	public static function forDropdown($business_id, $prepend_none = true, $include_commission_agents = false, $prepend_all = false, $check_location_permission = false)
	{
		$query = User::where('business_id', $business_id)
			->user();

		if (!$include_commission_agents) {
			$query->where('is_cmmsn_agnt', 0);
		}

		if ($check_location_permission) {
			$query->onlyPermittedLocations();
		}

		$all_users = $query->select('id', DB::raw("CONCAT(COALESCE(surname, ''),' ',COALESCE(first_name, ''),' ',COALESCE(last_name,'')) as full_name"))->get();
		$users = $all_users->pluck('full_name', 'id');

		//Prepend none
		if ($prepend_none) {
			$users = $users->prepend(__('lang_v1.none'), '');
		}

		//Prepend all
		if ($prepend_all) {
			$users = $users->prepend(__('lang_v1.all'), '');
		}

		return $users;
	}

	/**
	 * Return list of sales commission agents dropdown for a business
	 *
	 * @param $business_id int
	 * @param $prepend_none = true (boolean)
	 *
	 * @return array users
	 */
	public static function saleCommissionAgentsDropdown($business_id, $prepend_none = true)
	{
		$all_cmmsn_agnts = User::where('business_id', $business_id)
			->where('is_cmmsn_agnt', 1)
			->select('id', DB::raw("CONCAT(COALESCE(surname, ''),' ',COALESCE(first_name, ''),' ',COALESCE(last_name,'')) as full_name"));

		$users = $all_cmmsn_agnts->pluck('full_name', 'id');

		//Prepend none
		if ($prepend_none) {
			$users = $users->prepend(__('lang_v1.none'), '');
		}

		return $users;
	}

	/**
	 * Return list of users dropdown for a business
	 *
	 * @param $business_id int
	 * @param $prepend_none = true (boolean)
	 * @param $prepend_all = false (boolean)
	 *
	 * @return array users
	 */
	public static function allUsersDropdown($business_id, $prepend_none = true, $prepend_all = false)
	{
		$all_users = User::where('business_id', $business_id)
			->select('id', DB::raw("CONCAT(COALESCE(surname, ''),' ',COALESCE(first_name, ''),' ',COALESCE(last_name,'')) as full_name"));

		$users = $all_users->pluck('full_name', 'id');

		//Prepend none
		if ($prepend_none) {
			$users = $users->prepend(__('lang_v1.none'), '');
		}

		//Prepend all
		if ($prepend_all) {
			$users = $users->prepend(__('lang_v1.all'), '');
		}

		return $users;
	}

	/**
	 * Get the user's full name.
	 *
	 * @return string
	 */
	public function getUserFullNameAttribute()
	{
		return "{$this->surname} {$this->first_name} {$this->last_name}";
	}

	/**
	 * Return true/false based on selected_contact access
	 *
	 * @return boolean
	 */
	public static function isSelectedContacts($user_id)
	{
		$user = User::findOrFail($user_id);

		return (bool)$user->selected_contacts;
	}

	public function getRoleNameAttribute()
	{
		$role_name_array = $this->getRoleNames();
		$role_name = !empty($role_name_array[0]) ? explode('#', $role_name_array[0])[0] : '';
		return $role_name;
	}

	public function media()
	{
		return $this->morphOne(Media::class, 'model');
	}

	/**
	 * Find the user instance for the given username.
	 *
	 * @param  string  $username
	 * @return \App\User
	 */
	public function findForPassport($username)
	{
		return $this->where('username', $username)->first();
	}
}
