<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;

/**
 * Class Product
 * 
 * @property int $id
 * @property string $name
 * @property int $business_id
 * @property string|null $type
 * @property int|null $unit_id
 * @property string|null $sub_unit_ids
 * @property int|null $brand_id
 * @property int|null $category_id
 * @property int|null $sub_category_id
 * @property int|null $tax
 * @property string $tax_type
 * @property bool $enable_stock
 * @property float|null $alert_quantity
 * @property string $sku
 * @property string|null $barcode_type
 * @property float|null $expiry_period
 * @property string|null $expiry_period_type
 * @property bool $enable_sr_no
 * @property string|null $weight
 * @property string|null $product_custom_field1
 * @property string|null $product_custom_field2
 * @property string|null $product_custom_field3
 * @property string|null $product_custom_field4
 * @property string|null $image
 * @property string|null $product_description
 * @property int $created_by
 * @property int|null $warranty_id
 * @property bool $is_inactive
 * @property bool $not_for_selling
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 *
 * @package App\Models
 */
class Product extends Model
{
	protected $table = 'products';

	protected $casts = [
		'business_id' => 'int',
		'unit_id' => 'int',
		'brand_id' => 'int',
		'category_id' => 'int',
		'sub_category_id' => 'int',
		'tax' => 'int',
		'enable_stock' => 'bool',
		'alert_quantity' => 'float',
		'expiry_period' => 'float',
		'enable_sr_no' => 'bool',
		'created_by' => 'int',
		'warranty_id' => 'int',
		'is_inactive' => 'bool',
		'not_for_selling' => 'bool'
	];

	protected $fillable = [
		'name',
		'business_id',
		'type',
		'unit_id',
		'sub_unit_ids',
		'brand_id',
		'category_id',
		'sub_category_id',
		'tax',
		'tax_type',
		'enable_stock',
		'alert_quantity',
		'sku',
		'barcode_type',
		'expiry_period',
		'expiry_period_type',
		'enable_sr_no',
		'weight',
		'product_custom_field1',
		'product_custom_field2',
		'product_custom_field3',
		'product_custom_field4',
		'image',
		'product_description',
		'created_by',
		'warranty_id',
		'is_inactive',
		'not_for_selling'
	];

	/**
	 * Get the products image.
	 *
	 * @return string
	 */
	public function getImageUrlAttribute()
	{
		if (!empty($this->image)) {
			$image_url = asset('/uploads/img/' . rawurlencode($this->image));
		} else {
			$image_url = asset('/img/default.png');
		}
		return $image_url;
	}

	/**
	 * Get the products image path.
	 *
	 * @return string
	 */
	public function getImagePathAttribute()
	{
		if (!empty($this->image)) {
			$image_path = public_path('uploads') . '/' . config('constants.product_img_path') . '/' . $this->image;
		} else {
			$image_path = null;
		}
		return $image_path;
	}

	public function product_variations()
	{
		return $this->hasMany(ProductVariation::class);
	}

	/**
	 * Get the brand associated with the product.
	 */
	public function brand()
	{
		return $this->belongsTo(Brand::class);
	}

	/**
	 * Get the brand associated with the product.
	 */
	public function contact()
	{
		return $this->belongsTo(Contact::class);
	}

	/**
	 * Get the unit associated with the product.
	 */
	public function unit()
	{
		return $this->belongsTo(Unit::class);
	}
	/**
	 * Get category associated with the product.
	 */
	public function category()
	{
		return $this->belongsTo(Category::class);
	}
	/**
	 * Get sub-category associated with the product.
	 */
	public function sub_category()
	{
		return $this->belongsTo(Category::class, 'sub_category_id', 'id');
	}

	/**
	 * Get the brand associated with the product.
	 */
	public function product_tax()
	{
		return $this->belongsTo(TaxRate::class, 'tax', 'id');
	}

	/**
	 * Get the variations associated with the product.
	 */
	public function variations()
	{
		return $this->hasMany(Variation::class);
	}

	/**
	 * If product type is modifier get products associated with it.
	 */
	public function modifier_products()
	{
		return $this->belongsToMany(Product::class, 'res_product_modifier_sets', 'modifier_set_id', 'product_id');
	}

	/**
	 * If product type is modifier get products associated with it.
	 */
	public function modifier_sets()
	{
		return $this->belongsToMany(Product::class, 'res_product_modifier_sets', 'product_id', 'modifier_set_id');
	}

	/**
	 * Get the purchases associated with the product.
	 */
	public function purchase_lines()
	{
		return $this->hasMany(PurchaseLine::class);
	}

	/**
	 * Scope a query to only include active products.
	 *
	 * @param \Illuminate\Database\Eloquent\Builder $query
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	public function scopeActive($query)
	{
		return $query->where('products.is_inactive', 0);
	}

	/**
	 * Scope a query to only include inactive products.
	 *
	 * @param \Illuminate\Database\Eloquent\Builder $query
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	public function scopeInactive($query)
	{
		return $query->where('products.is_inactive', 1);
	}

	/**
	 * Scope a query to only include products for sales.
	 *
	 * @param \Illuminate\Database\Eloquent\Builder $query
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	public function scopeProductForSales($query)
	{
		return $query->where('not_for_selling', 0);
	}

	/**
	 * Scope a query to only include products not for sales.
	 *
	 * @param \Illuminate\Database\Eloquent\Builder $query
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	public function scopeProductNotForSales($query)
	{
		return $query->where('not_for_selling', 1);
	}

	public function product_locations()
	{
		return $this->belongsToMany(BusinessLocation::class, 'product_locations', 'product_id', 'location_id');
	}

	public function kit_products()
	{
		return $this->belongsToMany(Product::class, 'kit_products', 'kit_id', 'product_id');
	}

	/**
	 * Scope a query to only include products available for a location.
	 *
	 * @param \Illuminate\Database\Eloquent\Builder $query
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	public function scopeForLocation($query, $location_id)
	{
		return $query->where(function ($q) use ($location_id) {
			$q->whereHas('product_locations', function ($query) use ($location_id) {
				$query->where('product_locations.location_id', $location_id);
			});
		});
	}

	/**
	 * Get warranty associated with the product.
	 */
	public function warranty()
	{
		return $this->belongsTo(Warranty::class);
	}

	public static function forDropdown($business_id, $show_none = false)
	{
		$products = Product::where('business_id', $business_id)
			->pluck('name', 'id');

		if ($show_none) {
			$products->prepend(__('lang_v1.none'), '');
		}

		return $products;
	}
}
