<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;

/**
 * Class Contact
 * 
 * @property int $id
 * @property int $business_id
 * @property string $type
 * @property string|null $supplier_business_name
 * @property string $name
 * @property string|null $prefix
 * @property string|null $first_name
 * @property string|null $middle_name
 * @property string|null $last_name
 * @property string|null $email
 * @property string|null $contact_id
 * @property string $contact_status
 * @property string|null $tax_number
 * @property string|null $city
 * @property string|null $state
 * @property string|null $country
 * @property string|null $address_line_1
 * @property string|null $address_line_2
 * @property string|null $zip_code
 * @property Carbon|null $dob
 * @property string|null $mobile
 * @property string|null $landline
 * @property string|null $alternate_number
 * @property int|null $pay_term_number
 * @property string|null $pay_term_type
 * @property float|null $credit_limit
 * @property int $created_by
 * @property float $balance
 * @property int $total_rp
 * @property int $total_rp_used
 * @property int $total_rp_expired
 * @property bool $is_default
 * @property string|null $shipping_address
 * @property string|null $position
 * @property int|null $customer_group_id
 * @property string|null $custom_field1
 * @property string|null $custom_field2
 * @property string|null $custom_field3
 * @property string|null $custom_field4
 * @property string|null $deleted_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * 
 * @property Business $business
 * @property User $user
 * @property Collection|Booking[] $bookings
 *
 * @package App\Models
 */
class Contact extends Model
{
	use SoftDeletes;
	protected $table = 'contacts';

	protected $casts = [
		'business_id' => 'int',
		'dob' => 'datetime',
		'pay_term_number' => 'int',
		'credit_limit' => 'float',
		'created_by' => 'int',
		'balance' => 'float',
		'total_rp' => 'int',
		'total_rp_used' => 'int',
		'total_rp_expired' => 'int',
		'is_default' => 'bool',
		'customer_group_id' => 'int'
	];

	protected $fillable = [
		'business_id',
		'type',
		'supplier_business_name',
		'name',
		'prefix',
		'first_name',
		'middle_name',
		'last_name',
		'email',
		'contact_id',
		'contact_status',
		'tax_number',
		'city',
		'state',
		'country',
		'address_line_1',
		'address_line_2',
		'zip_code',
		'dob',
		'mobile',
		'landline',
		'alternate_number',
		'pay_term_number',
		'pay_term_type',
		'credit_limit',
		'created_by',
		'balance',
		'total_rp',
		'total_rp_used',
		'total_rp_expired',
		'is_default',
		'shipping_address',
		'position',
		'customer_group_id',
		'custom_field1',
		'custom_field2',
		'custom_field3',
		'custom_field4'
	];


	/**
	 * Get the business that owns the user.
	 */
	public function business()
	{
		return $this->belongsTo(Business::class);
	}

	public function scopeActive($query)
	{
		return $query->where('contacts.contact_status', 'active');
	}

	public function scopeOnlySuppliers($query)
	{
		return $query->whereIn('contacts.type', ['supplier', 'both']);
	}

	public function scopeOnlyCustomers($query)
	{
		return $query->whereIn('contacts.type', ['customer', 'both']);
	}

	/**
	 * Get all of the contacts's notes & documents.
	 */
	public function documentsAndnote()
	{
		return $this->morphMany('App\DocumentAndNote', 'notable');
	}

	/**
	 * Return list of contact dropdown for a business
	 *
	 * @param $business_id int
	 * @param $exclude_default = false (boolean)
	 * @param $prepend_none = true (boolean)
	 *
	 * @return array users
	 */
	public static function contactDropdown($business_id, $exclude_default = false, $prepend_none = true, $append_id = true)
	{
		$query = Contact::where('business_id', $business_id)
			->where('type', '!=', 'lead')
			->active();

		if ($exclude_default) {
			$query->where('is_default', 0);
		}

		if ($append_id) {
			$query->select(
				DB::raw("IF(contact_id IS NULL OR contact_id='', name, CONCAT(name, ' - ', COALESCE(supplier_business_name, ''), '(', contact_id, ')')) AS supplier"),
				'id'
			);
		} else {
			$query->select(
				'id',
				DB::raw("IF (supplier_business_name IS not null, CONCAT(name, ' (', supplier_business_name, ')'), name) as supplier")
			);
		}

		$contacts = $query->pluck('supplier', 'id');

		//Prepend none
		if ($prepend_none) {
			$contacts = $contacts->prepend(__('lang_v1.none'), '');
		}

		return $contacts;
	}

	/**
	 * Return list of suppliers dropdown for a business
	 *
	 * @param $business_id int
	 * @param $prepend_none = true (boolean)
	 *
	 * @return array users
	 */
	public static function suppliersDropdown($business_id, $prepend_none = true, $append_id = true)
	{
		$all_contacts = Contact::where('business_id', $business_id)
			->whereIn('type', ['supplier', 'both'])
			->active();

		if ($append_id) {
			$all_contacts->select(
				DB::raw("IF(contact_id IS NULL OR contact_id='', name, CONCAT(name, ' - ', COALESCE(supplier_business_name, ''), '(', contact_id, ')')) AS supplier"),
				'id'
			);
		} else {
			$all_contacts->select(
				'id',
				DB::raw("CONCAT(name, ' (', supplier_business_name, ')') as supplier")
			);
		}

		$suppliers = $all_contacts->pluck('supplier', 'id');

		//Prepend none
		if ($prepend_none) {
			$suppliers = $suppliers->prepend(__('lang_v1.none'), '');
		}

		return $suppliers;
	}

	public static function forDropdown($business_id, $show_none = false)
	{
		$contacts = Contact::where('business_id', $business_id)
			->pluck('name', 'id');

		if ($show_none) {
			$contacts->prepend(__('lang_v1.none'), '');
		}

		return $contacts;
	}

	/**
	 * Return list of customers dropdown for a business
	 *
	 * @param $business_id int
	 * @param $prepend_none = true (boolean)
	 *
	 * @return array users
	 */
	public static function customersDropdown($business_id, $prepend_none = true, $append_id = true)
	{
		$all_contacts = Contact::where('business_id', $business_id)
			->whereIn('type', ['customer', 'both'])
			->active();

		if ($append_id) {
			$all_contacts->select(
				DB::raw("IF(contact_id IS NULL OR contact_id='', name, CONCAT(name, ' (', contact_id, ')')) AS customer"),
				'id'
			);
		} else {
			$all_contacts->select('id', DB::raw("name as customer"));
		}

		$customers = $all_contacts->pluck('customer', 'id');

		//Prepend none
		if ($prepend_none) {
			$customers = $customers->prepend(__('lang_v1.none'), '');
		}

		return $customers;
	}

	/**
	 * Return list of contact type.
	 *
	 * @param $prepend_all = false (boolean)
	 * @return array
	 */
	public static function typeDropdown($prepend_all = false)
	{
		$types = [];

		if ($prepend_all) {
			$types[''] = __('lang_v1.all');
		}

		$types['customer'] = __('report.customer');
		$types['supplier'] = __('report.supplier');
		$types['both'] = __('lang_v1.both_supplier_customer');

		return $types;
	}

	/**
	 * Return list of contact type by permissions.
	 *
	 * @return array
	 */
	public static function getContactTypes()
	{
		$types = [];
		if (auth()->user()->can('supplier.create')) {
			$types['supplier'] = __('report.supplier');
		}
		if (auth()->user()->can('customer.create')) {
			$types['customer'] = __('report.customer');
		}
		if (auth()->user()->can('supplier.create') && auth()->user()->can('customer.create')) {
			$types['both'] = __('lang_v1.both_supplier_customer');
		}

		return $types;
	}

	public function getContactAddressAttribute()
	{
		$address_array = [];
		if (!empty($this->address_line_1)) {
			$address_array[] = $this->address_line_1;
		}
		if (!empty($this->address_line_2)) {
			$address_array[] = $this->address_line_2;
		}
		if (!empty($this->city)) {
			$address_array[] = $this->city;
		}
		if (!empty($this->state)) {
			$address_array[] = $this->state;
		}
		if (!empty($this->country)) {
			$address_array[] = $this->country;
		}

		$address = '';
		if (!empty($address_array)) {
			$address = implode(', ', $address_array);
		}
		if (!empty($this->zip_code)) {
			$address .= ',<br>' . $this->zip_code;
		}

		return $address;
	}
}
