<?php

namespace App\Imports;

use App\Models\Inventory;
use App\Models\Product;
use App\Models\InventoryProduct;
use App\Models\VariationLocationDetail;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\Importable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\ToModel;
use Illuminate\Support\Facades\Log;

class InventoryImport implements ToModel, WithHeadingRow, WithValidation
{
    use Importable;

    protected $location_id;
    protected $inventory = null;
    protected $sell_theoric_total = 0;
    protected $sell_physic_total = 0;

    public function __construct($location_id)
    {
        $this->location_id = $location_id;
    }

    /**
     * @param array $row
     *
     * @return \Illuminate\Database\Eloquent\Model|null
     */
    public function model(array $row)
    {
        // Debug - Afficher les en-têtes pour mieux comprendre le problème
        if ($this->inventory === null) {
            Log::info('Headers from imported file:', array_keys($row));
        }

        DB::beginTransaction();
        try {
            if ($this->inventory === null) {
                $this->inventory = Inventory::create([
                    'business_id' => Auth::user()->business_id,
                    'reference' => 'INVS-' . date('y') . substr(number_format(time() * mt_rand(), 0, '', ''), 0, 5),
                    'invent' => 'INVS-' . date('y') . substr(number_format(time() * mt_rand(), 0, '', ''), 0, 5),
                    'location_id' => $this->location_id,
                    'status' => 'closed',
                    'created_by' => Auth::id(),
                    'inv_create_at' => now(),
                    'sell_theoric_total' => 0,
                    'sell_physic_total' => 0
                ]);
            }

            // Correction: Utiliser le bon nom de colonne 'prix' tel qu'il apparaît dans l'en-tête
            // L'en-tête dans l'export est 'PRIX' qui devient 'prix' en minuscules après l'import
            $prix = $row['prix'] ?? 0;

            // Vérifier si le prix est vraiment 0 ou s'il y a un problème
            if ($prix == 0) {
                // Logging pour le débogage
                Log::warning('Prix is zero for product ID: ' . $row['product_id'], $row);

                // Solution alternative: essayer d'autres noms possibles de colonne
                $prix = $row['unit_price'] ?? $row['prix_unitaire'] ?? $row['prix'] ?? 0;

                // Si toujours 0, essayer de récupérer le prix depuis la base de données
                if ($prix == 0) {
                    $product = Product::find($row['product_id']);
                    if ($product) {
                        $variation = $product->variations()->first();
                        if ($variation) {
                            $prix = $variation->default_sell_price;
                            Log::info('Retrieved price from database: ' . $prix . ' for product ID: ' . $row['product_id']);
                        }
                    }
                }
            }

            // Calculer sell_theoric et sell_physic pour ce produit
            $stock_theorique = $row['stock_theorique'] ?? 0;
            $stock_physique = $row['stock_physique'] ?? 0;

            $sell_theoric = $stock_theorique * $prix;
            $sell_physic = $stock_physique * $prix;

            // Ajouter aux totaux
            $this->sell_theoric_total += $sell_theoric;
            $this->sell_physic_total += $sell_physic;

            // Créer l'inventaire produit
            $inventoryProduct = InventoryProduct::create([
                'inventaire_id' => $this->inventory->id,
                'product_id' => $row['product_id'],
                'stock_theorique' => $stock_theorique,
                'stock_physique' => $stock_physique,
                'ecart' => $stock_physique - $stock_theorique,
                'unit_price' => $prix,
                'sell_theoric' => $sell_theoric,
                'sell_physic' => $sell_physic,
                'observation' => $row['observation'] ?? null
            ]);

            // Mettre à jour le stock
            VariationLocationDetail::where('product_id', $row['product_id'])
                ->where('location_id', $this->location_id)
                ->update(['qty_available' => $stock_physique]);

            // Mettre à jour les totaux dans inventory après chaque ligne
            $this->inventory->update([
                'sell_theoric_total' => $this->sell_theoric_total,
                'sell_physic_total' => $this->sell_physic_total
            ]);

            DB::commit();
            return $inventoryProduct;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error importing inventory: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'row' => $row
            ]);
            throw $e;
        }
    }

    public function rules(): array
    {
        return [
            'product_id' => 'required|exists:products,id',
            'stock_theorique' => 'required|numeric|min:0',
            'stock_physique' => 'required|numeric|min:0',
            // Nous rendons la validation du prix plus flexible
            'observation' => 'nullable|string|max:191',
        ];
    }

    public function customValidationMessages()
    {
        return [
            'product_id.required' => 'L\'ID du produit est requis',
            'product_id.exists' => 'Produit invalide',
            'stock_theorique.required' => 'Le stock théorique est requis',
            'stock_theorique.numeric' => 'Le stock théorique doit être un nombre',
            'stock_physique.required' => 'Le stock physique est requis',
            'stock_physique.numeric' => 'Le stock physique doit être un nombre',
        ];
    }
}
