<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Warranty;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class WarrantyController extends BaseController
{
    /**
     * Display a listing of the warranties.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $business_id = $request->user()->business_id ?? session('user.business_id');

            $warranties = Warranty::where('business_id', $business_id)
                ->select(['id', 'name', 'description', 'duration', 'duration_type'])
                ->get()
                ->map(function ($warranty) {
                    $warranty->duration_formatted = $warranty->duration . ' ' . __('lang_v1.' . $warranty->duration_type);
                    return $warranty;
                });

            return $this->sendResponse($warranties, 'Warranties retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while retrieving warranties.', [], 500);
        }
    }

    /**
     * Store a newly created warranty in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'duration' => 'required|integer|min:1',
                'duration_type' => 'required|in:days,months,years'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $business_id = $request->user()->business_id ?? session('user.business_id');

            $input = $request->only(['name', 'description', 'duration', 'duration_type']);
            $input['business_id'] = $business_id;

            $warranty = Warranty::create($input);

            return $this->sendResponse($warranty, 'Warranty created successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while creating warranty.', [], 500);
        }
    }

    /**
     * Display the specified warranty.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request, $id)
    {
        try {
            $business_id = $request->user()->business_id ?? session('user.business_id');

            $warranty = Warranty::where('business_id', $business_id)->find($id);

            if (is_null($warranty)) {
                return $this->sendError('Warranty not found.');
            }

            $warranty->duration_formatted = $warranty->duration . ' ' . __('lang_v1.' . $warranty->duration_type);

            return $this->sendResponse($warranty, 'Warranty retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while retrieving warranty.', [], 500);
        }
    }

    /**
     * Update the specified warranty in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'duration' => 'required|integer|min:1',
                'duration_type' => 'required|in:days,months,years'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $business_id = $request->user()->business_id ?? session('user.business_id');

            $warranty = Warranty::where('business_id', $business_id)->find($id);

            if (is_null($warranty)) {
                return $this->sendError('Warranty not found.');
            }

            $input = $request->only(['name', 'description', 'duration', 'duration_type']);
            $warranty->update($input);

            return $this->sendResponse($warranty, 'Warranty updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while updating warranty.', [], 500);
        }
    }

    /**
     * Remove the specified warranty from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Request $request, $id)
    {
        try {
            $business_id = $request->user()->business_id ?? session('user.business_id');

            $warranty = Warranty::where('business_id', $business_id)->find($id);

            if (is_null($warranty)) {
                return $this->sendError('Warranty not found.');
            }

            $warranty->delete();

            return $this->sendResponse([], 'Warranty deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while deleting warranty.', [], 500);
        }
    }
}
