<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\ProductVariation;
use App\Models\Variation;
use App\Models\VariationTemplate;
use App\Models\VariationValueTemplate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class VariationTemplateController extends BaseController
{
    /**
     * Display a listing of variation templates
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $query = VariationTemplate::where('business_id', $business_id)
                ->with(['values'])
                ->select('id', 'name', DB::raw("(SELECT COUNT(id) FROM product_variations WHERE product_variations.variation_template_id=variation_templates.id) as total_products"));

            // Apply search filter
            if ($request->has('search')) {
                $search = $request->get('search');
                $query->where('name', 'like', "%{$search}%");
            }

            $variations = $query->orderBy('name')
                ->paginate($request->get('per_page', 15));

            // Transform data to include values as comma-separated string
            $variations->getCollection()->transform(function ($variation) {
                $values_arr = [];
                foreach ($variation->values as $value) {
                    $values_arr[] = $value->name;
                }
                $variation->values_string = implode(', ', $values_arr);
                return $variation;
            });

            return $this->sendResponse($variations, 'Variation templates retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Variation template index error: ' . $e->getMessage());
            return $this->sendError('Error retrieving variation templates', [], 500);
        }
    }

    /**
     * Store a newly created variation template
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:variation_templates,name,NULL,id,business_id,' . ($request->header('business-id') ?? $request->user()->business_id),
            'variation_values' => 'required|array|min:1',
            'variation_values.*' => 'required|string|max:255'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            DB::beginTransaction();

            $input = $request->only(['name']);
            $input['business_id'] = $business_id;

            $variation = VariationTemplate::create($input);

            // Create variation values
            if ($request->has('variation_values')) {
                $values = $request->get('variation_values');
                $data = [];
                foreach ($values as $value) {
                    if (!empty(trim($value))) {
                        $data[] = ['name' => trim($value)];
                    }
                }
                if (!empty($data)) {
                    $variation->values()->createMany($data);
                }
            }

            DB::commit();

            $variation->load('values');

            return $this->sendResponse($variation, 'Variation template created successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Variation template store error: ' . $e->getMessage());
            return $this->sendError('Error creating variation template', [], 500);
        }
    }

    /**
     * Display the specified variation template
     * 
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $variation = VariationTemplate::where('business_id', $business_id)
                ->with(['values'])
                ->find($id);

            if (!$variation) {
                return $this->sendError('Variation template not found', [], 404);
            }

            return $this->sendResponse($variation, 'Variation template retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Variation template show error: ' . $e->getMessage());
            return $this->sendError('Error retrieving variation template', [], 500);
        }
    }

    /**
     * Update the specified variation template
     * 
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $business_id = $request->header('business-id') ?? $request->user()->business_id;

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255|unique:variation_templates,name,' . $id . ',id,business_id,' . $business_id,
            'variation_values' => 'sometimes|array',
            'variation_values.*' => 'required|string|max:255',
            'edit_variation_values' => 'sometimes|array',
            'edit_variation_values.*' => 'required|string|max:255'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        try {
            DB::beginTransaction();

            $variation = VariationTemplate::where('business_id', $business_id)->find($id);

            if (!$variation) {
                return $this->sendError('Variation template not found', [], 404);
            }

            // Update variation template name
            if ($request->has('name') && $variation->name != $request->get('name')) {
                $variation->name = $request->get('name');
                $variation->save();

                // Update product variations that use this template
                ProductVariation::where('variation_template_id', $variation->id)
                    ->update(['name' => $variation->name]);
            }

            // Update existing variation values
            $data = [];
            if ($request->has('edit_variation_values')) {
                $values = $request->get('edit_variation_values');
                foreach ($values as $key => $value) {
                    if (!empty(trim($value))) {
                        $variation_val = VariationValueTemplate::find($key);
                        if ($variation_val && $variation_val->name != $value) {
                            $variation_val->name = trim($value);
                            $data[] = $variation_val;

                            // Update variations that use this value
                            Variation::where('variation_value_id', $key)
                                ->update(['name' => trim($value)]);
                        }
                    }
                }
                if (!empty($data)) {
                    $variation->values()->saveMany($data);
                }
            }

            // Add new variation values
            if ($request->has('variation_values')) {
                $values = $request->get('variation_values');
                $newValues = [];
                foreach ($values as $value) {
                    if (!empty(trim($value))) {
                        $newValues[] = new VariationValueTemplate(['name' => trim($value)]);
                    }
                }
                if (!empty($newValues)) {
                    $variation->values()->saveMany($newValues);
                }
            }

            DB::commit();

            $variation->load('values');

            return $this->sendResponse($variation, 'Variation template updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Variation template update error: ' . $e->getMessage());
            return $this->sendError('Error updating variation template', [], 500);
        }
    }

    /**
     * Remove the specified variation template
     * 
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $variation = VariationTemplate::where('business_id', $business_id)->find($id);

            if (!$variation) {
                return $this->sendError('Variation template not found', [], 404);
            }

            // Check if variation template is being used
            $productVariationsCount = ProductVariation::where('variation_template_id', $id)->count();
            if ($productVariationsCount > 0) {
                return $this->sendError('Cannot delete variation template - it is being used by products', [], 422);
            }

            $variation->delete();

            return $this->sendResponse([], 'Variation template deleted successfully');
        } catch (\Exception $e) {
            Log::error('Variation template destroy error: ' . $e->getMessage());
            return $this->sendError('Error deleting variation template', [], 500);
        }
    }

    /**
     * Get variation templates for dropdown
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getForDropdown(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $variations = VariationTemplate::where('business_id', $business_id)
                ->select(['id', 'name'])
                ->orderBy('name')
                ->get();

            return $this->sendResponse($variations, 'Variation templates for dropdown retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Variation template dropdown error: ' . $e->getMessage());
            return $this->sendError('Error retrieving variation templates', [], 500);
        }
    }

    /**
     * Get variation template with values
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getTemplateWithValues(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;
            $template_id = $request->get('template_id');

            if (!$template_id) {
                return $this->sendError('Template ID is required', [], 422);
            }

            $template = VariationTemplate::where('business_id', $business_id)
                ->where('id', $template_id)
                ->with(['values'])
                ->first();

            if (!$template) {
                return $this->sendError('Variation template not found', [], 404);
            }

            return $this->sendResponse($template, 'Variation template with values retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Get template with values error: ' . $e->getMessage());
            return $this->sendError('Error retrieving variation template', [], 500);
        }
    }

    /**
     * Get all variation templates (without pagination)
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAllVariationTemplates(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $variations = VariationTemplate::where('business_id', $business_id)
                ->with(['values'])
                ->select(['id', 'name'])
                ->orderBy('name')
                ->get();

            return $this->sendResponse($variations, 'All variation templates retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Get all variation templates error: ' . $e->getMessage());
            return $this->sendError('Error retrieving variation templates', [], 500);
        }
    }

    /**
     * Add variation value to existing template
     * 
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function addVariationValue(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $variation = VariationTemplate::where('business_id', $business_id)->find($id);

            if (!$variation) {
                return $this->sendError('Variation template not found', [], 404);
            }

            // Check if value already exists
            $existingValue = $variation->values()->where('name', $request->get('name'))->first();
            if ($existingValue) {
                return $this->sendError('Variation value already exists', [], 422);
            }

            $value = $variation->values()->create([
                'name' => $request->get('name')
            ]);

            return $this->sendResponse($value, 'Variation value added successfully');
        } catch (\Exception $e) {
            Log::error('Add variation value error: ' . $e->getMessage());
            return $this->sendError('Error adding variation value', [], 500);
        }
    }

    /**
     * Remove variation value from template
     * 
     * @param int $templateId
     * @param int $valueId
     * @return \Illuminate\Http\JsonResponse
     */
    public function removeVariationValue($templateId, $valueId)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $variation = VariationTemplate::where('business_id', $business_id)->find($templateId);

            if (!$variation) {
                return $this->sendError('Variation template not found', [], 404);
            }

            $value = $variation->values()->find($valueId);

            if (!$value) {
                return $this->sendError('Variation value not found', [], 404);
            }

            // Check if value is being used
            $usageCount = Variation::where('variation_value_id', $valueId)->count();
            if ($usageCount > 0) {
                return $this->sendError('Cannot delete variation value - it is being used by products', [], 422);
            }

            $value->delete();

            return $this->sendResponse([], 'Variation value removed successfully');
        } catch (\Exception $e) {
            Log::error('Remove variation value error: ' . $e->getMessage());
            return $this->sendError('Error removing variation value', [], 500);
        }
    }
}
