<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Media;
use App\Models\User;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class UserController extends BaseController
{
    /*
    |--------------------------------------------------------------------------
    | UserController
    |--------------------------------------------------------------------------
    |
    | Ce contrôleur gère la manipulation des utilisateurs
    |
    */

    protected $moduleUtil;

    /**
     * Constructeur
     *
     * @param ModuleUtil $moduleUtil
     * @return void
     */
    public function __construct(ModuleUtil $moduleUtil)
    {
        $this->moduleUtil = $moduleUtil; // Initialisation de l'instance de ModuleUtil
    }

    /**
     * Affiche le profil de l'utilisateur connecté
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProfile()
    {
        // Récupération de l'ID de l'utilisateur depuis la session
        $user_id = $user_id = auth()->user()->id;
        $user = User::where('id', $user_id)->with(['media'])->first();

        // Récupération des langues configurées
        $config_languages = config('constants.langs');
        $languages = [];
        foreach ($config_languages as $key => $value) {
            $languages[$key] = $value['full_name']; // Création d'un tableau de langues
        }

        return $this->sendResponse(compact('user', 'languages'), 'User profile retrieved successfully.');
    }

    /**
     * Met à jour le profil de l'utilisateur
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateProfile(Request $request)
    {
        // Désactivation dans la démo
        $notAllowed = $this->moduleUtil->notAllowedInDemo();
        if (!empty($notAllowed)) {
            return $this->sendError($notAllowed);
        }

        //try {
            $user_id = $user_id = auth()->user()->id;
            // Récupération des données du formulaire
            $input = $request->only([
                'surname',
                'first_name',
                'last_name',
                'email',
                'language',
                'marital_status',
                'blood_group',
                'contact_number',
                'fb_link',
                'twitter_link',
                'social_media_1',
                'social_media_2',
                'permanent_address',
                'current_address',
                'guardian_name',
                'custom_field_1',
                'custom_field_2',
                'custom_field_3',
                'custom_field_4',
                'id_proof_name',
                'id_proof_number',
                'gender'
            ]);

            if (!empty($request->input('dob'))) {
                $input['dob'] = $this->moduleUtil->uf_date($request->input('dob')); // Formatage de la date de naissance
            }
            if (!empty($request->input('bank_details'))) {
                $input['bank_details'] = json_encode($request->input('bank_details')); // Encodage des détails bancaires
            }

            // Mise à jour de l'utilisateur
            $user = User::find($user_id);
            $user->update($input);

            // Téléchargement de la photo de profil
            Media::uploadMedia($user->business_id, $user, request(), 'profile_photo', true);

            // Mise à jour de la session
            $input['id'] = $user_id;
            $input['business_id'] = auth()->user()->business_id;
            session()->put('user', $input);

            return $this->sendResponse(null, __('lang_v1.profile_updated_successfully'));
        //} catch (\Exception $e) {
        //    Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
        //    return $this->sendError(__('messages.something_went_wrong'));
        //}
    }

    /**
     * Met à jour le mot de passe de l'utilisateur
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updatePassword(Request $request)
    {
        // Désactivation dans la démo
        $notAllowed = $this->moduleUtil->notAllowedInDemo();
        if (!empty($notAllowed)) {
            return $this->sendError($notAllowed);
        }

        try {
            $user_id = auth()->user()->id;
            $user = User::where('id', $user_id)->first();

            // Vérification du mot de passe actuel
            if (Hash::check($request->input('current_password'), $user->password)) {
                $user->password = Hash::make($request->input('new_password')); // Mise à jour du mot de passe
                $user->save();
                return $this->sendResponse(null, __('lang_v1.password_updated_successfully'));
            } else {
                return $this->sendError(__('lang_v1.u_have_entered_wrong_password'));
            }
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError(__('messages.something_went_wrong'));
        }
    }
}
