<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Product;
use App\Models\Unit;
use App\Utils\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class UnitController extends BaseController
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of units
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        //try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $query = Unit::where('business_id', $business_id)
                ->with(['base_unit'])
                ->select(['actual_name', 'short_name', 'allow_decimal', 'id', 'base_unit_id', 'base_unit_multiplier']);

            // Apply search filter
            if ($request->has('search')) {
                $search = $request->get('search');
                $query->where(function ($q) use ($search) {
                    $q->where('actual_name', 'like', "%{$search}%")
                        ->orWhere('short_name', 'like', "%{$search}%");
                });
            }

            $units = $query->orderBy('actual_name')
                ->paginate($request->get('per_page', 15));

            // Transform the data to include formatted unit name
            $units->getCollection()->transform(function ($unit) {
                if (!empty($unit->base_unit_id)) {
                    $unit->formatted_name = $unit->actual_name . ' (' . (float)$unit->base_unit_multiplier . $unit->base_unit->short_name . ')';
                } else {
                    $unit->formatted_name = $unit->actual_name;
                }
                return $unit;
            });

            return $this->sendResponse($units, 'Units retrieved successfully');
        //} catch (\Exception $e) {
        //    Log::error('Unit index error: ' . $e->getMessage());
        //    return $this->sendError('Error retrieving units', [], 500);
        //}
    }

    /**
     * Store a newly created unit
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'actual_name' => 'required|string|max:255',
            'short_name' => 'required|string|max:10|unique:units,short_name,NULL,id,business_id,' . ($request->header('business-id') ?? $request->user()->business_id),
            'allow_decimal' => 'required|boolean',
            'base_unit_id' => 'nullable|exists:units,id',
            'base_unit_multiplier' => 'nullable|numeric|min:0.0001'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $input = $request->only(['actual_name', 'short_name', 'allow_decimal']);
            $input['business_id'] = $business_id;
            $input['created_by'] = auth()->user()->id;

            // Handle base unit setup
            if ($request->has('define_base_unit') && $request->get('define_base_unit')) {
                if ($request->has('base_unit_id') && $request->has('base_unit_multiplier')) {
                    $base_unit_multiplier = $this->commonUtil->num_uf($request->get('base_unit_multiplier'));
                    if ($base_unit_multiplier != 0) {
                        $input['base_unit_id'] = $request->get('base_unit_id');
                        $input['base_unit_multiplier'] = $base_unit_multiplier;
                    }
                }
            }

            $unit = Unit::create($input);
            $unit->load('base_unit');

            return $this->sendResponse($unit, 'Unit created successfully');
        } catch (\Exception $e) {
            Log::error('Unit store error: ' . $e->getMessage());
            return $this->sendError('Error creating unit', [], 500);
        }
    }

    /**
     * Display the specified unit
     * 
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $unit = Unit::where('business_id', $business_id)
                ->with(['base_unit'])
                ->find($id);

            if (!$unit) {
                return $this->sendError('Unit not found', [], 404);
            }

            return $this->sendResponse($unit, 'Unit retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Unit show error: ' . $e->getMessage());
            return $this->sendError('Error retrieving unit', [], 500);
        }
    }

    /**
     * Update the specified unit
     * 
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $business_id = $request->header('business-id') ?? $request->user()->business_id;

        $validator = Validator::make($request->all(), [
            'actual_name' => 'sometimes|required|string|max:255',
            'short_name' => 'sometimes|required|string|max:10|unique:units,short_name,' . $id . ',id,business_id,' . $business_id,
            'allow_decimal' => 'sometimes|required|boolean',
            'base_unit_id' => 'nullable|exists:units,id',
            'base_unit_multiplier' => 'nullable|numeric|min:0.0001'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        try {
            $unit = Unit::where('business_id', $business_id)->find($id);

            if (!$unit) {
                return $this->sendError('Unit not found', [], 404);
            }

            $input = $request->only(['actual_name', 'short_name', 'allow_decimal']);

            // Handle base unit setup
            if ($request->has('define_base_unit') && $request->get('define_base_unit')) {
                if ($request->has('base_unit_id') && $request->has('base_unit_multiplier')) {
                    $base_unit_multiplier = $this->commonUtil->num_uf($request->get('base_unit_multiplier'));
                    if ($base_unit_multiplier != 0) {
                        $input['base_unit_id'] = $request->get('base_unit_id');
                        $input['base_unit_multiplier'] = $base_unit_multiplier;
                    }
                }
            } else {
                $input['base_unit_id'] = null;
                $input['base_unit_multiplier'] = null;
            }

            $unit->update($input);
            $unit->load('base_unit');

            return $this->sendResponse($unit, 'Unit updated successfully');
        } catch (\Exception $e) {
            Log::error('Unit update error: ' . $e->getMessage());
            return $this->sendError('Error updating unit', [], 500);
        }
    }

    /**
     * Remove the specified unit
     * 
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $unit = Unit::where('business_id', $business_id)->find($id);

            if (!$unit) {
                return $this->sendError('Unit not found', [], 404);
            }

            // Check if unit has associated products
            $productsExist = Product::where('unit_id', $unit->id)->exists();
            if ($productsExist) {
                return $this->sendError('Cannot delete unit - products are using this unit', [], 422);
            }

            // Check if unit is used as base unit for other units
            $baseUnitUsage = Unit::where('base_unit_id', $unit->id)->exists();
            if ($baseUnitUsage) {
                return $this->sendError('Cannot delete unit - it is used as base unit for other units', [], 422);
            }

            $unit->delete();

            return $this->sendResponse([], 'Unit deleted successfully');
        } catch (\Exception $e) {
            Log::error('Unit destroy error: ' . $e->getMessage());
            return $this->sendError('Error deleting unit', [], 500);
        }
    }

    /**
     * Get units for dropdown
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getForDropdown(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;
            $include_sub_units = $request->get('include_sub_units', false);

            $query = Unit::where('business_id', $business_id)
                ->select(['id', 'actual_name', 'short_name', 'allow_decimal']);

            if ($include_sub_units) {
                $query->with(['base_unit']);
            }

            $units = $query->orderBy('actual_name')->get();

            return $this->sendResponse($units, 'Units for dropdown retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Unit dropdown error: ' . $e->getMessage());
            return $this->sendError('Error retrieving units', [], 500);
        }
    }

    /**
     * Get sub units for a given unit
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getSubUnits(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;
            $unit_id = $request->get('unit_id');

            if (!$unit_id) {
                return $this->sendError('Unit ID is required', [], 422);
            }

            // Using ProductUtil method if available, otherwise manual query
            if (method_exists($this->commonUtil, 'getSubUnits')) {
                $sub_units = $this->commonUtil->getSubUnits($business_id, $unit_id, true);
            } else {
                // Fallback: get units that have this unit as base unit
                $sub_units = Unit::where('business_id', $business_id)
                    ->where('base_unit_id', $unit_id)
                    ->select(['id', 'actual_name', 'short_name', 'base_unit_multiplier'])
                    ->get()
                    ->mapWithKeys(function ($unit) {
                        return [$unit->id => [
                            'name' => $unit->actual_name,
                            'short_name' => $unit->short_name,
                            'multiplier' => $unit->base_unit_multiplier
                        ]];
                    });
            }

            return $this->sendResponse($sub_units, 'Sub units retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Get sub units error: ' . $e->getMessage());
            return $this->sendError('Error retrieving sub units', [], 500);
        }
    }

    /**
     * Get all units (without pagination)
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAllUnits(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $units = Unit::where('business_id', $business_id)
                ->with(['base_unit'])
                ->select(['id', 'actual_name', 'short_name', 'allow_decimal', 'base_unit_id', 'base_unit_multiplier'])
                ->orderBy('actual_name')
                ->get();

            return $this->sendResponse($units, 'All units retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Get all units error: ' . $e->getMessage());
            return $this->sendError('Error retrieving units', [], 500);
        }
    }
}
