<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\BusinessLocation;
use App\Models\SellingPriceGroup;
use App\Models\TypesOfService;
use App\Utils\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class TypesOfServiceController extends BaseController
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    public function index()
    {
        if (!auth()->user()->can('access_types_of_service')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
        $types_of_services = TypesOfService::where('business_id', $business_id)->get();

        return $this->sendResponse($types_of_services, 'Types of service retrieved successfully.');
    }

    public function create()
    {
        if (!auth()->user()->can('access_types_of_service')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
        $locations = BusinessLocation::forDropdown($business_id);
        $price_groups = SellingPriceGroup::forDropdown($business_id);

        return $this->sendResponse(compact('locations', 'price_groups'), 'Data for creating types of service retrieved successfully.');
    }

    public function store(Request $request)
    {
        if (!auth()->user()->can('access_types_of_service')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $input = $request->only(['name', 'description', 'location_price_group', 'packing_charge_type', 'packing_charge']);
            $input['business_id'] = auth()->user()->business_id;
            $input['packing_charge'] = !empty($input['packing_charge']) ? $this->commonUtil->num_uf($input['packing_charge']) : 0;
            $input['enable_custom_fields'] = !empty($request->input('enable_custom_fields')) ? 1 : 0;

            TypesOfService::create($input);

            return $this->sendResponse(null, __("lang_v1.added_success"));
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError(__('messages.something_went_wrong'));
        }
    }

    public function edit($id)
    {
        if (!auth()->user()->can('access_types_of_service')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
        $locations = BusinessLocation::forDropdown($business_id);
        $price_groups = SellingPriceGroup::forDropdown($business_id);
        $type_of_service = TypesOfService::where('business_id', $business_id)->findOrFail($id);

        return $this->sendResponse(compact('locations', 'price_groups', 'type_of_service'), 'Data for editing type of service retrieved successfully.');
    }

    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('access_types_of_service')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $input = $request->only(['name', 'description', 'location_price_group', 'packing_charge_type', 'packing_charge']);
            $business_id = auth()->user()->business_id;
            $input['packing_charge'] = !empty($input['packing_charge']) ? $this->commonUtil->num_uf($input['packing_charge']) : 0;
            $input['enable_custom_fields'] = !empty($request->input('enable_custom_fields')) ? 1 : 0;
            $input['location_price_group'] = !empty($input['location_price_group']) ? json_encode($input['location_price_group']) : null;

            TypesOfService::where('business_id', $business_id)->where('id', $id)->update($input);

            return $this->sendResponse(null, __("lang_v1.updated_success"));
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError(__('messages.something_went_wrong'));
        }
    }

    public function destroy($id)
    {
        if (!auth()->user()->can('access_types_of_service')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            TypesOfService::where('business_id', $business_id)->where('id', $id)->delete();

            return $this->sendResponse(null, __("lang_v1.deleted_success"));
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError(__('messages.something_went_wrong'));
        }
    }
}
