<?php

namespace App\Http\Controllers;

use App\Events\TransactionPaymentAdded;
use App\Events\TransactionPaymentDeleted;

use App\Events\TransactionPaymentUpdated;
use App\Http\Controllers\Api\BaseController;
use App\Models\Transaction;
use App\Models\TransactionPayment;

use App\Utils\ModuleUtil;
use App\Utils\TransactionUtil;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TransactionPaymentController extends BaseController
{
    protected $transactionUtil;
    protected $moduleUtil;

    /**
     * Constructor
     *
     * @param TransactionUtil $transactionUtil
     * @return void
     */
    public function __construct(TransactionUtil $transactionUtil, ModuleUtil $moduleUtil)
    {
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
    }

    public function store(Request $request)
    {
        try {
            $business_id = auth()->user()->business_id;
            $transaction_id = $request->input('transaction_id');
            $transaction = Transaction::where('business_id', $business_id)
                ->with(['contact'])
                ->findOrFail($transaction_id);

            if (!auth()->user()->can('purchase.payments') || !auth()->user()->can('sell.payments')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            if ($transaction->payment_status != 'paid' && $transaction->payment_status != 'settle') {
                $inputs = $request->only([
                    'amount',
                    'method',
                    'note',
                    'card_number',
                    'card_holder_name',
                    'card_transaction_number',
                    'card_type',
                    'card_month',
                    'card_year',
                    'card_security',
                    'cheque_number',
                    'bank_account_number'
                ]);
                $inputs['paid_on'] = $this->transactionUtil->uf_date($request->input('paid_on'), true);
                $inputs['transaction_id'] = $transaction->id;
                $inputs['amount'] = $this->transactionUtil->num_uf($inputs['amount']);
                $inputs['created_by'] = auth()->user()->id;
                $inputs['payment_for'] = $transaction->contact_id;

                if ($inputs['method'] == 'custom_pay_1') {
                    $inputs['transaction_no'] = $request->input('transaction_no_1');
                } elseif ($inputs['method'] == 'custom_pay_2') {
                    $inputs['transaction_no'] = $request->input('transaction_no_2');
                } elseif ($inputs['method'] == 'custom_pay_3') {
                    $inputs['transaction_no'] = $request->input('transaction_no_3');
                }

                if (!empty($request->input('account_id')) && $inputs['method'] != 'advance') {
                    $inputs['account_id'] = $request->input('account_id');
                }

                $prefix_type = 'purchase_payment';
                if (in_array($transaction->type, ['sell', 'sell_return'])) {
                    $prefix_type = 'sell_payment';
                } elseif ($transaction->type == 'expense') {
                    $prefix_type = 'expense_payment';
                }

                DB::beginTransaction();

                $ref_count = $this->transactionUtil->setAndGetReferenceCount($prefix_type);
                $inputs['payment_ref_no'] = $this->transactionUtil->generateReferenceNumber($prefix_type, $ref_count);

                $inputs['business_id'] = $request->session()->get('business.id');
                $inputs['document'] = $this->transactionUtil->uploadFile($request, 'document', 'documents');

                $payment_amount = $inputs['amount'];
                $contact_balance = !empty($transaction->contact) ? $transaction->contact->balance : 0;
                if ($inputs['method'] == 'advance' && $inputs['amount'] > $contact_balance) {
                    throw new \Exception(__('lang_v1.required_advance_balance_not_available'));
                }

                if (!empty($inputs['amount'])) {
                    $tp = TransactionPayment::create($inputs);
                    $inputs['transaction_type'] = $transaction->type;
                    event(new TransactionPaymentAdded($tp, $inputs));
                }

                $this->transactionUtil->updatePaymentStatus($transaction_id, $transaction->final_total);
                DB::commit();
            }

            return $this->sendResponse([], __('purchase.payment_added_success'));
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError(__('messages.something_went_wrong'), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        if (!auth()->user()->can('purchase.create') && !auth()->user()->can('sell.create')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $transaction = Transaction::where('id', $id)
            ->with(['contact', 'business', 'transaction_for'])
            ->first();
        $payments_query = TransactionPayment::where('transaction_id', $id);

        $accounts_enabled = false;
        if ($this->moduleUtil->isModuleEnabled('account')) {
            $accounts_enabled = true;
            $payments_query->with(['payment_account']);
        }

        $payments = $payments_query->get();
        $payment_types = $this->transactionUtil->payment_types(null, true);

        return $this->sendResponse([
            'transaction' => $transaction,
            'payments' => $payments,
            'payment_types' => $payment_types,
            'accounts_enabled' => $accounts_enabled
        ], 'Payment details retrieved successfully.');
    }

    public function edit($id)
    {
        if (!auth()->user()->can('purchase.create') && !auth()->user()->can('sell.create')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
        $payment_line = TransactionPayment::where('method', '!=', 'advance')->findOrFail($id);
        $transaction = Transaction::where('id', $payment_line->transaction_id)
            ->where('business_id', $business_id)
            ->with(['contact', 'location'])
            ->first();
        $payment_types = $this->transactionUtil->payment_types($transaction->location);
        $accounts = $this->moduleUtil->accountsDropdown($business_id, true, false, true);

        return $this->sendResponse([
            'transaction' => $transaction,
            'payment_types' => $payment_types,
            'payment_line' => $payment_line,
            'accounts' => $accounts
        ], 'Payment edit data retrieved successfully.');
    }

    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('purchase.payments') && !auth()->user()->can('sell.payments')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $inputs = $request->only([
                'amount',
                'method',
                'note',
                'card_number',
                'card_holder_name',
                'card_transaction_number',
                'card_type',
                'card_month',
                'card_year',
                'card_security',
                'cheque_number',
                'bank_account_number'
            ]);
            $inputs['paid_on'] = $this->transactionUtil->uf_date($request->input('paid_on'), true);
            $inputs['amount'] = $this->transactionUtil->num_uf($inputs['amount']);

            if ($inputs['method'] == 'custom_pay_1') {
                $inputs['transaction_no'] = $request->input('transaction_no_1');
            } elseif ($inputs['method'] == 'custom_pay_2') {
                $inputs['transaction_no'] = $request->input('transaction_no_2');
            } elseif ($inputs['method'] == 'custom_pay_3') {
                $inputs['transaction_no'] = $request->input('transaction_no_3');
            }

            if (!empty($request->input('account_id'))) {
                $inputs['account_id'] = $request->input('account_id');
            }

            $payment = TransactionPayment::where('method', '!=', 'advance')->findOrFail($id);

            if (!empty($payment->parent_id)) {
                $parent_payment = TransactionPayment::find($payment->parent_id);
                $parent_payment->amount = $parent_payment->amount - ($payment->amount - $inputs['amount']);
                $parent_payment->save();
            }

            $business_id = auth()->user()->business_id;
            $transaction = Transaction::where('business_id', $business_id)->find($payment->transaction_id);

            $document_name = $this->transactionUtil->uploadFile($request, 'document', 'documents');
            if (!empty($document_name)) {
                $inputs['document'] = $document_name;
            }

            DB::beginTransaction();
            $payment->update($inputs);
            $this->transactionUtil->updatePaymentStatus($payment->transaction_id);
            DB::commit();

            event(new TransactionPaymentUpdated($payment, $transaction->type));

            return $this->sendResponse([], __('purchase.payment_updated_success'));
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError(__('messages.something_went_wrong'), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        if (!auth()->user()->can('purchase.payments') && !auth()->user()->can('sell.payments')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        if (request()->ajax()) {
            try {
                $payment = TransactionPayment::findOrFail($id);
                if (!empty($payment->parent_id)) {
                    $this->updateParentPaymentOnDelete($payment);
                }

                $payment->delete();
                $this->transactionUtil->updatePaymentStatus($payment->transaction_id);
                event(new TransactionPaymentDeleted($payment));

                return $this->sendResponse(null, __('purchase.payment_deleted_success'));
            } catch (\Exception $e) {
                Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
                return $this->sendError(__('messages.something_went_wrong'));
            }
        }

        return $this->sendError('Invalid request.');
    }

    public function addPayment($transaction_id)
    {
        if (!auth()->user()->can('purchase.payments') && !auth()->user()->can('sell.payments')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        if (request()->ajax()) {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $transaction = Transaction::where('business_id', $business_id)->with(['contact', 'location'])->findOrFail($transaction_id);
            if ($transaction->payment_status != 'paid' && $transaction->payment_status != 'settle') {
                $payment_types = $this->transactionUtil->payment_types($transaction->location, true);
                $paid_amount = $this->transactionUtil->getTotalPaid($transaction_id);
                $amount = max(0, $transaction->final_total - $paid_amount);
                $amount_formated = $this->transactionUtil->num_f($amount);

                $payment_line = new TransactionPayment();
                $payment_line->amount = $amount;
                $payment_line->method = 'cash';
                $payment_line->paid_on = Carbon::now()->toDateTimeString();
                $accounts = $this->moduleUtil->accountsDropdown($business_id, true, false, true);

                $view = view('transaction_payment.payment_row')->with(compact('transaction', 'payment_types', 'payment_line', 'amount_formated', 'accounts'))->render();
                return response()->json(['status' => 'due', 'view' => $view]);
            }

            return $this->sendError(__('purchase.amount_already_paid'));
        }
    }

    // Additional methods for handling custom payment methods and updating parent payments
    private function handleCustomPaymentMethods(&$inputs, $request)
    {
        if ($inputs['method'] == 'custom_pay_1') {
            $inputs['transaction_no'] = $request->input('transaction_no_1');
        } elseif ($inputs['method'] == 'custom_pay_2') {
            $inputs['transaction_no'] = $request->input('transaction_no_2');
        } elseif ($inputs['method'] == 'custom_pay_3') {
            $inputs['transaction_no'] = $request->input('transaction_no_3');
        }
    }

    private function determinePrefixType($transactionType)
    {
        if (in_array($transactionType, ['sell', 'sell_return'])) {
            return 'sell_payment';
        } elseif ($transactionType == 'expense') {
            return 'expense_payment';
        }
        return 'purchase_payment';
    }

    private function handleAdvancePayment(&$inputs, $transaction)
    {
        $payment_amount = $inputs['amount'];
        $contact_balance = !empty($transaction->contact) ? $transaction->contact->balance : 0;
        if ($inputs['method'] == 'advance' && $inputs['amount'] > $contact_balance) {
            throw new \Exception(__('lang_v1.required_advance_balance_not_available'));
        }
    }

    private function updateParentPayment($payment, $newAmount)
    {
        $parent_payment = TransactionPayment::find($payment->parent_id);
        $parent_payment->amount -= ($payment->amount - $newAmount);
        $parent_payment->save();
    }

    private function updateParentPaymentOnDelete($payment)
    {
        $parent_payment = TransactionPayment::find($payment->parent_id);
        $parent_payment->amount -= $payment->amount;
        if ($parent_payment->amount <= 0) {
            $parent_payment->delete();
        } else {
            $parent_payment->save();
        }
    }
}
