<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Category;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class TaxonomyController extends BaseController
{
    protected $moduleUtil;

    public function __construct(ModuleUtil $moduleUtil)
    {
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of categories
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;
            $category_type = $request->get('type', 'product');

            $query = Category::where('business_id', $business_id)
                ->where('category_type', $category_type)
                ->with(['parent'])
                ->select(['name', 'short_code', 'description', 'id', 'parent_id']);

            // Apply filters
            if ($request->has('search')) {
                $search = $request->get('search');
                $query->where('name', 'like', "%{$search}%");
            }

            if ($request->has('parent_id')) {
                $parent_id = $request->get('parent_id');
                $query->where('parent_id', $parent_id);
            }

            $categories = $query->orderBy('name')
                ->paginate($request->get('per_page', 15));

            return $this->sendResponse($categories, 'Categories retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Category index error: ' . $e->getMessage());
            return $this->sendError('Error retrieving categories', [], 500);
        }
    }

    /**
     * Store a newly created category
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'category_type' => 'required|string',
            'short_code' => 'nullable|string|max:10',
            'description' => 'nullable|string',
            'parent_id' => 'nullable|exists:categories,id'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $input = $request->only(['name', 'short_code', 'category_type', 'description']);

            if ($request->has('add_as_sub_cat') && $request->get('add_as_sub_cat') == 1 && $request->has('parent_id')) {
                $input['parent_id'] = $request->get('parent_id');
            } else {
                $input['parent_id'] = 0;
            }

            $input['business_id'] = $business_id;
            $input['created_by'] = $request->user()->id;

            $category = Category::create($input);
            $category->load('parent');

            return $this->sendResponse($category, 'Category created successfully');
        } catch (\Exception $e) {
            Log::error('Category store error: ' . $e->getMessage());
            return $this->sendError('Error creating category', [], 500);
        }
    }

    /**
     * Display the specified category
     * 
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $category = Category::where('business_id', $business_id)
                ->with(['parent', 'children'])
                ->find($id);

            if (!$category) {
                return $this->sendError('Category not found', [], 404);
            }

            return $this->sendResponse($category, 'Category retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Category show error: ' . $e->getMessage());
            return $this->sendError('Error retrieving category', [], 500);
        }
    }

    /**
     * Update the specified category
     * 
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'short_code' => 'nullable|string|max:10',
            'description' => 'nullable|string',
            'parent_id' => 'nullable|exists:categories,id'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $category = Category::where('business_id', $business_id)->find($id);

            if (!$category) {
                return $this->sendError('Category not found', [], 404);
            }

            $input = $request->only(['name', 'description', 'short_code']);

            if ($request->has('add_as_sub_cat') && $request->get('add_as_sub_cat') == 1 && $request->has('parent_id')) {
                $input['parent_id'] = $request->get('parent_id');
            } elseif ($request->has('parent_id')) {
                $input['parent_id'] = $request->get('parent_id') ?? 0;
            }

            $category->update($input);
            $category->load('parent');

            return $this->sendResponse($category, 'Category updated successfully');
        } catch (\Exception $e) {
            Log::error('Category update error: ' . $e->getMessage());
            return $this->sendError('Error updating category', [], 500);
        }
    }

    /**
     * Remove the specified category
     * 
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $category = Category::where('business_id', $business_id)->find($id);

            if (!$category) {
                return $this->sendError('Category not found', [], 404);
            }

            // Check if category has products
            $productsCount = $category->products()->count();
            if ($productsCount > 0) {
                return $this->sendError('Cannot delete category with associated products', [], 422);
            }

            // Check if category has subcategories
            $subcategoriesCount = Category::where('parent_id', $id)->count();
            if ($subcategoriesCount > 0) {
                return $this->sendError('Cannot delete category with subcategories', [], 422);
            }

            $category->delete();

            return $this->sendResponse([], 'Category deleted successfully');
        } catch (\Exception $e) {
            Log::error('Category destroy error: ' . $e->getMessage());
            return $this->sendError('Error deleting category', [], 500);
        }
    }

    /**
     * Get categories for dropdown
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getForDropdown(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;
            $category_type = $request->get('type', 'product');
            $parent_only = $request->get('parent_only', false);

            $query = Category::where('business_id', $business_id)
                ->where('category_type', $category_type)
                ->select(['id', 'name', 'parent_id']);

            if ($parent_only) {
                $query->where('parent_id', 0);
            }

            $categories = $query->orderBy('name')->get();

            return $this->sendResponse($categories, 'Categories for dropdown retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Category dropdown error: ' . $e->getMessage());
            return $this->sendError('Error retrieving categories', [], 500);
        }
    }

    /**
     * Get subcategories for a parent category
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getSubcategories(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;
            $parent_id = $request->get('parent_id');

            if (!$parent_id) {
                return $this->sendError('Parent ID is required', [], 422);
            }

            $subcategories = Category::where('business_id', $business_id)
                ->where('parent_id', $parent_id)
                ->select(['id', 'name'])
                ->orderBy('name')
                ->get();

            return $this->sendResponse($subcategories, 'Subcategories retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Get subcategories error: ' . $e->getMessage());
            return $this->sendError('Error retrieving subcategories', [], 500);
        }
    }

    /**
     * Get categories with subcategories tree
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCategoryTree(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;
            $category_type = $request->get('type', 'product');

            $categories = Category::catAndSubCategories($business_id, $category_type);

            return $this->sendResponse($categories, 'Category tree retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Category tree error: ' . $e->getMessage());
            return $this->sendError('Error retrieving category tree', [], 500);
        }
    }
}
