<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\BusinessLocation;
use App\Models\PurchaseLine;
use App\Models\Transaction;
use App\Models\TransactionSellLinesPurchaseLine;
use App\Utils\ModuleUtil;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class StockTransferController extends BaseController
{
    protected $productUtil;
    protected $transactionUtil;
    protected $moduleUtil;
    protected $status_colors;

    public function __construct(ProductUtil $productUtil, TransactionUtil $transactionUtil, ModuleUtil $moduleUtil)
    {
        $this->productUtil = $productUtil;
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
        $this->status_colors = [
            'in_transit' => 'bg-yellow',
            'completed' => 'bg-green',
            'pending' => 'bg-red',
            'assures' => 'bg-green',
            'personnel' => 'bg-orange',
            'indigents' => 'bg-blue',
            'dhl' => 'bg-purple',
        ];
    }

    /**
     * Display a listing of stock transfers
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id;

            $stock_transfers = Transaction::join('business_locations AS l1', 'transactions.location_id', '=', 'l1.id')
                ->join('transactions as t2', 't2.transfer_parent_id', '=', 'transactions.id')
                ->join('business_locations AS l2', 't2.location_id', '=', 'l2.id')
                ->leftJoin('users as u', 'transactions.created_by', '=', 'u.id')
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'sell_transfer')
                ->select([
                    'transactions.id',
                    'transactions.transaction_date',
                    'transactions.ref_no',
                    'l1.name as location_from',
                    'l2.name as location_to',
                    'transactions.final_total',
                    'transactions.shipping_charges',
                    'transactions.additional_notes',
                    'transactions.status',
                    DB::raw("CONCAT(COALESCE(u.surname, ''),' ',COALESCE(u.first_name, ''),' ',COALESCE(u.last_name,'')) as added_by")
                ]);

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $stock_transfers->whereIn('transactions.location_id', $permitted_locations);
            }

            // Apply filters
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $stock_transfers->whereDate('transactions.transaction_date', '>=', $request->start_date)
                    ->whereDate('transactions.transaction_date', '<=', $request->end_date);
            }

            if (!empty($request->location_id)) {
                $stock_transfers->where('transactions.location_id', $request->location_id);
            }

            if (!empty($request->status)) {
                $stock_transfers->where('transactions.status', $request->status);
            }

            $result = $stock_transfers->paginate($request->get('per_page', 15));

            $data = [
                'stock_transfers' => $result,
                'statuses' => $this->stockTransferStatuses()
            ];

            return $this->sendResponse($data, 'Stock transfers retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Failed to get stock transfers.', [$e->getMessage()], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if (!auth()->user()->can('purchase.create')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

        //Check if subscribed or not
        if (!$this->moduleUtil->isSubscribed($business_id)) {
            return $this->moduleUtil->expiredResponse(action('StockTransferController@index'));
        }

        $business_locations = BusinessLocation::forDropdown($business_id);

        $statuses = $this->stockTransferStatuses();

        $data = [
            'business_locations' => $business_locations,
            'statuses' => $statuses,
            'status_colors' => $this->status_colors
        ];
        return $this->sendResponse($data, 'Stock transfers retrieved successfully.');
    }

    /**
     * Store a newly created stock transfer
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            if (!auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id;

            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 403);
            }

            DB::beginTransaction();

            $input_data = $request->only(['location_id', 'ref_no', 'transaction_date', 'additional_notes', 'shipping_charges', 'final_total']);
            $status = $request->input('status');
            $user_id = $request->user()->id;

            $input_data['final_total'] = $this->productUtil->num_uf($input_data['final_total']);
            $input_data['total_before_tax'] = $input_data['final_total'];
            $input_data['type'] = 'sell_transfer';
            $input_data['business_id'] = $business_id;
            $input_data['created_by'] = $user_id;
            $input_data['transaction_date'] = $this->productUtil->uf_date($input_data['transaction_date'], true);
            $input_data['shipping_charges'] = $this->productUtil->num_uf($input_data['shipping_charges']);
            $input_data['payment_status'] = 'paid';
            $input_data['status'] = $status == 'completed' ? 'final' : $status;

            // Update reference count
            $ref_count = $this->productUtil->setAndGetReferenceCount('stock_transfer');
            if (empty($input_data['ref_no'])) {
                $input_data['ref_no'] = $this->productUtil->generateReferenceNumber('stock_transfer', $ref_count);
            }

            $products = $request->input('products');
            $sell_lines = [];
            $purchase_lines = [];

            if (!empty($products)) {
                foreach ($products as $product) {
                    $sell_line_arr = [
                        'product_id' => $product['product_id'],
                        'variation_id' => $product['variation_id'],
                        'quantity' => $this->productUtil->num_uf($product['quantity']),
                        'item_tax' => 0,
                        'tax_id' => null
                    ];

                    $purchase_line_arr = $sell_line_arr;
                    $sell_line_arr['unit_price'] = $this->productUtil->num_uf($product['unit_price']);
                    $sell_line_arr['unit_price_inc_tax'] = $sell_line_arr['unit_price'];

                    $purchase_line_arr['purchase_price'] = $sell_line_arr['unit_price'];
                    $purchase_line_arr['purchase_price_inc_tax'] = $sell_line_arr['unit_price'];

                    if (!empty($product['lot_no_line_id'])) {
                        $sell_line_arr['lot_no_line_id'] = $product['lot_no_line_id'];

                        $lot_details = PurchaseLine::find($product['lot_no_line_id']);
                        $purchase_line_arr['lot_number'] = $lot_details->lot_number;
                        $purchase_line_arr['mfg_date'] = $lot_details->mfg_date;
                        $purchase_line_arr['exp_date'] = $lot_details->exp_date;
                    }

                    $sell_lines[] = $sell_line_arr;
                    $purchase_lines[] = $purchase_line_arr;
                }
            }

            // Create Sell Transfer transaction
            $sell_transfer = Transaction::create($input_data);

            // Create Purchase Transfer at transfer location
            $input_data['type'] = 'purchase_transfer';
            $input_data['location_id'] = $request->input('transfer_location_id');
            $input_data['transfer_parent_id'] = $sell_transfer->id;
            $input_data['status'] = $status == 'completed' ? 'received' : $status;

            $purchase_transfer = Transaction::create($input_data);

            // Sell Product from first location
            if (!empty($sell_lines)) {
                $this->transactionUtil->createOrUpdateSellLines($sell_transfer, $sell_lines, $input_data['location_id']);
            }

            // Purchase product in second location
            if (!empty($purchase_lines)) {
                $purchase_transfer->purchase_lines()->createMany($purchase_lines);
            }

            // Handle stock movements based on status
            if (in_array($status, ['completed', 'assures', 'personnel', 'indigents', 'dhl'])) {
                foreach ($products as $product) {
                    if ($product['enable_stock']) {
                        $this->productUtil->decreaseProductQuantity(
                            $product['product_id'],
                            $product['variation_id'],
                            $sell_transfer->location_id,
                            $this->productUtil->num_uf($product['quantity'])
                        );

                        $this->productUtil->updateProductQuantity(
                            $purchase_transfer->location_id,
                            $product['product_id'],
                            $product['variation_id'],
                            $product['quantity']
                        );
                    }
                }

                $this->productUtil->adjustStockOverSelling($purchase_transfer);

                $business = [
                    'id' => $business_id,
                    'accounting_method' => $request->user()->business->accounting_method ?? 'fifo',
                    'location_id' => $sell_transfer->location_id
                ];
                $this->transactionUtil->mapPurchaseSell($business, $sell_transfer->sell_lines, 'purchase');
            }

            DB::commit();

            $data = [
                'sell_transfer' => $sell_transfer->load('sell_lines', 'location'),
                'purchase_transfer' => $purchase_transfer->load('purchase_lines', 'location')
            ];

            return $this->sendResponse($data, 'Stock transfer created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Failed to create stock transfer.', [$e->getMessage()], 500);
        }
    }


    /**
     * Checks if ref_number and supplier combination already exists.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function printInvoice($id)
    {
        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $sell_transfer = Transaction::where('business_id', $business_id)
                ->where('id', $id)
                ->where('type', 'sell_transfer')
                ->with(
                    'contact',
                    'sell_lines',
                    'sell_lines.product',
                    'sell_lines.variations',
                    'sell_lines.variations.product_variation',
                    'sell_lines.lot_details',
                    'location',
                    'sell_lines.product.unit'
                )
                ->first();

            $purchase_transfer = Transaction::where('business_id', $business_id)
                ->where('transfer_parent_id', $sell_transfer->id)
                ->where('type', 'purchase_transfer')
                ->first();

            $location_details = ['sell' => $sell_transfer->location, 'purchase' => $purchase_transfer->location];

            $lot_n_exp_enabled = false;
            if (request()->session()->get('business.enable_lot_number') == 1 || request()->session()->get('business.enable_product_expiry') == 1) {
                $lot_n_exp_enabled = true;
            }


            $output = ['success' => 1, 'receipt' => []];
            $output['receipt']['html_content'] = view('stock_transfer.print', compact('sell_transfer', 'location_details', 'lot_n_exp_enabled'))->render();
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            $output = [
                'success' => 0,
                'msg' => __('messages.something_went_wrong')
            ];
        }

        return $output;
    }

    /**
     * Display the specified stock transfer
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            if (!auth()->user()->can('purchase.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $stock_adjustment_details = Transaction::join('transaction_sell_lines as sl', 'sl.transaction_id', '=', 'transactions.id')
                ->join('products as p', 'sl.product_id', '=', 'p.id')
                ->join('variations as v', 'sl.variation_id', '=', 'v.id')
                ->join('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->where('transactions.id', $id)
                ->where('transactions.type', 'sell_transfer')
                ->leftjoin('purchase_lines as pl', 'sl.lot_no_line_id', '=', 'pl.id')
                ->select([
                    'p.name as product',
                    'p.type as type',
                    'pv.name as product_variation',
                    'v.name as variation',
                    'v.sub_sku',
                    'sl.quantity',
                    'sl.unit_price',
                    'pl.lot_number',
                    'pl.exp_date'
                ])
                ->groupBy('sl.id')
                ->get();

            $lot_n_exp_enabled = request()->user()->business->enable_lot_number == 1 ||
                request()->user()->business->enable_product_expiry == 1;

            $data = [
                'stock_adjustment_details' => $stock_adjustment_details,
                'lot_n_exp_enabled' => $lot_n_exp_enabled
            ];

            return $this->sendResponse($data, 'Stock transfer details retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Failed to get stock transfer details.', [$e->getMessage()], 500);
        }
    }

    /**
     * Update the specified stock transfer
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id;

            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 403);
            }

            $sell_transfer = Transaction::where('business_id', $business_id)
                ->where('type', 'sell_transfer')
                ->findOrFail($id);

            $purchase_transfer = Transaction::where('business_id', $business_id)
                ->where('transfer_parent_id', $id)
                ->where('type', 'purchase_transfer')
                ->with(['purchase_lines'])
                ->first();

            DB::beginTransaction();

            $input_data = $request->only(['transaction_date', 'additional_notes', 'shipping_charges', 'final_total']);
            $status = $request->input('status');

            $input_data['final_total'] = $this->productUtil->num_uf($input_data['final_total']);
            $input_data['total_before_tax'] = $input_data['final_total'];
            $input_data['transaction_date'] = $this->productUtil->uf_date($input_data['transaction_date'], true);
            $input_data['shipping_charges'] = $this->productUtil->num_uf($input_data['shipping_charges']);
            $input_data['status'] = $status == 'completed' ? 'final' : $status;

            $products = $request->input('products');
            $sell_lines = [];
            $purchase_lines = [];
            $edited_purchase_lines = [];

            if (!empty($products)) {
                foreach ($products as $product) {
                    $sell_line_arr = [
                        'product_id' => $product['product_id'],
                        'variation_id' => $product['variation_id'],
                        'quantity' => $this->productUtil->num_uf($product['quantity']),
                        'item_tax' => 0,
                        'tax_id' => null
                    ];

                    $purchase_line_arr = $sell_line_arr;
                    $sell_line_arr['unit_price'] = $this->productUtil->num_uf($product['unit_price']);
                    $sell_line_arr['unit_price_inc_tax'] = $sell_line_arr['unit_price'];

                    $purchase_line_arr['purchase_price'] = $sell_line_arr['unit_price'];
                    $purchase_line_arr['purchase_price_inc_tax'] = $sell_line_arr['unit_price'];

                    if (isset($product['transaction_sell_lines_id'])) {
                        $sell_line_arr['transaction_sell_lines_id'] = $product['transaction_sell_lines_id'];
                    }

                    if (!empty($product['lot_no_line_id'])) {
                        $sell_line_arr['lot_no_line_id'] = $product['lot_no_line_id'];

                        $lot_details = PurchaseLine::find($product['lot_no_line_id']);
                        $purchase_line_arr['lot_number'] = $lot_details->lot_number;
                        $purchase_line_arr['mfg_date'] = $lot_details->mfg_date;
                        $purchase_line_arr['exp_date'] = $lot_details->exp_date;
                    }

                    $sell_lines[] = $sell_line_arr;

                    $purchase_line = [];
                    foreach ($purchase_transfer->purchase_lines as $pl) {
                        if ($pl->variation_id == $purchase_line_arr['variation_id']) {
                            $pl->update($purchase_line_arr);
                            $edited_purchase_lines[] = $pl->id;
                            $purchase_line = $pl;
                            break;
                        }
                    }

                    if (empty($purchase_line)) {
                        $purchase_line = new PurchaseLine($purchase_line_arr);
                    }

                    $purchase_lines[] = $purchase_line;
                }
            }

            $sell_transfer->update($input_data);
            $input_data['status'] = $status == 'completed' ? 'received' : $status;
            $purchase_transfer->update($input_data);

            if (!empty($sell_lines)) {
                $this->transactionUtil->createOrUpdateSellLines($sell_transfer, $sell_lines, $sell_transfer->location_id);
            }

            if (!empty($purchase_lines)) {
                if (!empty($edited_purchase_lines)) {
                    PurchaseLine::where('transaction_id', $purchase_transfer->id)
                        ->whereNotIn('id', $edited_purchase_lines)
                        ->delete();
                }
                $purchase_transfer->purchase_lines()->saveMany($purchase_lines);
            }

            if ($status == 'completed') {
                foreach ($products as $product) {
                    if ($product['enable_stock']) {
                        $this->productUtil->decreaseProductQuantity(
                            $product['product_id'],
                            $product['variation_id'],
                            $sell_transfer->location_id,
                            $this->productUtil->num_uf($product['quantity'])
                        );

                        $this->productUtil->updateProductQuantity(
                            $purchase_transfer->location_id,
                            $product['product_id'],
                            $product['variation_id'],
                            $product['quantity']
                        );
                    }
                }

                $this->productUtil->adjustStockOverSelling($purchase_transfer);

                $business = [
                    'id' => $business_id,
                    'accounting_method' => $request->user()->business->accounting_method ?? 'fifo',
                    'location_id' => $sell_transfer->location_id
                ];
                $this->transactionUtil->mapPurchaseSell($business, $sell_transfer->sell_lines, 'purchase');
            }

            DB::commit();

            $data = [
                'sell_transfer' => $sell_transfer->load('sell_lines', 'location'),
                'purchase_transfer' => $purchase_transfer->load('purchase_lines', 'location')
            ];

            return $this->sendResponse($data, 'Stock transfer updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Failed to update stock transfer.', [$e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified stock transfer
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            if (!auth()->user()->can('purchase.delete')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $edit_days = request()->user()->business->transaction_edit_days ?? 0;
            if (!$this->transactionUtil->canBeEdited($id, $edit_days)) {
                return $this->sendError("Transaction cannot be edited after {$edit_days} days.", [], 400);
            }

            $sell_transfer = Transaction::where('id', $id)
                ->where('type', 'sell_transfer')
                ->with(['sell_lines'])
                ->first();

            if (!$sell_transfer) {
                return $this->sendError('Stock transfer not found.', [], 404);
            }

            $purchase_transfer = Transaction::where('transfer_parent_id', $sell_transfer->id)
                ->where('type', 'purchase_transfer')
                ->with(['purchase_lines'])
                ->first();

            $purchase_lines = $purchase_transfer->purchase_lines;
            foreach ($purchase_lines as $purchase_line) {
                if ($purchase_line->quantity_sold > 0) {
                    return $this->sendError('Stock transfer cannot be deleted as some stock has been sold.', [], 400);
                }
            }

            DB::beginTransaction();

            $sell_lines = $sell_transfer->sell_lines;
            $deleted_sell_purchase_ids = [];
            $products = [];

            foreach ($sell_lines as $sell_line) {
                $purchase_sell_line = TransactionSellLinesPurchaseLine::where('sell_line_id', $sell_line->id)->first();

                if (!empty($purchase_sell_line)) {
                    PurchaseLine::where('id', $purchase_sell_line->purchase_line_id)
                        ->decrement('quantity_sold', $sell_line->quantity);

                    $deleted_sell_purchase_ids[] = $purchase_sell_line->id;

                    if (isset($products[$sell_line->variation_id])) {
                        $products[$sell_line->variation_id]['quantity'] += $sell_line->quantity;
                        $products[$sell_line->variation_id]['product_id'] = $sell_line->product_id;
                    } else {
                        $products[$sell_line->variation_id]['quantity'] = $sell_line->quantity;
                        $products[$sell_line->variation_id]['product_id'] = $sell_line->product_id;
                    }
                }
            }

            if (!empty($products)) {
                foreach ($products as $key => $value) {
                    $this->productUtil->decreaseProductQuantity(
                        $products[$key]['product_id'],
                        $key,
                        $purchase_transfer->location_id,
                        $products[$key]['quantity']
                    );

                    $this->productUtil->updateProductQuantity(
                        $sell_transfer->location_id,
                        $products[$key]['product_id'],
                        $key,
                        $products[$key]['quantity']
                    );
                }
            }

            if (!empty($deleted_sell_purchase_ids)) {
                TransactionSellLinesPurchaseLine::whereIn('id', $deleted_sell_purchase_ids)->delete();
            }

            $sell_transfer->delete();
            $purchase_transfer->delete();

            DB::commit();

            return $this->sendResponse([], 'Stock transfer deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Failed to delete stock transfer.', [$e->getMessage()], 500);
        }
    }

    /**
     * Update stock transfer status
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateStatus(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('purchase.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id;

            $sell_transfer = Transaction::where('business_id', $business_id)
                ->where('type', 'sell_transfer')
                ->with(['sell_lines', 'sell_lines.product'])
                ->findOrFail($id);

            $purchase_transfer = Transaction::where('business_id', $business_id)
                ->where('transfer_parent_id', $id)
                ->where('type', 'purchase_transfer')
                ->with(['purchase_lines'])
                ->first();

            $status = $request->input('status');

            DB::beginTransaction();

            if ($status == 'completed' && $sell_transfer->status != 'completed') {
                foreach ($sell_transfer->sell_lines as $sell_line) {
                    if ($sell_line->product->enable_stock) {
                        $this->productUtil->decreaseProductQuantity(
                            $sell_line->product_id,
                            $sell_line->variation_id,
                            $sell_transfer->location_id,
                            $sell_line->quantity
                        );

                        $this->productUtil->updateProductQuantity(
                            $purchase_transfer->location_id,
                            $sell_line->product_id,
                            $sell_line->variation_id,
                            $sell_line->quantity,
                            0,
                            null,
                            false
                        );
                    }
                }

                $this->productUtil->adjustStockOverSelling($purchase_transfer);

                $business = [
                    'id' => $business_id,
                    'accounting_method' => $request->user()->business->accounting_method ?? 'fifo',
                    'location_id' => $sell_transfer->location_id
                ];
                $this->transactionUtil->mapPurchaseSell($business, $sell_transfer->sell_lines, 'purchase');
            }

            $purchase_transfer->status = $status == 'completed' ? 'received' : $status;
            $purchase_transfer->save();

            $sell_transfer->status = $status == 'completed' ? 'final' : $status;
            $sell_transfer->save();

            DB::commit();

            $data = [
                'sell_transfer' => $sell_transfer,
                'purchase_transfer' => $purchase_transfer
            ];

            return $this->sendResponse($data, 'Stock transfer status updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Failed to update stock transfer status.', [$e->getMessage()], 500);
        }
    }

    /**
     * Get stock transfer statuses
     * @return array
     */
    private function stockTransferStatuses()
    {
        return [
            'pending' => __('lang_v1.pending'),
            'in_transit' => __('lang_v1.in_transit'),
            'completed' => __('restaurant.completed'),
            'assures' => __('lang_v1.assures'),
            'personnel' => __('lang_v1.personnel'),
            'indigents' => __('lang_v1.indigents'),
            'dhl' => __('lang_v1.dhl')
        ];
    }
}
