<?php

namespace App\Http\Controllers;

use App\Exceptions\PurchaseSellMismatch;
use App\Http\Controllers\Api\BaseController;
use App\Models\BusinessLocation;
use App\Models\PurchaseLine;
use App\Models\Transaction;
use App\Utils\ModuleUtil;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class StockAdjustmentController extends BaseController
{
    /**
     * All Utils instance.
     */
    protected $productUtil;
    protected $transactionUtil;
    protected $moduleUtil;

    /**
     * Constructor
     *
     * @param ProductUtils $product
     * @return void
     */
    public function __construct(ProductUtil $productUtil, TransactionUtil $transactionUtil, ModuleUtil $moduleUtil)
    {
        $this->productUtil = $productUtil;
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = auth()->user()->business_id;

            $stock_adjustments = Transaction::join(
                'business_locations AS BL',
                'transactions.location_id',
                '=',
                'BL.id'
            )
                ->leftJoin('users as u', 'transactions.created_by', '=', 'u.id')
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'stock_adjustment')
                ->select(
                    'transactions.id',
                    'transaction_date',
                    'ref_no',
                    'BL.name as location_name',
                    'adjustment_type',
                    'final_total',
                    'total_amount_recovered',
                    'additional_notes',
                    'transactions.id as DT_RowId',
                    DB::raw("CONCAT(COALESCE(u.surname, ''),' ',COALESCE(u.first_name, ''),' ',COALESCE(u.last_name,'')) as added_by")
                );

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $stock_adjustments->whereIn('transactions.location_id', $permitted_locations);
            }

            $start_date = $request->get('start_date');
            $end_date = $request->get('end_date');
            if (!empty($start_date) && !empty($end_date)) {
                $stock_adjustments->whereBetween(DB::raw('date(transaction_date)'), [$start_date, $end_date]);
            }

            $location_id = $request->get('location_id');
            if (!empty($location_id)) {
                $stock_adjustments->where('transactions.location_id', $location_id);
            }

            // Pagination parameters
            $per_page = $request->get('per_page', 15);
            $page = $request->get('page', 1);

            $result = $stock_adjustments->paginate($per_page, ['*'], 'page', $page);

            // Transform data for API response
            $result->getCollection()->transform(function ($item) {
                $item->formatted_transaction_date = format_datetime($item->transaction_date);
                $item->formatted_final_total = number_format($item->final_total, 2);
                $item->formatted_total_amount_recovered = number_format($item->total_amount_recovered, 2);
                $item->adjustment_type_label = __('stock_adjustment.' . $item->adjustment_type);
                return $item;
            });

            return $this->sendResponse($result, 'Stock adjustments retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while retrieving stock adjustments.', [], 500);
        }
    }

    /**
     * Get data for creating a new stock adjustment.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create(Request $request)
    {
        if (!auth()->user()->can('purchase.create')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = auth()->user()->business_id;

            //Check if subscribed or not
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 403);
            }

            $business_locations = BusinessLocation::forDropdown($business_id);

            $data = [
                'business_locations' => $business_locations
            ];

            return $this->sendResponse($data, 'Stock adjustment creation data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('purchase.create')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            DB::beginTransaction();

            $input_data = $request->only(['location_id', 'transaction_date', 'adjustment_type', 'additional_notes', 'total_amount_recovered', 'final_total', 'ref_no']);
            $business_id = auth()->user()->business_id;

            //Check if subscribed or not
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 403);
            }

            $user_id = $user_id = auth()->user()->id;

            $input_data['type'] = 'stock_adjustment';
            $input_data['business_id'] = $business_id;
            $input_data['created_by'] = $user_id;
            $input_data['transaction_date'] = $this->productUtil->uf_date($input_data['transaction_date'], true);
            $input_data['total_amount_recovered'] = $this->productUtil->num_uf($input_data['total_amount_recovered']);

            //Update reference count
            $ref_count = $this->productUtil->setAndGetReferenceCount('stock_adjustment');
            //Generate reference number
            if (empty($input_data['ref_no'])) {
                $input_data['ref_no'] = $this->productUtil->generateReferenceNumber('stock_adjustment', $ref_count);
            }

            $products = $request->input('products');

            if (!empty($products)) {
                $product_data = [];

                foreach ($products as $product) {
                    $adjustment_line = [
                        'product_id' => $product['product_id'],
                        'variation_id' => $product['variation_id'],
                        'quantity' => $this->productUtil->num_uf($product['quantity']),
                        'unit_price' => $this->productUtil->num_uf($product['unit_price'])
                    ];
                    if (!empty($product['lot_no_line_id'])) {
                        //Add lot_no_line_id to stock adjustment line
                        $adjustment_line['lot_no_line_id'] = $product['lot_no_line_id'];
                    }
                    $product_data[] = $adjustment_line;

                    //Decrease available quantity
                    $this->productUtil->decreaseProductQuantity(
                        $product['product_id'],
                        $product['variation_id'],
                        $input_data['location_id'],
                        $this->productUtil->num_uf($product['quantity'])
                    );
                }

                $stock_adjustment = Transaction::create($input_data);
                $stock_adjustment->stock_adjustment_lines()->createMany($product_data);

                //Map Stock adjustment & Purchase.
                $business = [
                    'id' => $business_id,
                    'accounting_method' => $request->session()->get('business.accounting_method'),
                    'location_id' => $input_data['location_id']
                ];
                $this->transactionUtil->mapPurchaseSell($business, $stock_adjustment->stock_adjustment_lines, 'stock_adjustment');

                DB::commit();

                return $this->sendResponse($stock_adjustment, __('stock_adjustment.stock_adjustment_added_successfully'));
            } else {
                return $this->sendError('No products provided for stock adjustment.', [], 400);
            }
        } catch (\Exception $e) {
            DB::rollBack();

            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            $msg = trans("messages.something_went_wrong");

            if (get_class($e) == PurchaseSellMismatch::class) {
                $msg = $e->getMessage();
            }

            return $this->sendError($msg, [], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        if (!auth()->user()->can('purchase.view')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $stock_adjustment_details = Transaction::join(
                'stock_adjustment_lines as sl',
                'sl.transaction_id',
                '=',
                'transactions.id'
            )
                ->join('products as p', 'sl.product_id', '=', 'p.id')
                ->join('variations as v', 'sl.variation_id', '=', 'v.id')
                ->join('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->where('transactions.id', $id)
                ->where('transactions.type', 'stock_adjustment')
                ->leftjoin('purchase_lines as pl', 'sl.lot_no_line_id', '=', 'pl.id')
                ->select(
                    'p.name as product',
                    'p.type as type',
                    'pv.name as product_variation',
                    'v.name as variation',
                    'v.sub_sku',
                    'sl.quantity',
                    'sl.unit_price',
                    'pl.lot_number',
                    'pl.exp_date'
                )
                ->groupBy('sl.id')
                ->get();

            $lot_n_exp_enabled = false;
            if (request()->session()->get('business.enable_lot_number') == 1 || request()->session()->get('business.enable_product_expiry') == 1) {
                $lot_n_exp_enabled = true;
            }

            $data = [
                'stock_adjustment_details' => $stock_adjustment_details,
                'lot_n_exp_enabled' => $lot_n_exp_enabled
            ];

            return $this->sendResponse($data, 'Stock adjustment details retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while retrieving stock adjustment details.', [], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Transaction  $stockAdjustment
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        // Implementation for editing can be added here if needed
        return $this->sendError('Edit functionality not implemented yet.', [], 501);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        // Implementation for updating can be added here if needed
        return $this->sendError('Update functionality not implemented yet.', [], 501);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('purchase.delete')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            DB::beginTransaction();

            $stock_adjustment = Transaction::where('id', $id)
                ->where('type', 'stock_adjustment')
                ->with(['stock_adjustment_lines'])
                ->first();

            if (!$stock_adjustment) {
                return $this->sendError('Stock adjustment not found.', [], 404);
            }

            //Add deleted product quantity to available quantity
            $stock_adjustment_lines = $stock_adjustment->stock_adjustment_lines;
            if (!empty($stock_adjustment_lines)) {
                $line_ids = [];
                foreach ($stock_adjustment_lines as $stock_adjustment_line) {
                    $this->productUtil->updateProductQuantity(
                        $stock_adjustment->location_id,
                        $stock_adjustment_line->product_id,
                        $stock_adjustment_line->variation_id,
                        $this->productUtil->num_f($stock_adjustment_line->quantity)
                    );
                    $line_ids[] = $stock_adjustment_line->id;
                }

                $this->transactionUtil->mapPurchaseQuantityForDeleteStockAdjustment($line_ids);
            }
            $stock_adjustment->delete();

            DB::commit();

            return $this->sendResponse(null, __('stock_adjustment.delete_success'));
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            return $this->sendError(__('messages.something_went_wrong'), [], 500);
        }
    }

    /**
     * Return product rows
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProductRow(Request $request)
    {
        try {
            $row_index = $request->input('row_index');
            $variation_id = $request->input('variation_id');
            $location_id = $request->input('location_id');

            $business_id = auth()->user()->business_id;
            $product = $this->productUtil->getDetailsFromVariation($variation_id, $business_id, $location_id);
            $product->formatted_qty_available = $this->productUtil->num_f($product->qty_available);

            //Get lot number dropdown if enabled
            $lot_numbers = [];
            if ($request->session()->get('business.enable_lot_number') == 1 || $request->session()->get('business.enable_product_expiry') == 1) {
                $lot_number_obj = $this->transactionUtil->getLotNumbersFromVariation($variation_id, $business_id, $location_id, true);
                foreach ($lot_number_obj as $lot_number) {
                    $lot_number->qty_formated = $this->productUtil->num_f($lot_number->qty_available);
                    $lot_numbers[] = $lot_number;
                }
            }
            $product->lot_numbers = $lot_numbers;

            $data = [
                'product' => $product,
                'row_index' => $row_index
            ];

            return $this->sendResponse($data, 'Product row retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while retrieving product row.', [], 500);
        }
    }

    /**
     * Sets expired purchase line as stock adjustment
     *
     * @param int $purchase_line_id
     * @return \Illuminate\Http\JsonResponse
     */
    public function removeExpiredStock($purchase_line_id)
    {
        if (!auth()->user()->can('purchase.delete')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $purchase_line = PurchaseLine::where('id', $purchase_line_id)
                ->with(['transaction'])
                ->first();

            if (!$purchase_line) {
                return $this->sendError('Purchase line not found.', [], 404);
            }

            DB::beginTransaction();

            $qty_unsold = $purchase_line->quantity - $purchase_line->quantity_sold - $purchase_line->quantity_adjusted - $purchase_line->quantity_returned;
            $final_total = $purchase_line->purchase_price_inc_tax * $qty_unsold;

            $user_id = $user_id = auth()->user()->id;
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            //Update reference count
            $ref_count = $this->productUtil->setAndGetReferenceCount('stock_adjustment');

            $stock_adjstmt_data = [
                'type' => 'stock_adjustment',
                'business_id' => $business_id,
                'created_by' => $user_id,
                'transaction_date' => Carbon::now()->format('Y-m-d'),
                'total_amount_recovered' => 0,
                'location_id' => $purchase_line->transaction->location_id,
                'adjustment_type' => 'normal',
                'final_total' => $final_total,
                'ref_no' => $this->productUtil->generateReferenceNumber('stock_adjustment', $ref_count)
            ];

            //Create stock adjustment transaction
            $stock_adjustment = Transaction::create($stock_adjstmt_data);

            $stock_adjustment_line = [
                'product_id' => $purchase_line->product_id,
                'variation_id' => $purchase_line->variation_id,
                'quantity' => $qty_unsold,
                'unit_price' => $purchase_line->purchase_price_inc_tax,
                'removed_purchase_line' => $purchase_line->id
            ];

            //Create stock adjustment line with the purchase line
            $stock_adjustment->stock_adjustment_lines()->create($stock_adjustment_line);

            //Decrease available quantity
            $this->productUtil->decreaseProductQuantity(
                $purchase_line->product_id,
                $purchase_line->variation_id,
                $purchase_line->transaction->location_id,
                $qty_unsold
            );

            //Map Stock adjustment & Purchase.
            $business = [
                'id' => $business_id,
                'accounting_method' => request()->session()->get('business.accounting_method'),
                'location_id' => $purchase_line->transaction->location_id
            ];
            $this->transactionUtil->mapPurchaseSell($business, $stock_adjustment->stock_adjustment_lines, 'stock_adjustment', false, $purchase_line->id);

            DB::commit();

            return $this->sendResponse($stock_adjustment, __('lang_v1.stock_removed_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            $msg = trans("messages.something_went_wrong");

            if (get_class($e) == PurchaseSellMismatch::class) {
                $msg = $e->getMessage();
            }

            return $this->sendError($msg, [], 500);
        }
    }
}
