<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\SellingPriceGroup;
use App\Models\Variation;
use App\Models\VariationGroupPrice;
use App\Utils\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use Spatie\Permission\Models\Permission;

/**
 * @group Selling Price Groups
 *
 * APIs for managing selling price groups.
 */
class SellingPriceGroupController extends BaseController
{
    /**
     * All Utils instance.
     *
     */
    protected $commonUtil;

    /**
     * Constructor
     *
     * @param ProductUtils $product
     * @return void
     */
    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * List Selling Price Groups
     *
     * Returns a listing of selling price groups.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of selling price groups
     *   ],
     *   "message": "Selling price groups retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving selling price groups."
     * }
     */
    public function index(Request $request)
    {
        //try {
            $business_id = $request->input('business_id');

            $price_groups = SellingPriceGroup::where('business_id', $business_id)
                ->select(['name', 'description', 'id', 'is_active'])
                ->get();
                //dd($price_groups);

            return $this->sendResponse($price_groups, 'Selling price groups retrieved successfully.');
        //} catch (\Exception $e) {
        //    Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
        //    return $this->sendError('Error retrieving selling price groups.', [], 400);
        //}
    }

    /**
     * Create Selling Price Group
     *
     * Creates a new selling price group.
     *
     * @bodyParam business_id required The ID of the business. Example: 1
     * @bodyParam name required The name of the selling price group. Example: Wholesale
     * @bodyParam description The description of the selling price group. Example: For wholesale customers
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     // The newly created selling price group
     *   },
     *   "message": "Selling price group created successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error creating selling price group."
     * }
     */
    public function store(Request $request)
    {
        try {
            $input = $request->only(['name', 'description']);
            $business_id = auth()->user()->business_id ?? $request->input('business_id');
            $input['business_id'] = $business_id;

            $spg = SellingPriceGroup::create($input);

            //Create a new permission related to the created selling price group
            Permission::create(['name' => 'selling_price_group.' . $spg->id]);

            return $this->sendResponse($spg, 'Selling price group created successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error creating selling price group.', [], 400);
        }
    }

    /**
     * Get Selling Price Group Details for Editing
     *
     * Returns the details of a specific selling price group for editing.
     *
     * @urlParam id required The ID of the selling price group. Example: 1
     * @queryParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     // Selling price group details
     *   },
     *   "message": "Selling price group details retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving selling price group details."
     * }
     */
    public function edit(Request $request, $id)
    {
        try {
            $business_id = $request->input('business_id');
            $spg = SellingPriceGroup::where('business_id', $business_id)->find($id);

            if (!$spg) {
                return $this->sendError('Selling price group not found.', [], 404);
            }

            return $this->sendResponse($spg, 'Selling price group details retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving selling price group details.', [], 400);
        }
    }

    /**
     * Update Selling Price Group
     *
     * Updates the specified selling price group.
     *
     * @urlParam id required The ID of the selling price group. Example: 1
     * @bodyParam business_id required The ID of the business. Example: 1
     * @bodyParam name required The new name of the selling price group. Example: Retail
     * @bodyParam description The new description of the selling price group. Example: For retail customers
     *
     * @response {
     *   "success": true,
     *   "message": "Selling price group updated successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error updating selling price group."
     * }
     * @response 404 {
     *   "success": false,
     *   "message": "Selling price group not found."
     * }
     */
    public function update(Request $request, $id)
    {
        try {
            $input = $request->only(['name', 'description']);
            $business_id = $request->input('business_id');

            $spg = SellingPriceGroup::where('business_id', $business_id)->findOrFail($id);
            $spg->name = $input['name'];
            $spg->description = $input['description'];
            $spg->save();

            return $this->sendResponse([], 'Selling price group updated successfully.');
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->sendError('Selling price group not found.', [], 404);
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error updating selling price group.', [], 400);
        }
    }

    /**
     * Delete Selling Price Group
     *
     * Deletes the specified selling price group.
     *
     * @urlParam id required The ID of the selling price group. Example: 1
     * @queryParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "message": "Selling price group deleted successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error deleting selling price group."
     * }
     * @response 404 {
     *   "success": false,
     *   "message": "Selling price group not found."
     * }
     */
    public function destroy(Request $request, $id)
    {
        try {
            $business_id = $request->input('business_id');

            $spg = SellingPriceGroup::where('business_id', $business_id)->findOrFail($id);
            $spg->delete();

            return $this->sendResponse([], 'Selling price group deleted successfully.');
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->sendError('Selling price group not found.', [], 404);
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error deleting selling price group.', [], 400);
        }
    }

    /**
     * Activate/Deactivate Selling Price Group
     *
     * Activates or deactivates the specified selling price group.
     *
     * @urlParam id required The ID of the selling price group. Example: 1
     * @bodyParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "message": "Selling price group status updated successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error updating selling price group status."
     * }
     * @response 404 {
     *   "success": false,
     *   "message": "Selling price group not found."
     * }
     */
    public function activateDeactivate(Request $request, $id)
    {
        try {
            $business_id = $request->input('business_id');
            $spg = SellingPriceGroup::where('business_id', $business_id)->find($id);

            if (!$spg) {
                return $this->sendError('Selling price group not found.', [], 404);
            }

            $spg->is_active = $spg->is_active == 1 ? 0 : 1;
            $spg->save();

            return $this->sendResponse([], 'Selling price group status updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error updating selling price group status.', [], 400);
        }
    }

    /**
     * Exports selling price group prices for all the products in xls format
     *
     * @return \Illuminate\Http\Response
     */
    public function export()
    {
        $business_id = request()->user()->business_id;
        $price_groups = SellingPriceGroup::where('business_id', $business_id)->active()->get();

        $variations = Variation::join('products as p', 'variations.product_id', '=', 'p.id')
            ->join('product_variations as pv', 'variations.product_variation_id', '=', 'pv.id')
            ->where('p.business_id', $business_id)
            ->whereIn('p.type', ['single', 'variable'])
            ->select('sub_sku', 'p.name as product_name', 'variations.name as variation_name', 'p.type', 'variations.id', 'pv.name as product_variation_name', 'sell_price_inc_tax')
            ->with(['group_prices'])
            ->get();
        $export_data = [];
        foreach ($variations as $variation) {
            $temp = [];
            $temp['product'] = $variation->type == 'single' ? $variation->product_name : $variation->product_name . ' - ' . $variation->product_variation_name . ' - ' . $variation->variation_name;
            $temp['sku'] = $variation->sub_sku;
            $temp['Base Selling Price'] = $variation->sell_price_inc_tax;

            foreach ($price_groups as $price_group) {
                $price_group_id = $price_group->id;
                $variation_pg = $variation->group_prices->filter(function ($item) use ($price_group_id) {
                    return $item->price_group_id == $price_group_id;
                });

                $temp[$price_group->name] = $variation_pg->isNotEmpty() ? $variation_pg->first()->price_inc_tax : '';
            }
            $export_data[] = $temp;
        }

        if (ob_get_contents()) ob_end_clean();
        ob_start();
        return collect($export_data)->downloadExcel(
            'product_group_prices.xlsx',
            null,
            true
        );
    }

    /**
     * Imports the uploaded file to database.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function import(Request $request)
    {
        try {

            $notAllowed = $this->commonUtil->notAllowedInDemo();
            if (!empty($notAllowed)) {
                return $notAllowed;
            }

            //Set maximum php execution time
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', -1);

            if ($request->hasFile('product_group_prices')) {
                $file = $request->file('product_group_prices');

                $parsed_array = Excel::toArray([], $file);

                $headers = $parsed_array[0][0];

                //Remove header row
                $imported_data = array_splice($parsed_array[0], 1);

                $business_id = request()->user()->business_id;
                $price_groups = SellingPriceGroup::where('business_id', $business_id)->active()->get();

                //Get price group names from headers
                $imported_pgs = [];
                foreach ($headers as $key => $value) {
                    if (!empty($value) && $key > 2) {
                        $imported_pgs[$key] = $value;
                    }
                }

                $error_msg = '';
                DB::beginTransaction();
                foreach ($imported_data as $key => $value) {
                    $variation = Variation::where('sub_sku', $value[1])
                        ->first();
                    if (empty($variation)) {
                        $row = $key + 1;
                        $error_msg = __('lang_v1.product_not_found_exception', ['sku' => $value[1], 'row' => $row]);

                        throw new \Exception($error_msg);
                    }

                    foreach ($imported_pgs as $k => $v) {
                        $price_group = $price_groups->filter(function ($item) use ($v) {
                            return strtolower($item->name) == strtolower($v);
                        });

                        if ($price_group->isNotEmpty()) {
                            //Check if price is numeric
                            if (!is_null($value[$k]) && !is_numeric($value[$k])) {
                                $row = $key + 1;
                                $error_msg = __('lang_v1.price_group_non_numeric_exception', ['row' => $row]);

                                throw new \Exception($error_msg);
                            }

                            if (!is_null($value[$k])) {
                                VariationGroupPrice::updateOrCreate(
                                    [
                                        'variation_id' => $variation->id,
                                        'price_group_id' => $price_group->first()->id
                                    ],
                                    [
                                        'price_inc_tax' => $value[$k]
                                    ]
                                );
                            }
                        } else {
                            $row = $key + 1;
                            $error_msg = __('lang_v1.price_group_not_found_exception', ['pg' => $v, 'row' => $row]);

                            throw new \Exception($error_msg);
                        }
                    }
                }
                DB::commit();
            }
            $output = [
                'success' => 1,
                'msg' => __('lang_v1.product_grp_prices_imported_successfully')
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            $output = [
                'success' => 0,
                'msg' => $e->getMessage()
            ];
            return redirect('selling-price-group')->with('notification', $output);
        }

        return redirect('selling-price-group')->with('status', $output);
    }
}
