<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\BusinessLocation;
use App\Models\Transaction;
use App\Models\Contact;
use App\Models\User;
use App\Utils\BusinessUtil;
use App\Utils\ContactUtil;

use App\Utils\ModuleUtil;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SellReturnController extends BaseController
{
    /**
     * All Utils instance.
     *
     */
    protected $productUtil;
    protected $transactionUtil;
    protected $contactUtil;
    protected $businessUtil;
    protected $moduleUtil;

    /**
     * Constructor
     *
     * @param ProductUtils $product
     * @return void
     */
    public function __construct(ProductUtil $productUtil, TransactionUtil $transactionUtil, ContactUtil $contactUtil, BusinessUtil $businessUtil, ModuleUtil $moduleUtil)
    {
        $this->productUtil = $productUtil;
        $this->transactionUtil = $transactionUtil;
        $this->contactUtil = $contactUtil;
        $this->businessUtil = $businessUtil;
        $this->moduleUtil = $moduleUtil;
    }

    public function index(Request $request)
    {
        if (!auth()->user()->can('sell.view') && !auth()->user()->can('sell.create')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $business_id = auth()->user()->business_id;

        $sells = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
            ->join('business_locations AS bl', 'transactions.location_id', '=', 'bl.id')
            ->join('transactions as T1', 'transactions.return_parent_id', '=', 'T1.id')
            ->leftJoin('transaction_payments AS TP', 'transactions.id', '=', 'TP.transaction_id')
            ->where('transactions.business_id', $business_id)
            ->where('transactions.type', 'sell_return')
            ->where('transactions.status', 'final')
            ->select(
                'transactions.id',
                'transactions.transaction_date',
                'transactions.invoice_no',
                'contacts.name',
                'transactions.final_total',
                'transactions.payment_status',
                'bl.name as business_location',
                'T1.invoice_no as parent_sale',
                'T1.id as parent_sale_id',
                DB::raw('SUM(TP.amount) as amount_paid')
            );

        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $sells->whereIn('transactions.location_id', $permitted_locations);
        }

        if ($request->has('created_by') && !empty($request->created_by)) {
            $sells->where('transactions.created_by', $request->created_by);
        }

        if ($request->has('location_id') && !empty($request->location_id)) {
            $sells->where('transactions.location_id', $request->location_id);
        }

        if (!empty($request->customer_id)) {
            $sells->where('contacts.id', $request->customer_id);
        }

        if (!empty($request->start_date) && !empty($request->end_date)) {
            $sells->whereDate('transactions.transaction_date', '>=', $request->start_date)
                ->whereDate('transactions.transaction_date', '<=', $request->end_date);
        }

        $sells->groupBy('transactions.id');
        $result = $sells->get();

        return $this->sendResponse($result, 'Liste des retours de vente récupérée avec succès.');
    }

    public function add($id, Request $request)
    {
        if (!auth()->user()->can('sell.create')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $business_id = auth()->user()->business_id;

        if (!$this->moduleUtil->isSubscribed($business_id)) {
            return $this->sendError('Subscription expired.', [], 403);
        }

        $sell = Transaction::where('business_id', $business_id)
            ->with(['sell_lines', 'location', 'return_parent', 'contact', 'tax', 'sell_lines.sub_unit', 'sell_lines.product', 'sell_lines.product.unit'])
            ->find($id);

        if (!$sell) {
            return $this->sendError('Vente non trouvée.', [], 404);
        }

        foreach ($sell->sell_lines as $key => $value) {
            if (!empty($value->sub_unit_id)) {
                $formated_sell_line = $this->transactionUtil->recalculateSellLineTotals($business_id, $value);
                $sell->sell_lines[$key] = $formated_sell_line;
            }
            $sell->sell_lines[$key]->formatted_qty = $this->transactionUtil->num_f($value->quantity, false, null, true);
        }

        return $this->sendResponse($sell, 'Détails de la vente récupérés avec succès.');
    }

    public function store(Request $request)
    {
        if (!auth()->user()->can('sell.create')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $input = $request->except('_token');

            if (!empty($input['products'])) {
                $business_id = auth()->user()->business_id;

                if (!$this->moduleUtil->isSubscribed($business_id)) {
                    return $this->sendError('Subscription expired.', [], 403);
                }

                $user_id = $user_id = auth()->user()->id;

                $discount = [
                    'discount_type' => $input['discount_type'],
                    'discount_amount' => $input['discount_amount']
                ];

                $invoice_total = $this->productUtil->calculateInvoiceTotal($input['products'], $input['tax_id'], $discount);

                $sell = Transaction::where('business_id', $business_id)
                    ->with(['sell_lines', 'sell_lines.sub_unit'])
                    ->findOrFail($input['transaction_id']);

                $sell_return = Transaction::where('business_id', $business_id)
                    ->where('type', 'sell_return')
                    ->where('return_parent_id', $sell->id)
                    ->first();

                $sell_return_data = [
                    'transaction_date' => $this->productUtil->uf_date($request->input('transaction_date')),
                    'invoice_no' => $input['invoice_no'],
                    'discount_type' => $discount['discount_type'],
                    'discount_amount' => $this->productUtil->num_uf($input['discount_amount']),
                    'tax_id' => $input['tax_id'],
                    'tax_amount' => $invoice_total['tax'],
                    'total_before_tax' => $invoice_total['total_before_tax'],
                    'final_total' => $invoice_total['final_total'],
                    'payment_status' => "return"
                ];

                DB::beginTransaction();

                if (empty($sell_return_data['invoice_no'])) {
                    $ref_count = $this->productUtil->setAndGetReferenceCount('sell_return');
                    $sell_return_data['invoice_no'] = $this->productUtil->generateReferenceNumber('sell_return', $ref_count);
                }

                if (empty($sell_return)) {
                    $sell_return_data['business_id'] = $business_id;
                    $sell_return_data['location_id'] = $sell->location_id;
                    $sell_return_data['contact_id'] = $sell->contact_id;
                    $sell_return_data['customer_group_id'] = $sell->customer_group_id;
                    $sell_return_data['type'] = 'sell_return';
                    $sell_return_data['status'] = 'final';
                    $sell_return_data['payment_status'] = 'return';
                    $sell_return_data['created_by'] = $user_id;
                    $sell_return_data['return_parent_id'] = $sell->id;
                    $sell_return = Transaction::create($sell_return_data);
                } else {
                    $sell_return->update($sell_return_data);
                }

                if ($request->session()->get('business.enable_rp') == 1 && !empty($sell->rp_earned)) {
                    $is_reward_expired = $this->transactionUtil->isRewardExpired($sell->transaction_date, $business_id);
                    if (!$is_reward_expired) {
                        $diff = $sell->final_total - $sell_return->final_total;
                        $new_reward_point = $this->transactionUtil->calculateRewardPoints($business_id, $diff);
                        $this->transactionUtil->updateCustomerRewardPoints($sell->contact_id, $new_reward_point, $sell->rp_earned);
                        $sell->rp_earned = $new_reward_point;
                        $sell->save();
                    }
                }

                $this->transactionUtil->updatePaymentStatusReturn($sell_return->id, $sell_return->final_total);

                $returns = [];
                foreach ($request->input('products') as $product_line) {
                    $returns[$product_line['sell_line_id']] = $product_line['quantity'];
                }

                foreach ($sell->sell_lines as $sell_line) {
                    if (array_key_exists($sell_line->id, $returns)) {
                        $multiplier = !empty($sell_line->sub_unit) ? $sell_line->sub_unit->base_unit_multiplier : 1;
                        $quantity = $this->transactionUtil->num_uf($returns[$sell_line->id]) * $multiplier;
                        $quantity_before = $this->transactionUtil->num_f($sell_line->quantity_returned);
                        $quantity_formated = $this->transactionUtil->num_f($quantity);

                        $sell_line->quantity_returned = $quantity;
                        $sell_line->save();

                        $this->transactionUtil->updateQuantitySoldFromSellLine($sell_line, $quantity_formated, $quantity_before);
                        $this->productUtil->updateProductQuantity($sell_return->location_id, $sell_line->product_id, $sell_line->variation_id, $quantity_formated, $quantity_before);
                    }
                }

                DB::commit();

                return $this->sendResponse($sell_return, 'Retour de vente enregistré avec succès.');
            } else {
                return $this->sendError('Aucun produit fourni.', [], 400);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError(__('messages.something_went_wrong'), [], 500);
        }
    }

    public function show($id, Request $request)
    {
        if (!auth()->user()->can('sell.view')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $business_id = auth()->user()->business_id;
        $sell = Transaction::where('business_id', $business_id)
            ->where('id', $id)
            ->with(
                'contact',
                'return_parent',
                'tax',
                'sell_lines',
                'sell_lines.product',
                'sell_lines.variations',
                'sell_lines.sub_unit',
                'sell_lines.product.unit',
                'location'
            )
            ->first();

        if (!$sell) {
            return $this->sendError('Retour de vente non trouvé.', [], 404);
        }

        foreach ($sell->sell_lines as $key => $value) {
            if (!empty($value->sub_unit_id)) {
                $formated_sell_line = $this->transactionUtil->recalculateSellLineTotals($business_id, $value);
                $sell->sell_lines[$key] = $formated_sell_line;
            }
        }

        $sell_taxes = [];
        if (!empty($sell->return_parent->tax)) {
            if ($sell->return_parent->tax->is_tax_group) {
                $sell_taxes = $this->transactionUtil->sumGroupTaxDetails(
                    $this->transactionUtil->groupTaxDetails($sell->return_parent->tax, $sell->return_parent->tax_amount)
                );
            } else {
                $sell_taxes[$sell->return_parent->tax->name] = $sell->return_parent->tax_amount;
            }
        }

        $total_discount = 0;
        if ($sell->return_parent->discount_type == 'fixed') {
            $total_discount = $sell->return_parent->discount_amount;
        } elseif ($sell->return_parent->discount_type == 'percentage') {
            $discount_percent = $sell->return_parent->discount_amount;
            if ($discount_percent == 100) {
                $total_discount = $sell->return_parent->total_before_tax;
            } else {
                $total_after_discount = $sell->return_parent->final_total - $sell->return_parent->tax_amount;
                $total_before_discount = $total_after_discount * 100 / (100 - $discount_percent);
                $total_discount = $total_before_discount - $total_after_discount;
            }
        }

        return $this->sendResponse([
            'sell' => $sell,
            'sell_taxes' => $sell_taxes,
            'total_discount' => $total_discount
        ], 'Détails du retour de vente récupérés avec succès.');
    }

    /**
     * Génère le contenu d'un reçu de transaction
     *
     * @param  int     $business_id     ID de l'entreprise
     * @param  int     $location_id     ID du point de vente
     * @param  int     $transaction_id  ID de la transaction
     * @param  string  $printer_type    Type d'imprimante (optionnel)
     *
     * @return array
     */
    private function receiptContent($business_id, $location_id, $transaction_id, $printer_type = null)
    {
        // Structure de base de la réponse
        $output = [
            'is_enabled'      => false,
            'print_type'      => 'browser',
            'html_content'    => null,
            'printer_config'  => [],
            'data'            => []
        ];

        // Récupération des détails de l’entreprise et du point de vente
        $business_details = $this->businessUtil->getDetails($business_id);
        $location_details = BusinessLocation::find($location_id);

        // Vérifier si l’impression de la facture est activée
        if ($location_details && $location_details->print_receipt_on_invoice == 1) {
            $output['is_enabled'] = true;

            // Récupérer la mise en page de la facture
            $invoice_layout = $this->businessUtil->invoiceLayout(
                $business_id,
                $location_id,
                $location_details->invoice_layout_id
            );

            // Déterminer le type d’imprimante
            $receipt_printer_type = $printer_type ?? $location_details->receipt_printer_type;

            // Générer les détails du reçu
            $receipt_details = $this->transactionUtil->getReceiptDetails(
                $transaction_id,
                $location_id,
                $invoice_layout,
                $business_details,
                $location_details,
                $receipt_printer_type
            );

            // Si impression directe via imprimante
            if ($receipt_printer_type === 'printer') {
                $output['print_type'] = 'printer';
                $output['printer_config'] = $this->businessUtil->printerConfig(
                    $business_id,
                    $location_details->printer_id
                );
                $output['data'] = $receipt_details;
            } else {
                // Si impression via navigateur
                $output['html_content'] = view('sell_return.receipt', compact('receipt_details'))->render();
            }
        }

        return $output;
    }

    /**
     * Imprime une facture pour une vente
     *
     * @param  \Illuminate\Http\Request $request
     * @param  int $transaction_id
     * @return \Illuminate\Http\JsonResponse
     */
    public function printInvoice(Request $request, $transaction_id)
    {
        if ($request->ajax()) {
            try {
                $output = [
                    'success' => 0,
                    'msg'     => __("messages.something_went_wrong")
                ];

                $business_id = auth()->user()->business_id;

                // Récupération de la transaction
                $transaction = Transaction::where('business_id', $business_id)
                    ->where('id', $transaction_id)
                    ->first();

                // Si la transaction n’existe pas
                if (!$transaction) {
                    return response()->json($output);
                }

                // Génération du reçu
                $receipt = $this->receiptContent(
                    $business_id,
                    $transaction->location_id,
                    $transaction_id,
                    'browser'
                );

                // Si reçu généré avec succès
                if (!empty($receipt)) {
                    $output = [
                        'success' => 1,
                        'receipt' => $receipt
                    ];
                }
            } catch (\Exception $e) {
                Log::error("Erreur impression facture : " . $e->getMessage());
                $output = [
                    'success' => 0,
                    'msg'     => __("messages.something_went_wrong")
                ];
            }

            return response()->json($output);
        }
    }
}
