<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Account;
use App\Models\Business;
use App\Models\BusinessLocation;
use App\Models\Contact;
use App\Models\CustomerGroup;
use App\Models\InvoiceScheme;
use App\Models\SellingPriceGroup;
use App\Models\TaxRate;
use App\Models\Transaction;
use App\Models\TransactionSellLine;
use App\Models\TypesOfService;
use App\Models\User;
use App\Utils\BusinessUtil;
use App\Utils\ContactUtil;
use App\Utils\ModuleUtil;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use App\Models\Warranty;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SellController extends BaseController
{
    /**
     * All Utils instance.
     *
     */
    protected $contactUtil;
    protected $businessUtil;
    protected $transactionUtil;
    protected $productUtil;
    protected $moduleUtil;

    /**
     * Constructor
     *
     * @param ProductUtils $product
     * @return void
     */
    public function __construct(ContactUtil $contactUtil, BusinessUtil $businessUtil, TransactionUtil $transactionUtil, ModuleUtil $moduleUtil, ProductUtil $productUtil)
    {
        $this->contactUtil = $contactUtil;
        $this->businessUtil = $businessUtil;
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
        $this->productUtil = $productUtil;

        $this->dummyPaymentLine = [
            'method' => '',
            'amount' => 0,
            'note' => '',
            'card_transaction_number' => '',
            'card_number' => '',
            'card_type' => '',
            'card_holder_name' => '',
            'card_month' => '',
            'card_year' => '',
            'card_security' => '',
            'cheque_number' => '',
            'bank_account_number' => '',
            'is_return' => 0,
            'transaction_no' => ''
        ];

        $this->shipping_status_colors = [
            'ordered' => 'bg-yellow',
            'packed' => 'bg-info',
            'shipped' => 'bg-navy',
            'delivered' => 'bg-green',
            'cancelled' => 'bg-red',
        ];
    }

    /**
     * List Sells
     *
     * Returns a listing of sell transactions.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id Filter by location ID. Example: 2
     * @queryParam customer_id Filter by customer ID. Example: 3
     * @queryParam start_date Filter by start date. Example: 2023-01-01
     * @queryParam end_date Filter by end date. Example: 2023-12-31
     * @queryParam payment_status Filter by payment status (due, partial, paid). Example: paid
     * @queryParam created_by Filter by user ID who created the transaction. Example: 4
     * @queryParam commission_agent Filter by commission agent ID. Example: 5
     * @queryParam service_staffs Filter by service staff ID. Example: 6
     * @queryParam shipping_status Filter by shipping status. Example: shipped
     * @queryParam only_shipments Filter only shipments. Example: true
     * @queryParam suspended Filter suspended sales. Example: true
     * @queryParam transaction_sub_type Filter by transaction sub type. Example: pos
     * @queryParam rewards_only Filter only rewards transactions. Example: true
     * @queryParam only_woocommerce_sells Filter only woocommerce sells. Example: true
     * @queryParam only_subscriptions Filter only subscriptions. Example: true
     * @queryParam list_for Filter list for service staff report. Example: service_staff_report
     * @queryParam res_waiter_id Filter by waiter ID. Example: 7
     * @queryParam sub_type Filter by sub type. Example: pos
     * @queryParam sales_cmsn_agnt Filter by sales commission agent. Example: 8
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of sell transactions
     *   ],
     *   "message": "Sell transactions retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving sell transactions."
     * }
     */
    public function index(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $is_woocommerce = $this->moduleUtil->isModuleInstalled('Woocommerce');
            $is_tables_enabled = $this->transactionUtil->isModuleEnabled('tables');
            $is_service_staff_enabled = $this->transactionUtil->isModuleEnabled('service_staff');
            $is_types_service_enabled = $this->moduleUtil->isModuleEnabled('types_of_service');

            $payment_types = $this->transactionUtil->payment_types(null, true);
            $with = [];
            $shipping_statuses = $this->transactionUtil->shipping_statuses();
            $sells = $this->transactionUtil->getListSells($business_id);

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sells->whereIn('transactions.location_id', $permitted_locations);
            }

            //Add condition for created_by,used in sales representative sales report
            if ($request->has('created_by')) {
                $created_by = $request->get('created_by');
                if (!empty($created_by)) {
                    $sells->where('transactions.created_by', $created_by);
                }
            }

            if (!empty($request->input('payment_status')) && $request->input('payment_status') != 'overdue') {
                $sells->where('transactions.payment_status', $request->input('payment_status'));
            } elseif ($request->input('payment_status') == 'overdue') {
                $sells->whereIn('transactions.payment_status', ['due', 'partial'])
                    ->whereNotNull('transactions.pay_term_number')
                    ->whereNotNull('transactions.pay_term_type')
                    ->whereRaw("IF(transactions.pay_term_type='days', DATE_ADD(transactions.transaction_date, INTERVAL transactions.pay_term_number DAY) < CURDATE(), DATE_ADD(transactions.transaction_date, INTERVAL transactions.pay_term_number MONTH) < CURDATE())");
            }

            //Add condition for location,used in sales representative expense report
            if ($request->has('location_id')) {
                $location_id = $request->get('location_id');
                if (!empty($location_id)) {
                    $sells->where('transactions.location_id', $location_id);
                }
            }

            if (!empty($request->input('rewards_only')) && $request->input('rewards_only') == true) {
                $sells->where(function ($q) {
                    $q->whereNotNull('transactions.rp_earned')
                        ->orWhere('transactions.rp_redeemed', '>', 0);
                });
            }

            if (!empty($request->customer_id)) {
                $customer_id = $request->customer_id;
                $sells->where('contacts.id', $customer_id);
            }
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $start = $request->start_date;
                $end =  $request->end_date;
                $sells->whereDate('transactions.transaction_date', '>=', $start)
                    ->whereDate('transactions.transaction_date', '<=', $end);
            }

            //Check is_direct sell
            if ($request->has('is_direct_sale')) {
                $is_direct_sale = $request->is_direct_sale;
                if ($is_direct_sale == 0) {
                    $sells->where('transactions.is_direct_sale', 0);
                    $sells->whereNull('transactions.sub_type');
                }
            }

            //Add condition for commission_agent,used in sales representative sales with commission report
            if ($request->has('commission_agent')) {
                $commission_agent = $request->get('commission_agent');
                if (!empty($commission_agent)) {
                    $sells->where('transactions.commission_agent', $commission_agent);
                }
            }

            if ($is_woocommerce) {
                $sells->addSelect('transactions.woocommerce_order_id');
                if ($request->only_woocommerce_sells) {
                    $sells->whereNotNull('transactions.woocommerce_order_id');
                }
            }

            if ($request->only_subscriptions) {
                $sells->where(function ($q) {
                    $q->whereNotNull('transactions.recur_parent_id')
                        ->orWhere('transactions.is_recurring', 1);
                });
            }

            if (!empty($request->list_for) && $request->list_for == 'service_staff_report') {
                $sells->whereNotNull('transactions.res_waiter_id');
            }

            if (!empty($request->res_waiter_id)) {
                $sells->where('transactions.res_waiter_id', $request->res_waiter_id);
            }

            if (!empty($request->input('sub_type'))) {
                $sells->where('transactions.sub_type', $request->input('sub_type'));
            }

            if (!empty($request->input('created_by'))) {
                $sells->where('transactions.created_by', $request->input('created_by'));
            }

            if (!empty($request->input('sales_cmsn_agnt'))) {
                $sells->where('transactions.commission_agent', $request->input('sales_cmsn_agnt'));
            }

            if (!empty($request->input('service_staffs'))) {
                $sells->where('transactions.res_waiter_id', $request->input('service_staffs'));
            }
            $only_shipments = $request->only_shipments == 'true' ? true : false;
            if ($only_shipments) {
                $sells->whereNotNull('transactions.shipping_status');
            }

            if (!empty($request->input('shipping_status'))) {
                $sells->where('transactions.shipping_status', $request->input('shipping_status'));
            }

            $sells->groupBy('transactions.id');

            if (!empty($request->suspended)) {
                $transaction_sub_type = $request->get('transaction_sub_type');
                if (!empty($transaction_sub_type)) {
                    $sells->where('transactions.sub_type', $transaction_sub_type);
                } else {
                    $sells->where('transactions.sub_type', null);
                }

                $with = ['sell_lines'];

                if ($is_tables_enabled) {
                    $with[] = 'table';
                }

                if ($is_service_staff_enabled) {
                    $with[] = 'service_staff';
                }

                $sales = $sells->where('transactions.is_suspend', 1)
                    ->with($with)
                    ->addSelect('transactions.is_suspend', 'transactions.res_table_id', 'transactions.res_waiter_id', 'transactions.additional_notes')
                    ->get();

                return $this->sendResponse($sales, 'Suspended sales retrieved successfully.');
            }

            $with[] = 'payment_lines';
            if (!empty($with)) {
                $sells->with($with);
            }

            if ($this->businessUtil->isModuleEnabled('subscription')) {
                $sells->addSelect('transactions.is_recurring', 'transactions.recur_parent_id');
            }

            $sellsData = $sells->get();

            return $this->sendResponse($sellsData, 'Sell transactions retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving sell transactions.', [], 400);
        }
    }

    /**
     * Get Sell Details
     *
     * Returns the details of a specific sell transaction.
     *
     * @urlParam id required The ID of the sell transaction. Example: 1
     * @queryParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     // Sell transaction details
     *   },
     *   "message": "Sell transaction details retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving sell transaction details."
     * }
     */
    public function show(Request $request, $id)
    {
        try {
            $business_id = $request->input('business_id');
            $taxes = TaxRate::where('business_id', $business_id)
                ->pluck('name', 'id');
            $query = Transaction::where('business_id', $business_id)
                ->where('id', $id)
                ->with(['contact', 'cancel_person', 'sell_lines' => function ($q) {
                    $q->whereNull('parent_sell_line_id');
                }, 'sell_lines.product', 'sell_lines.product.unit', 'sell_lines.variations', 'sell_lines.variations.product_variation', 'payment_lines', 'sell_lines.modifiers', 'sell_lines.lot_details', 'tax', 'sell_lines.sub_unit', 'table', 'service_staff', 'sell_lines.service_staff', 'types_of_service', 'sell_lines.warranties']);

            $sell = $query->firstOrFail();

            foreach ($sell->sell_lines as $key => $value) {
                if (!empty($value->sub_unit_id)) {
                    $formated_sell_line = $this->transactionUtil->recalculateSellLineTotals($business_id, $value);
                    $sell->sell_lines[$key] = $formated_sell_line;
                }
            }

            $payment_types = $this->transactionUtil->payment_types(null, true);

            $order_taxes = [];
            if (!empty($sell->tax)) {
                if ($sell->tax->is_tax_group) {
                    $order_taxes = $this->transactionUtil->sumGroupTaxDetails($this->transactionUtil->groupTaxDetails($sell->tax, $sell->tax_amount));
                } else {
                    $order_taxes[$sell->tax->name] = $sell->tax_amount;
                }
            }

            $business_details = $this->businessUtil->getDetails($business_id);
            $pos_settings = empty($business_details->pos_settings) ? $this->businessUtil->defaultPosSettings() : json_decode($business_details->pos_settings, true);
            $shipping_statuses = $this->transactionUtil->shipping_statuses();
            $shipping_status_colors = $this->shipping_status_colors;
            $common_settings = session()->get('business.common_settings');
            $is_warranty_enabled = !empty($common_settings['enable_product_warranty']) ? true : false;

            $data = [
                'taxes' => $taxes,
                'sell' => $sell,
                'payment_types' => $payment_types,
                'order_taxes' => $order_taxes,
                'pos_settings' => $pos_settings,
                'shipping_statuses' => $shipping_statuses,
                'shipping_status_colors' => $shipping_status_colors,
                'is_warranty_enabled' => $is_warranty_enabled
            ];

            return $this->sendResponse($data, 'Sell transaction details retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving sell transaction details.', [], 400);
        }
    }

    /**
     * Update Shipping
     *
     * Updates shipping details for a specific sell transaction.
     *
     * @urlParam id required The ID of the sell transaction. Example: 1
     * @bodyParam business_id required The ID of the business. Example: 1
     * @bodyParam shipping_details The shipping details. Example: Test Shipping Details
     * @bodyParam shipping_address The shipping address. Example: Test Shipping Address
     * @bodyParam shipping_status The shipping status. Example: shipped
     * @bodyParam delivered_to The delivered to information. Example: Test Delivered To
     *
     * @response {
     *   "success": true,
     *   "message": "Shipping details updated successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error updating shipping details."
     * }
     */
    public function shipping_status_colors(Request $request, $id)
    {
        try {
            $input = $request->only([
                'shipping_details',
                'shipping_address',
                'shipping_status',
                'delivered_to'
            ]);
            $business_id = $request->input('business_id');

            $transaction = Transaction::where('business_id', $business_id)
                ->where('id', $id)
                ->update($input);

            return $this->sendResponse([], 'Shipping details updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            return $this->sendError('Error updating shipping details.', [], 400);
        }
    }

    /**
     * List Drafts
     *
     * Returns a listing of draft sell transactions.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id Filter by location ID. Example: 2
     * @queryParam customer_id Filter by customer ID. Example: 3
     * @queryParam start_date Filter by start date. Example: 2023-01-01
     * @queryParam end_date Filter by end date. Example: 2023-12-31
     * @queryParam created_by Filter by user ID who created the transaction. Example: 4
     * @queryParam is_quotation Filter by is_quotation. Example: 0
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of draft sell transactions
     *   ],
     *   "message": "Draft sell transactions retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving draft sell transactions."
     * }
     */
    public function getDrafts(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $is_quotation = $request->only('is_quotation', 0);

            $is_woocommerce = $this->moduleUtil->isModuleInstalled('Woocommerce');

            $sells = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
                ->join(
                    'business_locations AS bl',
                    'transactions.location_id',
                    '=',
                    'bl.id'
                )
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'sell')
                ->where('transactions.status', 'draft')
                ->where('is_quotation', $is_quotation)
                ->select(
                    'transactions.id',
                    'transaction_date',
                    'invoice_no',
                    'contacts.name',
                    'bl.name as business_location',
                    'is_direct_sale'
                );

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sells->whereIn('transactions.location_id', $permitted_locations);
            }

            if (!empty($request->start_date) && !empty($request->end_date)) {
                $start = $request->start_date;
                $end =  $request->end_date;
                $sells->whereDate('transaction_date', '>=', $start)
                    ->whereDate('transaction_date', '<=', $end);
            }

            if ($request->has('location_id')) {
                $location_id = $request->get('location_id');
                if (!empty($location_id)) {
                    $sells->where('transactions.location_id', $location_id);
                }
            }

            if ($request->has('created_by')) {
                $created_by = $request->get('created_by');
                if (!empty($created_by)) {
                    $sells->where('transactions.created_by', $created_by);
                }
            }

            if (!empty($request->customer_id)) {
                $customer_id = $request->customer_id;
                $sells->where('contacts.id', $customer_id);
            }

            if ($is_woocommerce) {
                $sells->addSelect('transactions.woocommerce_order_id');
            }

            $sells->groupBy('transactions.id');

            $drafts = $sells->get();

            return $this->sendResponse($drafts, 'Draft sell transactions retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving draft sell transactions.', [], 400);
        }
    }

    /**
     * List Final Sells
     *
     * Returns a listing of final sell transactions.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id Filter by location ID. Example: 2
     * @queryParam customer_id Filter by customer ID. Example: 3
     * @queryParam start_date Filter by start date. Example: 2023-01-01
     * @queryParam end_date Filter by end date. Example: 2023-12-31
     * @queryParam created_by Filter by user ID who created the transaction. Example: 4
     * @queryParam is_quotation Filter by is_quotation. Example: 0
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of final sell transactions
     *   ],
     *   "message": "Final sell transactions retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving final sell transactions."
     * }
     */
    public function getFinal(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $is_quotation = $request->only('is_quotation', 0);

            $is_woocommerce = $this->moduleUtil->isModuleInstalled('Woocommerce');

            $sells = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
                ->join(
                    'business_locations AS bl',
                    'transactions.location_id',
                    '=',
                    'bl.id'
                )
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'sell')
                ->where('transactions.status', 'cancel')
                ->where('is_quotation', $is_quotation)
                ->select(
                    'transactions.id',
                    'transaction_date',
                    'invoice_no',
                    'contacts.name',
                    'bl.name as business_location',
                    'is_direct_sale'
                );

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sells->whereIn('transactions.location_id', $permitted_locations);
            }

            if (!empty($request->start_date) && !empty($request->end_date)) {
                $start = $request->start_date;
                $end =  $request->end_date;
                $sells->whereDate('transaction_date', '>=', $start)
                    ->whereDate('transaction_date', '<=', $end);
            }

            if ($request->has('location_id')) {
                $location_id = $request->get('location_id');
                if (!empty($location_id)) {
                    $sells->where('transactions.location_id', $location_id);
                }
            }

            if ($request->has('created_by')) {
                $created_by = $request->get('created_by');
                if (!empty($created_by)) {
                    $sells->where('transactions.created_by', $created_by);
                }
            }

            if (!empty($request->customer_id)) {
                $customer_id = $request->customer_id;
                $sells->where('contacts.id', $customer_id);
            }

            if ($is_woocommerce) {
                $sells->addSelect('transactions.woocommerce_order_id');
            }

            $sells->groupBy('transactions.id');

            $finals = $sells->get();

            return $this->sendResponse($finals, 'Final sell transactions retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving final sell transactions.', [], 400);
        }
    }

    /**
     * List Quotations
     *
     * Returns a listing of quotation sell transactions.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id Filter by location ID. Example: 2
     * @queryParam customer_id Filter by customer ID. Example: 3
     * @queryParam start_date Filter by start date. Example: 2023-01-01
     * @queryParam end_date Filter by end date. Example: 2023-12-31
     * @queryParam created_by Filter by user ID who created the transaction. Example: 4
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of quotation sell transactions
     *   ],
     *   "message": "Quotation sell transactions retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving quotation sell transactions."
     * }
     */
    public function getQuotations(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $is_woocommerce = $this->moduleUtil->isModuleInstalled('Woocommerce');

            $sells = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
                ->join(
                    'business_locations AS bl',
                    'transactions.location_id',
                    '=',
                    'bl.id'
                )
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'sell')
                ->where('transactions.status', 'draft')
                ->where('is_quotation', 1)
                ->select(
                    'transactions.id',
                    'transaction_date',
                    'invoice_no',
                    'contacts.name',
                    'bl.name as business_location',
                    'is_direct_sale'
                );

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sells->whereIn('transactions.location_id', $permitted_locations);
            }

            if (!empty($request->start_date) && !empty($request->end_date)) {
                $start = $request->start_date;
                $end =  $request->end_date;
                $sells->whereDate('transaction_date', '>=', $start)
                    ->whereDate('transaction_date', '<=', $end);
            }

            if ($request->has('location_id')) {
                $location_id = $request->get('location_id');
                if (!empty($location_id)) {
                    $sells->where('transactions.location_id', $location_id);
                }
            }

            if ($request->has('created_by')) {
                $created_by = $request->get('created_by');
                if (!empty($created_by)) {
                    $sells->where('transactions.created_by', $created_by);
                }
            }

            if (!empty($request->customer_id)) {
                $customer_id = $request->customer_id;
                $sells->where('contacts.id', $customer_id);
            }

            if ($is_woocommerce) {
                $sells->addSelect('transactions.woocommerce_order_id');
            }

            $sells->groupBy('transactions.id');

            $quotations = $sells->get();

            return $this->sendResponse($quotations, 'Quotation sell transactions retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving quotation sell transactions.', [], 400);
        }
    }

    /**
     * Duplicate Sell
     *
     * Creates a copy of the requested sell transaction.
     *
     * @urlParam id required The ID of the sell transaction to duplicate. Example: 1
     * @bodyParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     // Duplicated sell transaction details
     *   },
     *   "message": "Sell transaction duplicated successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error duplicating sell transaction."
     * }
     */
    public function duplicateSell(Request $request, $id)
    {
        try {
            $business_id = $request->input('business_id');
            $user_id = 1; //$user_id = auth()->user()->id;

            $transaction = Transaction::where('business_id', $business_id)
                ->where('type', 'sell')
                ->findorfail($id);
            $duplicate_transaction_data = [];
            foreach ($transaction->toArray() as $key => $value) {
                if (!in_array($key, ['id', 'created_at', 'updated_at'])) {
                    $duplicate_transaction_data[$key] = $value;
                }
            }
            $duplicate_transaction_data['status'] = 'draft';
            $duplicate_transaction_data['payment_status'] = null;
            $duplicate_transaction_data['transaction_date'] =  Carbon::now();
            $duplicate_transaction_data['created_by'] = $user_id;
            $duplicate_transaction_data['invoice_token'] = null;

            DB::beginTransaction();
            $duplicate_transaction_data['invoice_no'] = $this->transactionUtil->getInvoiceNumber($business_id, 'draft', $duplicate_transaction_data['location_id']);

            //Create duplicate transaction
            $duplicate_transaction = Transaction::create($duplicate_transaction_data);

            //Create duplicate transaction sell lines
            $duplicate_sell_lines_data = [];

            foreach ($transaction->sell_lines as $sell_line) {
                $new_sell_line = [];
                foreach ($sell_line->toArray() as $key => $value) {
                    if (!in_array($key, ['id', 'transaction_id', 'created_at', 'updated_at', 'lot_no_line_id'])) {
                        $new_sell_line[$key] = $value;
                    }
                }

                $duplicate_sell_lines_data[] = $new_sell_line;
            }

            $duplicate_transaction->sell_lines()->createMany($duplicate_sell_lines_data);

            DB::commit();

            $output = [
                'success' => 0,
                'msg' => trans("lang_v1.duplicate_sell_created_successfully")
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            $output = [
                'success' => 0,
                'msg' => trans("messages.something_went_wrong")
            ];
        }

        if (!empty($duplicate_transaction)) {
            return $this->sendResponse($duplicate_transaction, 'Sell transaction duplicated successfully.');
        } else {
            return $this->sendError('Error duplicating sell transaction.', [], 400);
        }
    }

    /**
     * Edit Shipping (GET)
     *
     * Retrieves data required to display the shipping edit modal.
     *
     * @urlParam id required The ID of the sell transaction. Example: 1
     * @queryParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "transaction": { // Transaction data },
     *     "shipping_statuses": { // Shipping statuses array }
     *   },
     *   "message": "Shipping details data retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving shipping details data."
     * }
     */
    public function editShipping(Request $request, $id)
    {
        try {
            $business_id = $request->input('business_id');

            $transaction = Transaction::where('business_id', $business_id)
                ->findorfail($id);
            $shipping_statuses = $this->transactionUtil->shipping_statuses();

            $data = [
                'transaction' => $transaction,
                'shipping_statuses' => $shipping_statuses,
            ];

            return $this->sendResponse($data, 'Shipping details data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving shipping details data.', [], 400);
        }
    }

    /**
     * Update Shipping (PUT)
     *
     * Updates the shipping details for a specific sell transaction.
     *
     * @urlParam id required The ID of the sell transaction. Example: 1
     * @bodyParam business_id required The ID of the business. Example: 1
     * @bodyParam shipping_details string The shipping details. Example: Fragile, handle with care
     * @bodyParam shipping_address string The shipping address. Example: 123 Main St, Anytown
     * @bodyParam shipping_status string The shipping status. Example: shipped
     * @bodyParam delivered_to string Information about who the package was delivered to. Example: John Doe
     *
     * @response {
     *   "success": true,
     *   "message": "Shipping details updated successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error updating shipping details."
     * }
     */
    public function updateShipping(Request $request, $id)
    {
        try {
            $input = $request->only([
                'shipping_details',
                'shipping_address',
                'shipping_status',
                'delivered_to'
            ]);
            $business_id = $request->input('business_id');

            Transaction::where('business_id', $business_id)
                ->where('id', $id)
                ->update($input);

            return $this->sendResponse([], 'Shipping details updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error updating shipping details.', [], 400);
        }
    }

    /**
     * Shipments (GET)
     *
     * Retrieves data for displaying a list of shipments.  This API returns a list of data
     * that would be used to populate filters on a shipments listing page.  It does NOT return
     * the shipment data itself.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "shipping_statuses": { // Shipping statuses array },
     *     "business_locations": { // Business locations array },
     *     "customers": { // Customers array },
     *     "sales_representative": { // Sales representatives array },
     *     "is_service_staff_enabled": boolean,
     *     "service_staffs": { // Service staffs array }
     *   },
     *   "message": "Shipments data retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving shipments data."
     * }
     */
    public function shipments(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $shipping_statuses = $this->transactionUtil->shipping_statuses();
            $business_locations = BusinessLocation::forDropdown($business_id, false);
            $customers = Contact::customersDropdown($business_id, false);
            $sales_representative = User::forDropdown($business_id, false, false, true);
            $is_service_staff_enabled = $this->transactionUtil->isModuleEnabled('service_staff');
            $service_staffs = $this->productUtil->serviceStaffDropdown($business_id);

            $data = [
                'shipping_statuses' => $shipping_statuses,
                'business_locations' => $business_locations,
                'customers' => $customers,
                'sales_representative' => $sales_representative,
                'is_service_staff_enabled' => $is_service_staff_enabled,
                'service_staffs' => $service_staffs,
            ];

            return $this->sendResponse($data, 'Shipments data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving shipments data.', [], 400);
        }
    }
}
