<?php

namespace App\Http\Controllers;

//use App\Http\Controllers\Controller;

use App\Http\Controllers\Api\BaseController;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use App\Utils\Util;
use Yajra\DataTables\Facades\DataTables;

class SalesCommissionAgentController extends BaseController
{
    /**
     * All Utils instance.
     *
     */
    protected $commonUtil;

    /**
     * Constructor
     *
     * @param Util $commonUtil
     * @return void
     */
    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('user.view') && !auth()->user()->can('user.create')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }
        //try {

            $business_id = auth()->user()->business_id;

            $users = User::where('business_id', $business_id)
                ->where('is_cmmsn_agnt', 1)
                ->select([
                    'id',
                    DB::raw("CONCAT(COALESCE(surname, ''), ' ', COALESCE(first_name, ''), ' ', COALESCE(last_name, '')) as full_name"),
                    'email',
                    'contact_no',
                    'address',
                    'cmmsn_percent'
                ])
                ->get();

            // Transform data for API response
            $agents = $users->map(function ($user) {
                return [
                    'id' => $user->id,
                    'full_name' => $user->full_name,
                    'email' => $user->email,
                    'contact_no' => $user->contact_no,
                    'address' => $user->address,
                    'cmmsn_percent' => $user->cmmsn_percent,
                ];
            });

            return $this->sendResponse($agents, 'Commission agents retrieved successfully.');
        //} catch (\Exception $e) {
        //    Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
        //    return $this->sendError('Something went wrong.', [], 500);
        //}
    }

    /**
     * Display a listing of the resource for DataTables.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function datatable(Request $request)
    {
        try {
            if (!auth()->user()->can('user.view') && !auth()->user()->can('user.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $users = User::where('business_id', $business_id)
                ->where('is_cmmsn_agnt', 1)
                ->select([
                    'id',
                    DB::raw("CONCAT(COALESCE(surname, ''), ' ', COALESCE(first_name, ''), ' ', COALESCE(last_name, '')) as full_name"),
                    'email',
                    'contact_no',
                    'address',
                    'cmmsn_percent'
                ]);

            return DataTables::of($users)
                ->addColumn('action', function ($user) {
                    $actions = [];

                    if (auth()->user()->can('user.update')) {
                        $actions[] = [
                            'type' => 'edit',
                            'url' => route('api.sales-commission-agents.show', $user->id),
                            'label' => 'Edit'
                        ];
                    }

                    if (auth()->user()->can('user.delete')) {
                        $actions[] = [
                            'type' => 'delete',
                            'url' => route('api.sales-commission-agents.destroy', $user->id),
                            'label' => 'Delete'
                        ];
                    }

                    return $actions;
                })
                ->filterColumn('full_name', function ($query, $keyword) {
                    $query->whereRaw("CONCAT(COALESCE(surname, ''), ' ', COALESCE(first_name, ''), ' ', COALESCE(last_name, '')) like ?", ["%{$keyword}%"]);
                })
                ->removeColumn('id')
                ->make(true);
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            if (!auth()->user()->can('user.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            // Validation rules
            $validator = Validator::make($request->all(), [
                'surname' => 'nullable|string|max:255',
                'first_name' => 'required|string|max:255',
                'last_name' => 'nullable|string|max:255',
                'email' => 'nullable|email|unique:users,email',
                'contact_no' => 'nullable|string|max:20',
                'address' => 'nullable|string',
                'cmmsn_percent' => 'required|numeric|min:0|max:100'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $input = $request->only(['surname', 'first_name', 'last_name', 'email', 'address', 'contact_no', 'cmmsn_percent']);
            $input['cmmsn_percent'] = $this->commonUtil->num_uf($input['cmmsn_percent']);
            $business_id = auth()->user()->business_id;
            $input['business_id'] = $business_id;
            $input['allow_login'] = 0;
            $input['is_cmmsn_agnt'] = 1;

            $user = User::create($input);

            // Transform user data for response
            $agent = [
                'id' => $user->id,
                'full_name' => trim($user->surname . ' ' . $user->first_name . ' ' . $user->last_name),
                'email' => $user->email,
                'contact_no' => $user->contact_no,
                'address' => $user->address,
                'cmmsn_percent' => $user->cmmsn_percent,
            ];

            return $this->sendResponse($agent, 'Commission agent added successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id, Request $request)
    {
        try {
            if (!auth()->user()->can('user.view') && !auth()->user()->can('user.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $user = User::where('id', $id)
                ->where('business_id', $business_id)
                ->where('is_cmmsn_agnt', 1)
                ->first();

            if (!$user) {
                return $this->sendError('Commission agent not found.', [], 404);
            }

            $agent = [
                'id' => $user->id,
                'surname' => $user->surname,
                'first_name' => $user->first_name,
                'last_name' => $user->last_name,
                'full_name' => trim($user->surname . ' ' . $user->first_name . ' ' . $user->last_name),
                'email' => $user->email,
                'contact_no' => $user->contact_no,
                'address' => $user->address,
                'cmmsn_percent' => $user->cmmsn_percent,
            ];

            return $this->sendResponse($agent, 'Commission agent retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('user.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            // Validation rules
            $validator = Validator::make($request->all(), [
                'surname' => 'nullable|string|max:255',
                'first_name' => 'required|string|max:255',
                'last_name' => 'nullable|string|max:255',
                'email' => 'nullable|email|unique:users,email,' . $id,
                'contact_no' => 'nullable|string|max:20',
                'address' => 'nullable|string',
                'cmmsn_percent' => 'required|numeric|min:0|max:100'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $input = $request->only(['surname', 'first_name', 'last_name', 'email', 'address', 'contact_no', 'cmmsn_percent']);
            $input['cmmsn_percent'] = $this->commonUtil->num_uf($input['cmmsn_percent']);
            $business_id = auth()->user()->business_id;

            $user = User::where('id', $id)
                ->where('business_id', $business_id)
                ->where('is_cmmsn_agnt', 1)
                ->first();

            if (!$user) {
                return $this->sendError('Commission agent not found.', [], 404);
            }

            $user->update($input);

            // Transform user data for response
            $agent = [
                'id' => $user->id,
                'full_name' => trim($user->surname . ' ' . $user->first_name . ' ' . $user->last_name),
                'email' => $user->email,
                'contact_no' => $user->contact_no,
                'address' => $user->address,
                'cmmsn_percent' => $user->cmmsn_percent,
            ];

            return $this->sendResponse($agent, 'Commission agent updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id, Request $request)
    {
        try {
            if (!auth()->user()->can('user.delete')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $user = User::where('id', $id)
                ->where('business_id', $business_id)
                ->where('is_cmmsn_agnt', 1)
                ->first();

            if (!$user) {
                return $this->sendError('Commission agent not found.', [], 404);
            }

            $user->delete();

            return $this->sendResponse([], 'Commission agent deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Get commission agents for dropdown/select
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function dropdown(Request $request)
    {
        try {
            if (!auth()->user()->can('user.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $agents = User::where('business_id', $business_id)
                ->where('is_cmmsn_agnt', 1)
                ->select([
                    'id',
                    DB::raw("CONCAT(COALESCE(surname, ''), ' ', COALESCE(first_name, ''), ' ', COALESCE(last_name, '')) as full_name")
                ])
                ->get()
                ->map(function ($agent) {
                    return [
                        'id' => $agent->id,
                        'name' => $agent->full_name
                    ];
                });

            return $this->sendResponse($agents, 'Commission agents dropdown retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }
}
