<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\SellingPriceGroup;
use App\Utils\ModuleUtil;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class RoleController extends BaseController
{
    /**
     * All Utils instance.
     *
     */
    protected $moduleUtil;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(ModuleUtil $moduleUtil)
    {
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('roles.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $roles = Role::where('business_id', $business_id)
                ->select(['name', 'id', 'is_default', 'business_id', 'is_service_staff'])
                ->get()
                ->map(function ($role) use ($business_id) {
                    // Format role name
                    $role_name = str_replace('#' . $business_id, '', $role->name);
                    if (in_array($role_name, ['Admin', 'Cashier'])) {
                        $role_name = __('lang_v1.' . $role_name);
                    }

                    // Determine available actions
                    $actions = [];
                    if (!$role->is_default || $role->name == "Cashier#" . $business_id) {
                        if (auth()->user()->can('roles.update')) {
                            $actions[] = 'edit';
                        }
                        if (auth()->user()->can('roles.delete')) {
                            $actions[] = 'delete';
                        }
                    }

                    return [
                        'id' => $role->id,
                        'name' => $role_name,
                        'is_default' => $role->is_default,
                        'is_service_staff' => $role->is_service_staff,
                        'actions' => $actions
                    ];
                });

            return $this->sendResponse($roles, 'Roles retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Get data needed for creating a new role.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function create(Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('roles.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $selling_price_groups = SellingPriceGroup::where('business_id', $business_id)
                ->active()
                ->get()
                ->map(function ($spg) {
                    return [
                        'id' => $spg->id,
                        'name' => $spg->name,
                        'permission_key' => 'selling_price_group.' . $spg->id
                    ];
                });

            $module_permissions = $this->moduleUtil->getModuleData('user_permissions');

            $data = [
                'selling_price_groups' => $selling_price_groups,
                'module_permissions' => $module_permissions
            ];

            return $this->sendResponse($data, 'Role creation data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function store(Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('roles.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            // Validation
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'permissions' => 'array',
                'is_service_staff' => 'boolean',
                'spg_permissions' => 'array'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $role_name = $request->input('name');
            $permissions = $request->input('permissions', []);
            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            // Check if role already exists
            $count = Role::where('name', $role_name . '#' . $business_id)
                ->where('business_id', $business_id)
                ->count();

            if ($count > 0) {
                return $this->sendError('Role already exists.', [], 409);
            }

            $is_service_staff = $request->input('is_service_staff', 0) == 1 ? 1 : 0;

            $role = Role::create([
                'name' => $role_name . '#' . $business_id,
                'business_id' => $business_id,
                'is_service_staff' => $is_service_staff,
                'guard_name' => 'web'
            ]);

            // Include selling price group permissions
            $spg_permissions = $request->input('spg_permissions', []);
            if (!empty($spg_permissions)) {
                $permissions = array_merge($permissions, $spg_permissions);
            }

            $this->createPermissionIfNotExists($permissions);

            if (!empty($permissions)) {
                $role->syncPermissions($permissions);
            }

            $role_data = [
                'id' => $role->id,
                'name' => str_replace('#' . $business_id, '', $role->name),
                'is_service_staff' => $role->is_service_staff,
                'permissions' => $role->permissions->pluck('name')->toArray()
            ];

            return $this->sendResponse($role_data, 'Role created successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @param Request $request
     * @return JsonResponse
     */
    public function show($id, Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('roles.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $role = Role::where('business_id', $business_id)
                ->with(['permissions'])
                ->find($id);

            if (!$role) {
                return $this->sendError('Role not found.', [], 404);
            }

            $role_permissions = $role->permissions->pluck('name')->toArray();

            $role_data = [
                'id' => $role->id,
                'name' => str_replace('#' . $business_id, '', $role->name),
                'is_default' => $role->is_default,
                'is_service_staff' => $role->is_service_staff,
                'permissions' => $role_permissions
            ];

            return $this->sendResponse($role_data, 'Role retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Get data needed for editing a role.
     *
     * @param int $id
     * @param Request $request
     * @return JsonResponse
     */
    public function edit($id, Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('roles.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $role = Role::where('business_id', $business_id)
                ->with(['permissions'])
                ->find($id);

            if (!$role) {
                return $this->sendError('Role not found.', [], 404);
            }

            $role_permissions = $role->permissions->pluck('name')->toArray();

            $selling_price_groups = SellingPriceGroup::where('business_id', $business_id)
                ->active()
                ->get()
                ->map(function ($spg) {
                    return [
                        'id' => $spg->id,
                        'name' => $spg->name,
                        'permission_key' => 'selling_price_group.' . $spg->id
                    ];
                });

            $module_permissions = $this->moduleUtil->getModuleData('user_permissions');

            $data = [
                'role' => [
                    'id' => $role->id,
                    'name' => str_replace('#' . $business_id, '', $role->name),
                    'is_default' => $role->is_default,
                    'is_service_staff' => $role->is_service_staff,
                    'permissions' => $role_permissions
                ],
                'selling_price_groups' => $selling_price_groups,
                'module_permissions' => $module_permissions
            ];

            return $this->sendResponse($data, 'Role edit data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function update(Request $request, $id): JsonResponse
    {
        try {
            if (!auth()->user()->can('roles.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            // Validation
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'permissions' => 'array',
                'is_service_staff' => 'boolean',
                'spg_permissions' => 'array'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $role_name = $request->input('name');
            $permissions = $request->input('permissions', []);
            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            // Check if role name already exists (excluding current role)
            $count = Role::where('name', $role_name . '#' . $business_id)
                ->where('id', '!=', $id)
                ->where('business_id', $business_id)
                ->count();

            if ($count > 0) {
                return $this->sendError('Role already exists.', [], 409);
            }

            $role = Role::findOrFail($id);

            // Check if role can be updated
            if ($role->is_default && $role->name != 'Cashier#' . $business_id) {
                return $this->sendError('Cannot update default role.', [], 403);
            }

            if ($role->name == 'Cashier#' . $business_id) {
                $role->is_default = 0;
            }

            $is_service_staff = $request->input('is_service_staff', 0) == 1 ? 1 : 0;
            $role->is_service_staff = $is_service_staff;
            $role->name = $role_name . '#' . $business_id;
            $role->save();

            // Include selling price group permissions
            $spg_permissions = $request->input('spg_permissions', []);
            if (!empty($spg_permissions)) {
                $permissions = array_merge($permissions, $spg_permissions);
            }

            $this->createPermissionIfNotExists($permissions);

            if (!empty($permissions)) {
                $role->syncPermissions($permissions);
            }

            $role_data = [
                'id' => $role->id,
                'name' => str_replace('#' . $business_id, '', $role->name),
                'is_service_staff' => $role->is_service_staff,
                'permissions' => $role->permissions->pluck('name')->toArray()
            ];

            return $this->sendResponse($role_data, 'Role updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @param Request $request
     * @return JsonResponse
     */
    public function destroy($id, Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('roles.delete')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $role = Role::where('business_id', $business_id)->find($id);

            if (!$role) {
                return $this->sendError('Role not found.', [], 404);
            }

            // Check if role can be deleted
            if ($role->is_default && $role->name != 'Cashier#' . $business_id) {
                return $this->sendError('Cannot delete default role.', [], 403);
            }

            $role->delete();

            return $this->sendResponse([], 'Role deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Creates new permission if doesn't exist
     *
     * @param array $permissions
     * @return void
     */
    private function createPermissionIfNotExists($permissions)
    {
        if (empty($permissions)) {
            return;
        }

        $existing_permissions = Permission::whereIn('name', $permissions)
            ->pluck('name')
            ->toArray();

        $non_existing_permissions = array_diff($permissions, $existing_permissions);

        if (!empty($non_existing_permissions)) {
            foreach ($non_existing_permissions as $new_permission) {
                Permission::create([
                    'name' => $new_permission,
                    'guard_name' => 'web'
                ]);
            }
        }
    }

    /**
     * Get all available permissions
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getPermissions(Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('roles.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $selling_price_groups = SellingPriceGroup::where('business_id', $business_id)
                ->active()
                ->get()
                ->map(function ($spg) {
                    return [
                        'id' => $spg->id,
                        'name' => $spg->name,
                        'permission_key' => 'selling_price_group.' . $spg->id
                    ];
                });

            $module_permissions = $this->moduleUtil->getModuleData('user_permissions');

            $data = [
                'selling_price_groups' => $selling_price_groups,
                'module_permissions' => $module_permissions
            ];

            return $this->sendResponse($data, 'Permissions retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }
}
