<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\BusinessLocation;
use App\Models\Category;
use App\Models\Brands;
use App\Models\Unit;
use App\Models\Contact;
use App\Models\CustomerGroup;
use App\Models\SellingPriceGroup;
use App\Models\TaxRate;
use App\Models\Transaction;
use App\Models\Product;
use App\Models\PurchaseLine;
use App\Models\TransactionSellLine;
use App\Models\ResTable;
use App\Models\ExpenseCategory;
use App\Models\CashRegister;
use App\Models\Variation;
use App\Models\VariationLocationDetails;
use App\Models\TransactionSellLinesPurchaseLines;
use App\Models\User;
use App\Utils\TransactionUtil;
use App\Utils\ProductUtil;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;
use App\Charts\CommonChart;
use App\Models\TransactionPayment;
use App\Models\TransactionSellLinesPurchaseLine;
use Carbon\Carbon;

/**
 * @group Reports
 *
 * APIs for managing reports.
 */
class ReportController extends BaseController
{
    /**
     * All Utils instance.
     *
     */
    protected $transactionUtil;
    protected $productUtil;
    protected $moduleUtil;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(TransactionUtil $transactionUtil, ProductUtil $productUtil, ModuleUtil $moduleUtil)
    {
        $this->transactionUtil = $transactionUtil;
        $this->productUtil = $productUtil;
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Get Profit/Loss Report
     *
     * Returns the profit/loss details for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id The ID of the location. Example: 2
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     // Profit/loss details
     *   },
     *   "message": "Profit/Loss details retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving profit/loss details."
     * }
     */
    public function getProfitLoss(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            $location_id = $request->input('location_id');

            $data = $this->transactionUtil->getProfitLossDetails($business_id, $location_id, $start_date, $end_date);

            return $this->sendResponse($data, 'Profit/Loss details retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving profit/loss details.', [], 400);
        }
    }

    /**
     * Get Purchase/Sell Report
     *
     * Returns the purchase and sell totals for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     * @queryParam location_id The ID of the location. Example: 2
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "purchase": {
     *       // Purchase details
     *     },
     *     "sell": {
     *       // Sell details
     *     },
     *     "total_purchase_return": 100,
     *     "total_sell_return": 50,
     *     "difference": {
     *       // Difference details
     *     }
     *   },
     *   "message": "Purchase/Sell details retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving purchase/sell details."
     * }
     */
    public function getPurchaseSell(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            $location_id = $request->input('location_id');

            $purchase_details = $this->transactionUtil->getPurchaseTotals($business_id, $start_date, $end_date, $location_id);

            $sell_details = $this->transactionUtil->getSellTotals(
                $business_id,
                $start_date,
                $end_date,
                $location_id
            );

            $transaction_types = [
                'purchase_return',
                'sell_return'
            ];

            $transaction_totals = $this->transactionUtil->getTransactionTotals(
                $business_id,
                $transaction_types,
                $start_date,
                $end_date,
                $location_id
            );

            $total_purchase_return_inc_tax = $transaction_totals['total_purchase_return_inc_tax'];
            $total_sell_return_inc_tax = $transaction_totals['total_sell_return_inc_tax'];

            $difference = [
                'total' => $sell_details['total_sell_inc_tax'] + $total_sell_return_inc_tax - $purchase_details['total_purchase_inc_tax'] - $total_purchase_return_inc_tax,
                'due' => $sell_details['invoice_due'] - $purchase_details['purchase_due']
            ];

            $data = [
                'purchase' => $purchase_details,
                'sell' => $sell_details,
                'total_purchase_return' => $total_purchase_return_inc_tax,
                'total_sell_return' => $total_sell_return_inc_tax,
                'difference' => $difference
            ];

            return $this->sendResponse($data, 'Purchase/Sell details retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving purchase/sell details.', [], 400);
        }
    }

    /**
     * Get Customer/Supplier Report
     *
     * Returns the customer and supplier details for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam customer_group_id The ID of the customer group. Example: 2
     * @queryParam contact_type The type of contact (customer, supplier, both). Example: customer
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of customer/supplier details
     *   ],
     *   "message": "Customer/Supplier details retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving customer/supplier details."
     * }
     */
    public function getCustomerSuppliers(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $customer_group_id = $request->input('customer_group_id');
            $contact_type = $request->input('contact_type');

            $contacts = Contact::where('contacts.business_id', $business_id)
                ->join('transactions AS t', 'contacts.id', '=', 't.contact_id')
                ->active()
                ->groupBy('contacts.id')
                ->select(
                    DB::raw("SUM(IF(t.type = 'purchase', final_total, 0)) as total_purchase"),
                    DB::raw("SUM(IF(t.type = 'purchase_return', final_total, 0)) as total_purchase_return"),
                    DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', final_total, 0)) as total_invoice"),
                    DB::raw("SUM(IF(t.type = 'purchase', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as purchase_paid"),
                    DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', (SELECT SUM(IF(is_return = 1,-1*amount,amount)) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as invoice_received"),
                    DB::raw("SUM(IF(t.type = 'sell_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as sell_return_paid"),
                    DB::raw("SUM(IF(t.type = 'purchase_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as purchase_return_received"),
                    DB::raw("SUM(IF(t.type = 'sell_return', final_total, 0)) as total_sell_return"),
                    DB::raw("SUM(IF(t.type = 'opening_balance', final_total, 0)) as opening_balance"),
                    DB::raw("SUM(IF(t.type = 'opening_balance', (SELECT SUM(IF(is_return = 1,-1*amount,amount)) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as opening_balance_paid"),
                    'contacts.supplier_business_name',
                    'contacts.name',
                    'contacts.id',
                    'contacts.type as contact_type'
                );
            $permitted_locations = auth()->user()->permitted_locations();

            if ($permitted_locations != 'all') {
                $contacts->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($customer_group_id)) {
                $contacts->where('contacts.customer_group_id', $customer_group_id);
            }

            if (!empty($contact_type)) {
                $contacts->whereIn('contacts.type', [$contact_type, 'both']);
            }

            $contacts = $contacts->get();

            return $this->sendResponse($contacts, 'Customer/Supplier details retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving customer/supplier details.', [], 400);
        }
    }

    /**
     * Get Stock Report
     *
     * Returns the stock report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id The ID of the location. Example: 2
     * @queryParam category_id The ID of the category. Example: 3
     * @queryParam sub_category_id The ID of the sub-category. Example: 4
     * @queryParam brand_id The ID of the brand. Example: 5
     * @queryParam unit_id The ID of the unit. Example: 6
     * @queryParam tax_id The ID of the tax. Example: 7
     * @queryParam type The type of product. Example: variable
     * @queryParam only_mfg_products Filter only manufacturing products. Example: 1
     * @queryParam active_state Filter by active state. Example: active
     * @queryParam not_for_selling Filter products not for selling. Example: 1
     * @queryParam repair_model_id The ID of the repair model. Example: 8
     * @queryParam product_id The ID of the product. Example: 9
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of stock report details
     *   ],
     *   "message": "Stock report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving stock report."
     * }
     */
    public function getStockReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $filters = $request->only([
                'location_id',
                'category_id',
                'sub_category_id',
                'brand_id',
                'unit_id',
                'tax_id',
                'type',
                'only_mfg_products',
                'active_state',
                'not_for_selling',
                'repair_model_id',
                'product_id',
                'active_state'
            ]);

            $filters['not_for_selling'] = isset($filters['not_for_selling']) && $filters['not_for_selling'] == 'true' ? 1 : 0;

            $filters['show_manufacturing_data'] = 0; // Assuming this is not needed for API

            $products = $this->productUtil->getProductStockDetails($business_id, $filters, 'datatables');

            return $this->sendResponse($products, 'Stock report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving stock report.', [], 400);
        }
    }

    /**
     * Get Stock Details
     *
     * Returns the stock details for a product.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam product_id required The ID of the product. Example: 9
     * @queryParam location_id The ID of the location. Example: 2
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of stock details
     *   ],
     *   "message": "Stock details retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving stock details."
     * }
     */
    public function getStockDetails(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $product_id = $request->input('product_id');
            $location_id = $request->input('location_id');

            $query = Product::leftjoin('units as u', 'products.unit_id', '=', 'u.id')
                ->join('variations as v', 'products.id', '=', 'v.product_id')
                ->join('product_variations as pv', 'pv.id', '=', 'v.product_variation_id')
                ->leftjoin('variation_location_details as vld', 'v.id', '=', 'vld.variation_id')
                ->where('products.business_id', $business_id)
                ->where('products.id', $product_id)
                ->whereNull('v.deleted_at');

            $permitted_locations = auth()->user()->permitted_locations();
            $location_filter = '';
            if ($permitted_locations != 'all') {
                $query->whereIn('vld.location_id', $permitted_locations);
                $locations_imploded = implode(', ', $permitted_locations);
                $location_filter .= "AND transactions.location_id IN ($locations_imploded) ";
            }

            if (!empty($location_id)) {
                $query->where('vld.location_id', $location_id);
                $location_filter .= "AND transactions.location_id=$location_id";
            }

            $product_details =  $query->select(
                'products.name as product',
                'u.short_name as unit',
                'pv.name as product_variation',
                'v.name as variation',
                'v.sub_sku as sub_sku',
                'v.sell_price_inc_tax',
                DB::raw("SUM(vld.qty_available) as stock"),
                DB::raw("(SELECT SUM(IF(transactions.type='sell', TSL.quantity - TSL.quantity_returned, -1* TPL.quantity) ) FROM transactions 
                        LEFT JOIN transaction_sell_lines AS TSL ON transactions.id=TSL.transaction_id

                        LEFT JOIN purchase_lines AS TPL ON transactions.id=TPL.transaction_id

                        WHERE transactions.status='final' AND transactions.type='sell' $location_filter 
                        AND (TSL.variation_id=v.id OR TPL.variation_id=v.id)) as total_sold"),
                DB::raw("(SELECT SUM(IF(transactions.type='sell_transfer', TSL.quantity, 0) ) FROM transactions 
                        LEFT JOIN transaction_sell_lines AS TSL ON transactions.id=TSL.transaction_id
                        WHERE transactions.status='final' AND transactions.type='sell_transfer' $location_filter 
                        AND (TSL.variation_id=v.id)) as total_transfered"),
                DB::raw("(SELECT SUM(IF(transactions.type='stock_adjustment', SAL.quantity, 0) ) FROM transactions 
                        LEFT JOIN stock_adjustment_lines AS SAL ON transactions.id=SAL.transaction_id
                        WHERE transactions.status='received' AND transactions.type='stock_adjustment' $location_filter 
                        AND (SAL.variation_id=v.id)) as total_adjusted")
            )
                ->groupBy('v.id')
                ->get();

            return $this->sendResponse($product_details, 'Stock details retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving stock details.', [], 400);
        }
    }

    /**
     * Get Tax Details
     *
     * Returns the tax details for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam type The type of transaction (sell, purchase, expense). Example: sell
     * @queryParam location_id The ID of the location. Example: 2
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of tax details
     *   ],
     *   "message": "Tax details retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving tax details."
     * }
     */
    public function getTaxDetails(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $taxes = TaxRate::forBusiness($business_id);
            $type = $request->input('type');

            $sells = Transaction::leftJoin('tax_rates as tr', 'transactions.tax_id', '=', 'tr.id')
                ->leftJoin('contacts as c', 'transactions.contact_id', '=', 'c.id')
                ->where('transactions.business_id', $business_id)
                ->select(
                    'c.name as contact_name',
                    'c.tax_number',
                    'transactions.ref_no',
                    'transactions.invoice_no',
                    'transactions.transaction_date',
                    'transactions.total_before_tax',
                    'transactions.tax_id',
                    'transactions.tax_amount',
                    'transactions.id',
                    'transactions.type',
                    'transactions.discount_type',
                    'transactions.discount_amount'
                );
            if ($type == 'sell') {
                $sells->where('transactions.type', 'sell')
                    ->where('transactions.status', 'final')
                    ->where(function ($query) {
                        $query->whereHas('sell_lines', function ($q) {
                            $q->whereNotNull('transaction_sell_lines.tax_id');
                        })->orWhereNotNull('transactions.tax_id');
                    })
                    ->with(['sell_lines' => function ($q) {
                        $q->whereNotNull('transaction_sell_lines.tax_id');
                    }]);
            }
            if ($type == 'purchase') {
                $sells->where('transactions.type', 'purchase')
                    ->where('transactions.status', 'received')
                    ->where(function ($query) {
                        $query->whereHas('purchase_lines', function ($q) {
                            $q->whereNotNull('purchase_lines.tax_id');
                        })->orWhereNotNull('transactions.tax_id');
                    })
                    ->with(['purchase_lines' => function ($q) {
                        $q->whereNotNull('purchase_lines.tax_id');
                    }]);
            }

            if ($type == 'expense') {
                $sells->where('transactions.type', 'expense')
                    ->whereNotNull('transactions.tax_id');
            }

            if (request()->has('location_id')) {
                $location_id = request()->get('location_id');
                if (!empty($location_id)) {
                    $sells->where('transactions.location_id', $location_id);
                }
            }
            if (!empty(request()->start_date) && !empty(request()->end_date)) {
                $start = request()->start_date;
                $end =  request()->end_date;
                $sells->whereDate('transactions.transaction_date', '>=', $start)
                    ->whereDate('transactions.transaction_date', '<=', $end);
            }
            $sells = $sells->get();
            $data = [];
            foreach ($sells as $row) {
                $item = [
                    'contact_name' => $row->contact_name,
                    'tax_number' => $row->tax_number,
                    'ref_no' => $row->ref_no,
                    'invoice_no' => $row->invoice_no,
                    'transaction_date' => $row->transaction_date,
                    'total_before_tax' => $row->total_before_tax,
                    'tax_id' => $row->tax_id,
                    'tax_amount' => $row->tax_amount,
                    'id' => $row->id,
                    'type' => $row->type,
                    'discount_type' => $row->discount_type,
                    'discount_amount' => $row->discount_amount,
                ];
                foreach ($taxes as $tax) {
                    $tax_amount = 0;
                    if ($type == 'sell') {
                        foreach ($row->sell_lines as $sell_line) {
                            if ($sell_line->tax_id == $tax['id']) {
                                $tax_amount += ($sell_line->item_tax * ($sell_line->quantity - $sell_line->quantity_returned));
                            }
                        }
                    } elseif ($type == 'purchase') {
                        foreach ($row->purchase_lines as $purchase_line) {
                            if ($purchase_line->tax_id == $tax['id']) {
                                $tax_amount += ($purchase_line->item_tax * ($purchase_line->quantity - $purchase_line->quantity_returned));
                            }
                        }
                    }
                    if ($row->tax_id == $tax['id']) {
                        $tax_amount += $row->tax_amount;
                    }
                    $item['tax_' . $tax['id']] = $tax_amount;
                }
                $data[] = $item;
            }

            return $this->sendResponse($data, 'Tax details retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving tax details.', [], 400);
        }
    }

    /**
     * Get Tax Report
     *
     * Returns the tax report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     * @queryParam location_id The ID of the location. Example: 2
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "input_tax": {
     *       // Input tax details
     *     },
     *     "output_tax": {
     *       // Output tax details
     *     },
     *     "expense_tax": {
     *       // Expense tax details
     *     },
     *     "tax_diff": 100
     *   },
     *   "message": "Tax report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving tax report."
     * }
     */
    public function getTaxReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            $location_id = $request->input('location_id');

            $input_tax_details = $this->transactionUtil->getInputTax($business_id, $start_date, $end_date, $location_id);
            $output_tax_details = $this->transactionUtil->getOutputTax($business_id, $start_date, $end_date, $location_id);
            $expense_tax_details = $this->transactionUtil->getExpenseTax($business_id, $start_date, $end_date, $location_id);

            $module_output_taxes = $this->moduleUtil->getModuleData('getModuleOutputTax', ['start_date' => $start_date, 'end_date' => $end_date]);

            $total_module_output_tax = 0;
            foreach ($module_output_taxes as $key => $module_output_tax) {
                $total_module_output_tax += $module_output_tax;
            }

            $total_output_tax = $output_tax_details['total_tax'] + $total_module_output_tax;

            $tax_diff = $total_output_tax - $input_tax_details['total_tax'] - $expense_tax_details['total_tax'];

            $data = [
                'input_tax' => $input_tax_details,
                'output_tax' => $output_tax_details,
                'expense_tax' => $expense_tax_details,
                'tax_diff' => $tax_diff
            ];

            return $this->sendResponse($data, 'Tax report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving tax report.', [], 400);
        }
    }

    /**
     * Get Trending Products
     *
     * Returns the trending products for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam category The category of the product. Example: food
     * @queryParam sub_category The sub-category of the product. Example: drinks
     * @queryParam brand The brand of the product. Example: coca-cola
     * @queryParam unit The unit of the product. Example: kg
     * @queryParam limit The number of products to return. Example: 10
     * @queryParam location_id The ID of the location. Example: 2
     * @queryParam product_type The type of product. Example: variable
     * @queryParam date_range The date range for the report. Example: 2023-01-01~2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of trending products
     *   ],
     *   "message": "Trending products retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving trending products."
     * }
     */
    public function getTrendingProducts(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $filters = $request->only(['category', 'sub_category', 'brand', 'unit', 'limit', 'location_id', 'product_type']);

            $date_range = $request->input('date_range');

            if (!empty($date_range)) {
                $date_range_array = explode('~', $date_range);
                $filters['start_date'] = $this->transactionUtil->uf_date(trim($date_range_array[0]));
                $filters['end_date'] = $this->transactionUtil->uf_date(trim($date_range_array[1]));
            }

            $products = $this->productUtil->getTrendingProducts($business_id, $filters);

            return $this->sendResponse($products, 'Trending products retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving trending products.', [], 400);
        }
    }

    /**
     * Get Expense Report
     *
     * Returns the expense report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam category The category of the expense. Example: food
     * @queryParam location_id The ID of the location. Example: 2
     * @queryParam date_range The date range for the report. Example: 2023-01-01~2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of expenses
     *   ],
     *   "message": "Expense report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving expense report."
     * }
     */
    public function getExpenseReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $filters = $request->only(['category', 'location_id']);

            $date_range = $request->input('date_range');

            if (!empty($date_range)) {
                $date_range_array = explode('~', $date_range);
                $filters['start_date'] = $this->transactionUtil->uf_date(trim($date_range_array[0]));
                $filters['end_date'] = $this->transactionUtil->uf_date(trim($date_range_array[1]));
            } else {
                $filters['start_date'] = Carbon::now()->startOfMonth()->format('Y-m-d');
                $filters['end_date'] = Carbon::now()->endOfMonth()->format('Y-m-d');
            }

            $expenses = $this->transactionUtil->getExpenseReport($business_id, $filters);

            return $this->sendResponse($expenses, 'Expense report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving expense report.', [], 400);
        }
    }

    /**
     * Get Stock Adjustment Report
     *
     * Returns the stock adjustment report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     * @queryParam location_id The ID of the location. Example: 2
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "total_amount": 100,
     *     "total_recovered": 50,
     *     "total_normal": 75,
     *     "total_abnormal": 25
     *   },
     *   "message": "Stock adjustment report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving stock adjustment report."
     * }
     */
    public function getStockAdjustmentReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            $location_id = $request->input('location_id');

            $query =  Transaction::where('business_id', $business_id)
                ->where('type', 'stock_adjustment');

            //Check for permitted locations of a user
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('location_id', $permitted_locations);
            }

            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween(DB::raw('date(transaction_date)'), [$start_date, $end_date]);
            }
            if (!empty($location_id)) {
                $query->where('location_id', $location_id);
            }

            $stock_adjustment_details = $query->select(
                DB::raw("SUM(final_total) as total_amount"),
                DB::raw("SUM(total_amount_recovered) as total_recovered"),
                DB::raw("SUM(IF(adjustment_type = 'normal', final_total, 0)) as total_normal"),
                DB::raw("SUM(IF(adjustment_type = 'abnormal', final_total, 0)) as total_abnormal")
            )->first();

            return $this->sendResponse($stock_adjustment_details, 'Stock adjustment report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving stock adjustment report.', [], 400);
        }
    }

    /**
     * Get Register Report
     *
     * Returns the register report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam user_id The ID of the user. Example: 2
     * @queryParam status The status of the register (open, close). Example: close
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of register details
     *   ],
     *   "message": "Register report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving register report."
     * }
     */
    public function getRegisterReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $registers = CashRegister::join(
                'users as u',
                'u.id',
                '=',
                'cash_registers.user_id'
            )
                ->leftJoin(
                    'business_locations as bl',
                    'bl.id',
                    '=',
                    'cash_registers.location_id'
                )
                ->where('cash_registers.business_id', $business_id)
                ->select(
                    'cash_registers.*',
                    DB::raw(
                        "CONCAT(COALESCE(surname, ''), ' ', COALESCE(first_name, ''), ' ', COALESCE(last_name, ''), '<br>', COALESCE(u.email, '')) as user_name"
                    ),
                    'bl.name as location_name'
                );

            if (!empty($request->input('user_id'))) {
                $registers->where('cash_registers.user_id', $request->input('user_id'));
            }
            if (!empty($request->input('status'))) {
                $registers->where('cash_registers.status', $request->input('status'));
            }
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');

            if (!empty($start_date) && !empty($end_date)) {
                $registers->whereDate('cash_registers.created_at', '>=', $start_date)
                    ->whereDate('cash_registers.created_at', '<=', $end_date);
            }

            $registers = $registers->get();

            return $this->sendResponse($registers, 'Register report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving register report.', [], 400);
        }
    }

    /**
     * Get Sales Representative Report
     *
     * Returns the sales representative report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "users": [
     *       // Array of users
     *     ],
     *     "business_locations": [
     *       // Array of business locations
     *     ]
     *   },
     *   "message": "Sales representative report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving sales representative report."
     * }
     */
    public function getSalesRepresentativeReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $users = User::allUsersDropdown($business_id, false);
            $business_locations = BusinessLocation::forDropdown($business_id, true);

            $data = [
                'users' => $users,
                'business_locations' => $business_locations
            ];

            return $this->sendResponse($data, 'Sales representative report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving sales representative report.', [], 400);
        }
    }

    /**
     * Get Sales Representative Total Expense
     *
     * Returns the total expense for a sales representative.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam expense_for The ID of the user for whom to retrieve expenses. Example: 2
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": 100,
     *   "message": "Sales representative total expense retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving sales representative total expense."
     * }
     */
    public function getSalesRepresentativeTotalExpense(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $filters = $request->only(['expense_for', 'location_id', 'start_date', 'end_date']);

            $total_expense = $this->transactionUtil->getExpenseReport($business_id, $filters, 'total');

            return $this->sendResponse($total_expense, 'Sales representative total expense retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving sales representative total expense.', [], 400);
        }
    }

    /**
     * Get Sales Representative Total Sell
     *
     * Returns the total sell for a sales representative.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam created_by The ID of the user who created the sell transaction. Example: 2
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "total_sell_exc_tax": 1000,
     *     "total_sell_return_exc_tax": 100,
     *     "total_sell": 900
     *   },
     *   "message": "Sales representative total sell retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving sales representative total sell."
     * }
     */
    public function getSalesRepresentativeTotalSell(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');

            $location_id = $request->input('location_id');
            $created_by = $request->input('created_by');

            $sell_details = $this->transactionUtil->getSellTotals($business_id, $start_date, $end_date, $location_id, $created_by);

            //Get Sell Return details
            $transaction_types = [
                'sell_return'
            ];
            $sell_return_details = $this->transactionUtil->getTransactionTotals(
                $business_id,
                $transaction_types,
                $start_date,
                $end_date,
                $location_id,
                $created_by
            );

            $total_sell_return = !empty($sell_return_details['total_sell_return_exc_tax']) ? $sell_return_details['total_sell_return_exc_tax'] : 0;
            $total_sell = $sell_details['total_sell_exc_tax'] - $total_sell_return;

            $data = [
                'total_sell_exc_tax' => $sell_details['total_sell_exc_tax'],
                'total_sell_return_exc_tax' => $total_sell_return,
                'total_sell' => $total_sell
            ];

            return $this->sendResponse($data, 'Sales representative total sell retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving sales representative total sell.', [], 400);
        }
    }

    /**
     * Get Sales Representative Total Commission
     *
     * Returns the total commission for a sales representative.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam commission_agent The ID of the commission agent (user). Example: 2
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "total_sales_with_commission": 1000,
     *     "total_commission": 100,
     *     "commission_percentage": 10
     *   },
     *   "message": "Sales representative total commission retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving sales representative total commission."
     * }
     */
    public function getSalesRepresentativeTotalCommission(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');

            $location_id = $request->input('location_id');
            $commission_agent = $request->input('commission_agent');

            $sell_details = $this->transactionUtil->getTotalSellCommission($business_id, $start_date, $end_date, $location_id, $commission_agent);

            //Get Commision
            $commission_percentage = User::find($commission_agent)->cmmsn_percent;
            $total_commission = $commission_percentage * $sell_details['total_sales_with_commission'] / 100;

            $data = [
                'total_sales_with_commission' => $sell_details['total_sales_with_commission'],
                'total_commission' => $total_commission,
                'commission_percentage' => $commission_percentage
            ];

            return $this->sendResponse($data, 'Sales representative total commission retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving sales representative total commission.', [], 400);
        }
    }

    /**
     * Get Stock Expiry Report
     *
     * Returns the stock expiry report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id The ID of the location. Example: 2
     * @queryParam category_id The ID of the category. Example: 3
     * @queryParam sub_category_id The ID of the sub-category. Example: 4
     * @queryParam brand_id The ID of the brand. Example: 5
     * @queryParam unit_id The ID of the unit. Example: 6
     * @queryParam exp_date_filter Filter by expiry date. Example: 2024-12-31
     * @queryParam only_mfg_products Filter only manufacturing products. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of stock expiry details
     *   ],
     *   "message": "Stock expiry report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving stock expiry report."
     * }
     */
    public function getStockExpiryReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $query = PurchaseLine::leftjoin(
                'transactions as t',
                'purchase_lines.transaction_id',
                '=',
                't.id'
            )
                ->leftjoin(
                    'products as p',
                    'purchase_lines.product_id',
                    '=',
                    'p.id'
                )
                ->leftjoin(
                    'variations as v',
                    'purchase_lines.variation_id',
                    '=',
                    'v.id'
                )
                ->leftjoin(
                    'product_variations as pv',
                    'v.product_variation_id',
                    '=',
                    'pv.id'
                )
                ->leftjoin('business_locations as l', 't.location_id', '=', 'l.id')
                ->leftjoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->where('p.enable_stock', 1);

            $permitted_locations = auth()->user()->permitted_locations();

            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->input('location_id'))) {
                $location_id = $request->input('location_id');
                $query->where('t.location_id', $location_id)
                    ->join('product_locations as pl', 'pl.product_id', '=', 'p.id')
                    ->where(function ($q) use ($location_id) {
                        $q->where('pl.location_id', $location_id);
                    });
            }

            if (!empty($request->input('category_id'))) {
                $query->where('p.category_id', $request->input('category_id'));
            }
            if (!empty($request->input('sub_category_id'))) {
                $query->where('p.sub_category_id', $request->input('sub_category_id'));
            }
            if (!empty($request->input('brand_id'))) {
                $query->where('p.brand_id', $request->input('brand_id'));
            }
            if (!empty($request->input('unit_id'))) {
                $query->where('p.unit_id', $request->input('unit_id'));
            }
            if (!empty($request->input('exp_date_filter'))) {
                $query->whereDate('exp_date', '<=', $request->input('exp_date_filter'));
            }

            $only_mfg_products = request()->get('only_mfg_products', 0);
            if (!empty($only_mfg_products)) {
                $query->where('t.type', 'production_purchase');
            }

            $report = $query->select(
                'p.name as product',
                'p.sku',
                'p.type as product_type',
                'v.name as variation',
                'pv.name as product_variation',
                'l.name as location',
                'mfg_date',
                'exp_date',
                'u.short_name as unit',
                DB::raw("SUM(COALESCE(quantity, 0) - COALESCE(quantity_sold, 0) - COALESCE(quantity_adjusted, 0) - COALESCE(quantity_returned, 0)) as stock_left"),
                't.ref_no',
                't.id as transaction_id',
                'purchase_lines.id as purchase_line_id',
                'purchase_lines.lot_number'
            )
                ->having('stock_left', '>', 0)
                ->groupBy('purchase_lines.exp_date')
                ->groupBy('purchase_lines.lot_number')
                ->get();

            return $this->sendResponse($report, 'Stock expiry report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving stock expiry report.', [], 400);
        }
    }

    /**
     * Get Stock Expiry Report Edit Modal
     *
     * Returns the data for the stock expiry report edit modal.
     *
     * @urlParam purchase_line_id required The ID of the purchase line. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     // Purchase line details
     *   },
     *   "message": "Stock expiry report edit modal data retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving stock expiry report edit modal data."
     * }
     */
    public function getStockExpiryReportEditModal(Request $request, $purchase_line_id)
    {
        try {
            $business_id = $request->input('business_id');

            $purchase_line = PurchaseLine::join(
                'transactions as t',
                'purchase_lines.transaction_id',
                '=',
                't.id'
            )
                ->join(
                    'products as p',
                    'purchase_lines.product_id',
                    '=',
                    'p.id'
                )
                ->where('purchase_lines.id', $purchase_line_id)
                ->where('t.business_id', $business_id)
                ->select(['purchase_lines.*', 'p.name', 't.ref_no'])
                ->first();

            if (!empty($purchase_line)) {
                if (!empty($purchase_line->exp_date)) {
                    $purchase_line->exp_date = date('m/d/Y', strtotime($purchase_line->exp_date));
                }
            }

            return $this->sendResponse($purchase_line, 'Stock expiry report edit modal data retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving stock expiry report edit modal data.', [], 400);
        }
    }

    /**
     * Update Stock Expiry Report
     *
     * Updates the stock expiry report.
     *
     * @bodyParam purchase_line_id required The ID of the purchase line. Example: 1
     * @bodyParam exp_date The new expiry date. Example: 2024-12-31
     *
     * @response {
     *   "success": true,
     *   "message": "Stock expiry report updated successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error updating stock expiry report."
     * }
     */
    public function updateStockExpiryReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $input = $request->only(['purchase_line_id', 'exp_date']);

            $purchase_line = PurchaseLine::join(
                'transactions as t',
                'purchase_lines.transaction_id',
                '=',
                't.id'
            )
                ->join(
                    'products as p',
                    'purchase_lines.product_id',
                    '=',
                    'p.id'
                )
                ->where('purchase_lines.id', $input['purchase_line_id'])
                ->where('t.business_id', $business_id)
                ->select(['purchase_lines.*', 'p.name', 't.ref_no'])
                ->first();

            if (!empty($purchase_line) && !empty($input['exp_date'])) {
                $purchase_line->exp_date = $this->productUtil->uf_date($input['exp_date']);
                $purchase_line->save();
            }

            return $this->sendResponse(null, 'Stock expiry report updated successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error updating stock expiry report.', [], 400);
        }
    }

    /**
     * Get Customer Group Report
     *
     * Returns the customer group report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam customer_group_id The ID of the customer group. Example: 2
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of customer group details
     *   ],
     *   "message": "Customer group report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving customer group report."
     * }
     */
    public function getCustomerGroup(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $query = Transaction::leftjoin('customer_groups AS CG', 'transactions.customer_group_id', '=', 'CG.id')
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'sell')
                ->where('transactions.status', 'final')
                ->groupBy('transactions.customer_group_id')
                ->select(DB::raw("SUM(final_total) as total_sell"), 'CG.name');

            $group_id = $request->input('customer_group_id', null);
            if (!empty($group_id)) {
                $query->where('transactions.customer_group_id', $group_id);
            }

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('transactions.location_id', $permitted_locations);
            }

            $location_id = $request->input('location_id', null);
            if (!empty($location_id)) {
                $query->where('transactions.location_id', $location_id);
            }

            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');

            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween(DB::raw('date(transaction_date)'), [$start_date, $end_date]);
            }

            $customer_groups = $query->get();

            return $this->sendResponse($customer_groups, 'Customer group report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving customer group report.', [], 400);
        }
    }

    /**
     * Get Product Purchase Report
     *
     * Returns the product purchase report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam variation_id The ID of the variation. Example: 2
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam supplier_id The ID of the supplier. Example: 4
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of product purchase details
     *   ],
     *   "message": "Product purchase report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving product purchase report."
     * }
     */
    public function getproductPurchaseReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $variation_id = $request->input('variation_id', null);
            $query = PurchaseLine::join(
                'transactions as t',
                'purchase_lines.transaction_id',
                '=',
                't.id'
            )
                ->join(
                    'variations as v',
                    'purchase_lines.variation_id',
                    '=',
                    'v.id'
                )
                ->join('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->join('contacts as c', 't.contact_id', '=', 'c.id')
                ->join('products as p', 'pv.product_id', '=', 'p.id')
                ->leftjoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase')
                ->select(
                    'p.name as product_name',
                    'p.type as product_type',
                    'pv.name as product_variation',
                    'v.name as variation_name',
                    'v.sub_sku',
                    'c.name as supplier',
                    't.id as transaction_id',
                    't.ref_no',
                    't.transaction_date as transaction_date',
                    'purchase_lines.purchase_price_inc_tax as unit_purchase_price',
                    DB::raw('(purchase_lines.quantity - purchase_lines.quantity_returned) as purchase_qty'),
                    'purchase_lines.quantity_adjusted',
                    'u.short_name as unit',
                    DB::raw('((purchase_lines.quantity - purchase_lines.quantity_returned - purchase_lines.quantity_adjusted) * purchase_lines.purchase_price_inc_tax) as subtotal')
                )
                ->groupBy('purchase_lines.id');
            if (!empty($variation_id)) {
                $query->where('purchase_lines.variation_id', $variation_id);
            }
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween(DB::raw('date(transaction_date)'), [$start_date, $end_date]);
            }

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            $location_id = $request->input('location_id', null);
            if (!empty($location_id)) {
                $query->where('t.location_id', $location_id);
            }

            $supplier_id = $request->input('supplier_id', null);
            if (!empty($supplier_id)) {
                $query->where('t.contact_id', $supplier_id);
            }

            $product_purchases = $query->get();

            return $this->sendResponse($product_purchases, 'Product purchase report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving product purchase report.', [], 400);
        }
    }

    /**
     * Get Product Sell Report
     *
     * Returns the product sell report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam variation_id The ID of the variation. Example: 2
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam customer_id The ID of the customer. Example: 4
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of product sell details
     *   ],
     *   "message": "Product sell report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving product sell report."
     * }
     */
    public function getproductSellReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $variation_id = $request->input('variation_id', null);
            $query = TransactionSellLine::join(
                'transactions as t',
                'transaction_sell_lines.transaction_id',
                '=',
                't.id'
            )
                ->join(
                    'variations as v',
                    'transaction_sell_lines.variation_id',
                    '=',
                    'v.id'
                )
                ->join('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->join('contacts as c', 't.contact_id', '=', 'c.id')
                ->join('products as p', 'pv.product_id', '=', 'p.id')
                ->leftjoin('tax_rates', 'transaction_sell_lines.tax_id', '=', 'tax_rates.id')
                ->leftjoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->select(
                    'p.name as product_name',
                    'p.type as product_type',
                    'pv.name as product_variation',
                    'v.name as variation_name',
                    'v.sub_sku',
                    'c.name as customer',
                    'c.contact_id',
                    't.id as transaction_id',
                    't.invoice_no',
                    't.transaction_date as transaction_date',
                    'transaction_sell_lines.unit_price_before_discount as unit_price',
                    'transaction_sell_lines.unit_price_inc_tax as unit_sale_price',
                    DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sell_qty'),
                    'transaction_sell_lines.line_discount_type as discount_type',
                    'transaction_sell_lines.line_discount_amount as discount_amount',
                    'transaction_sell_lines.item_tax',
                    'tax_rates.name as tax',
                    'u.short_name as unit',
                    DB::raw('((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as subtotal')
                )
                ->groupBy('transaction_sell_lines.id')
                ->get();

            if (!empty($variation_id)) {
                $query->where('transaction_sell_lines.variation_id', $variation_id);
            }
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween(DB::raw('date(transaction_date)'), [$start_date, $end_date]);
            }

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            $location_id = $request->input('location_id', null);
            if (!empty($location_id)) {
                $query->where('t.location_id', $location_id);
            }

            $customer_id = $request->input('customer_id', null);
            if (!empty($customer_id)) {
                $query->where('t.contact_id', $customer_id);
            }

            return $this->sendResponse($query, 'Product sell report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving product sell report.', [], 400);
        }
    }

    /**
     * Get Product Sell Report With Purchase
     *
     * Returns the product sell report with purchase details for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam variation_id The ID of the variation. Example: 2
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam customer_id The ID of the customer. Example: 4
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of product sell details with purchase information
     *   ],
     *   "message": "Product sell report with purchase details retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving product sell report with purchase details."
     * }
     */
    public function getproductSellReportWithPurchase(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $variation_id = $request->input('variation_id', null);
            $query = TransactionSellLine::join(
                'transactions as t',
                'transaction_sell_lines.transaction_id',
                '=',
                't.id'
            )
                ->join(
                    'transaction_sell_lines_purchase_lines as tspl',
                    'transaction_sell_lines.id',
                    '=',
                    'tspl.sell_line_id'
                )
                ->join(
                    'purchase_lines as pl',
                    'tspl.purchase_line_id',
                    '=',
                    'pl.id'
                )
                ->join(
                    'transactions as purchase',
                    'pl.transaction_id',
                    '=',
                    'purchase.id'
                )
                ->leftjoin('contacts as supplier', 'purchase.contact_id', '=', 'supplier.id')
                ->join(
                    'variations as v',
                    'transaction_sell_lines.variation_id',
                    '=',
                    'v.id'
                )
                ->join('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftjoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->join('products as p', 'pv.product_id', '=', 'p.id')
                ->leftjoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->select(
                    'p.name as product_name',
                    'p.type as product_type',
                    'pv.name as product_variation',
                    'v.name as variation_name',
                    'v.sub_sku',
                    'c.name as customer',
                    't.id as transaction_id',
                    't.invoice_no',
                    't.transaction_date as transaction_date',
                    'tspl.quantity as purchase_quantity',
                    'u.short_name as unit',
                    'supplier.name as supplier_name',
                    'purchase.ref_no as ref_no',
                    'purchase.type as purchase_type',
                    'pl.lot_number'
                )
                ->get();

            if (!empty($variation_id)) {
                $query->where('transaction_sell_lines.variation_id', $variation_id);
            }
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween(DB::raw('date(t.transaction_date)'), [$start_date, $end_date]);
            }

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            $location_id = $request->input('location_id', null);
            if (!empty($location_id)) {
                $query->where('t.location_id', $location_id);
            }

            $customer_id = $request->input('customer_id', null);
            if (!empty($customer_id)) {
                $query->where('t.contact_id', $customer_id);
            }

            return $this->sendResponse($query, 'Product sell report with purchase details retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving product sell report with purchase details.', [], 400);
        }
    }

    /**
     * Get Lot Report
     *
     * Returns the lot report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id The ID of the location. Example: 2
     * @queryParam category_id The ID of the category. Example: 3
     * @queryParam sub_category_id The ID of the sub-category. Example: 4
     * @queryParam brand_id The ID of the brand. Example: 5
     * @queryParam unit_id The ID of the unit. Example: 6
     * @queryParam only_mfg_products Filter only manufacturing products. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of lot details
     *   ],
     *   "message": "Lot report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving lot report."
     * }
     */
    public function getLotReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $query = Product::where('products.business_id', $business_id)
                ->leftjoin('units', 'products.unit_id', '=', 'units.id')
                ->join('variations as v', 'products.id', '=', 'v.product_id')
                ->join('purchase_lines as pl', 'v.id', '=', 'pl.variation_id')
                ->leftjoin(
                    'transaction_sell_lines_purchase_lines as tspl',
                    'pl.id',
                    '=',
                    'tspl.purchase_line_id'
                )
                ->join('transactions as t', 'pl.transaction_id', '=', 't.id');

            $permitted_locations = auth()->user()->permitted_locations();
            $location_filter = 'WHERE ';

            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);

                $locations_imploded = implode(', ', $permitted_locations);
                $location_filter = " LEFT JOIN transactions as t2 on pls.transaction_id=t2.id WHERE t2.location_id IN ($locations_imploded) AND ";
            }

            if (!empty($request->input('location_id'))) {
                $location_id = $request->input('location_id');
                $query->where('t.location_id', $location_id)
                    ->ForLocation($location_id);

                $location_filter = "LEFT JOIN transactions as t2 on pls.transaction_id=t2.id WHERE t2.location_id=$location_id AND ";
            }

            if (!empty($request->input('category_id'))) {
                $query->where('products.category_id', $request->input('category_id'));
            }

            if (!empty($request->input('sub_category_id'))) {
                $query->where('products.sub_category_id', $request->input('sub_category_id'));
            }

            if (!empty($request->input('brand_id'))) {
                $query->where('products.brand_id', $request->input('brand_id'));
            }

            if (!empty($request->input('unit_id'))) {
                $query->where('products.unit_id', $request->input('unit_id'));
            }

            $only_mfg_products = request()->get('only_mfg_products', 0);
            if (!empty($only_mfg_products)) {
                $query->where('t.type', 'production_purchase');
            }

            $products = $query->select(
                'products.name as product',
                'v.name as variation_name',
                'sub_sku',
                'pl.lot_number',
                'pl.exp_date as exp_date',
                DB::raw("( COALESCE((SELECT SUM(quantity - quantity_returned) from purchase_lines as pls $location_filter variation_id = v.id AND lot_number = pl.lot_number), 0) - 
                    SUM(COALESCE((tspl.quantity - tspl.qty_returned), 0))) as stock"),
                DB::raw("COALESCE(SUM(IF(tspl.sell_line_id IS NULL, 0, (tspl.quantity - tspl.qty_returned)) ), 0) as total_sold"),
                DB::raw("COALESCE(SUM(IF(tspl.stock_adjustment_line_id IS NULL, 0, tspl.quantity ) ), 0) as total_adjusted"),
                'products.type',
                'units.short_name as unit'
            )
                ->whereNotNull('pl.lot_number')
                ->groupBy('v.id')
                ->groupBy('pl.lot_number')
                ->get();

            return $this->sendResponse($products, 'Lot report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving lot report.', [], 400);
        }
    }

    /**
     * Get Purchase Payment Report
     *
     * Returns the purchase payment report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam supplier_id The ID of the supplier. Example: 2
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of purchase payment details
     *   ],
     *   "message": "Purchase payment report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving purchase payment report."
     * }
     */
    public function purchasePaymentReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $supplier_id = $request->input('supplier_id', null);
            $contact_filter1 = !empty($supplier_id) ? "AND t.contact_id=$supplier_id" : '';
            $contact_filter2 = !empty($supplier_id) ? "AND transactions.contact_id=$supplier_id" : '';

            $location_id = $request->input('location_id', null);

            $parent_payment_query_part = empty($location_id) ? "AND transaction_payments.parent_id IS NULL" : "";

            $query = TransactionPayment::leftjoin('transactions as t', function ($join) use ($business_id) {
                $join->on('transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->whereIn('t.type', ['purchase', 'opening_balance']);
            })
                ->where('transaction_payments.business_id', $business_id)
                ->where(function ($q) use ($business_id, $contact_filter1, $contact_filter2, $parent_payment_query_part) {
                    $q->whereRaw("(transaction_payments.transaction_id IS NOT NULL AND t.type IN ('purchase', 'opening_balance')  $parent_payment_query_part $contact_filter1)")
                        ->orWhereRaw("EXISTS(SELECT * FROM transaction_payments as tp JOIN transactions ON tp.transaction_id = transactions.id WHERE transactions.type IN ('purchase', 'opening_balance') AND transactions.business_id = $business_id AND tp.parent_id=transaction_payments.id $contact_filter2)");
                })

                ->select(
                    DB::raw("IF(transaction_payments.transaction_id IS NULL, 
                                (SELECT c.name FROM transactions as ts
                                JOIN contacts as c ON ts.contact_id=c.id 
                                WHERE ts.id=(
                                        SELECT tps.transaction_id FROM transaction_payments as tps
                                        WHERE tps.parent_id=transaction_payments.id LIMIT 1
                                    )
                                ),
                                (SELECT c.name FROM transactions as ts JOIN
                                    contacts as c ON ts.contact_id=c.id
                                    WHERE ts.id=t.id 
                                )
                            ) as supplier"),
                    'transaction_payments.amount',
                    'method',
                    'paid_on',
                    'transaction_payments.payment_ref_no',
                    'transaction_payments.document',
                    't.ref_no',
                    't.id as transaction_id',
                    'cheque_number',
                    'card_transaction_number',
                    'bank_account_number',
                    'transaction_no',
                    'transaction_payments.id as DT_RowId'
                )
                ->groupBy('transaction_payments.id')
                ->get();

            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween(DB::raw('date(paid_on)'), [$start_date, $end_date]);
            }

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                $query->where('t.location_id', $location_id);
            }

            return $this->sendResponse($query, 'Purchase payment report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving purchase payment report.', [], 400);
        }
    }

    /**
     * Get Sell Payment Report
     *
     * Returns the sell payment report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam supplier_id The ID of the customer. Example: 2
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     * @queryParam customer_group_id The ID of the customer group. Example: 4
     * @queryParam payment_types The payment type. Example: cash
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of sell payment details
     *   ],
     *   "message": "Sell payment report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving sell payment report."
     * }
     */
    public function sellPaymentReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $customer_id = $request->input('supplier_id', null);
            $contact_filter1 = !empty($customer_id) ? "AND t.contact_id=$customer_id" : '';
            $contact_filter2 = !empty($customer_id) ? "AND transactions.contact_id=$customer_id" : '';

            $location_id = $request->input('location_id', null);
            $parent_payment_query_part = empty($location_id) ? "AND transaction_payments.parent_id IS NULL" : "";

            $query = TransactionPayment::leftjoin('transactions as t', function ($join) use ($business_id) {
                $join->on('transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->whereIn('t.type', ['sell', 'opening_balance']);
            })
                ->leftjoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftjoin('customer_groups AS CG', 'c.customer_group_id', '=', 'CG.id')
                ->where('transaction_payments.business_id', $business_id)
                ->where(function ($q) use ($business_id, $contact_filter1, $contact_filter2, $parent_payment_query_part) {
                    $q->whereRaw("(transaction_payments.transaction_id IS NOT NULL AND t.type IN ('sell', 'opening_balance') $parent_payment_query_part $contact_filter1)")
                        ->orWhereRaw("EXISTS(SELECT * FROM transaction_payments as tp JOIN transactions ON tp.transaction_id = transactions.id WHERE transactions.type IN ('sell', 'opening_balance') AND transactions.business_id = $business_id AND tp.parent_id=transaction_payments.id $contact_filter2)");
                })
                ->select(
                    DB::raw("IF(transaction_payments.transaction_id IS NULL, 
                                (SELECT c.name FROM transactions as ts
                                JOIN contacts as c ON ts.contact_id=c.id 
                                WHERE ts.id=(
                                        SELECT tps.transaction_id FROM transaction_payments as tps
                                        WHERE tps.parent_id=transaction_payments.id LIMIT 1
                                    )
                                ),
                                (SELECT c.name FROM transactions as ts JOIN
                                    contacts as c ON ts.contact_id=c.id
                                    WHERE ts.id=t.id 
                                )
                            ) as customer"),
                    'transaction_payments.amount',
                    'transaction_payments.is_return',
                    'method',
                    'paid_on',
                    'transaction_payments.payment_ref_no',
                    'transaction_payments.document',
                    'transaction_payments.transaction_no',
                    't.invoice_no',
                    't.id as transaction_id',
                    'cheque_number',
                    'card_transaction_number',
                    'bank_account_number',
                    'transaction_payments.id as DT_RowId',
                    'CG.name as customer_group'
                )
                ->groupBy('transaction_payments.id')
                ->get();

            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween(DB::raw('date(paid_on)'), [$start_date, $end_date]);
            }

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->input('customer_group_id'))) {
                $query->where('CG.id', $request->input('customer_group_id'));
            }

            if (!empty($location_id)) {
                $query->where('t.location_id', $location_id);
            }

            if (!empty($request->input('payment_types'))) {
                $query->where('transaction_payments.method', $request->input('payment_types'));
            }

            return $this->sendResponse($query, 'Sell payment report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving sell payment report.', [], 400);
        }
    }

    /**
     * Get Table Report
     *
     * Returns the table report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id The ID of the location. Example: 2
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of table details
     *   ],
     *   "message": "Table report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving table report."
     * }
     */
    public function getTableReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $query = ResTable::leftjoin('transactions AS T', 'T.res_table_id', '=', 'res_tables.id')
                ->where('T.business_id', $business_id)
                ->where('T.type', 'sell')
                ->where('T.status', 'final')
                ->groupBy('res_tables.id')
                ->select(DB::raw("SUM(final_total) as total_sell"), 'res_tables.name as table')
                ->get();

            $location_id = $request->input('location_id', null);
            if (!empty($location_id)) {
                $query->where('T.location_id', $location_id);
            }

            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');

            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween(DB::raw('date(transaction_date)'), [$start_date, $end_date]);
            }

            return $this->sendResponse($query, 'Table report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving table report.', [], 400);
        }
    }

    /**
     * Get Service Staff Report
     *
     * Returns the service staff report for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "business_locations": [
     *       // Array of business locations
     *     ],
     *     "waiters": [
     *       // Array of waiters
     *     ]
     *   },
     *   "message": "Service staff report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving service staff report."
     * }
     */
    public function getServiceStaffReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $business_locations = BusinessLocation::forDropdown($business_id, true);

            $waiters = $this->transactionUtil->serviceStaffDropdown($business_id);

            $data = [
                'business_locations' => $business_locations,
                'waiters' => $waiters
            ];

            return $this->sendResponse($data, 'Service staff report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving service staff report.', [], 400);
        }
    }

    /**
     * Get Product Sell Grouped Report
     *
     * Returns the product sell report grouped by date for a business.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id The ID of the location. Example: 2
     * @queryParam variation_id The ID of the variation. Example: 3
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     * @queryParam customer_id The ID of the customer. Example: 4
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of product sell details grouped by date
     *   ],
     *   "message": "Product sell grouped report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving product sell grouped report."
     * }
     */
    public function getproductSellGroupedReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $location_id = $request->input('location_id', null);

            $vld_str = '';
            if (!empty($location_id)) {
                $vld_str = "AND vld.location_id=$location_id";
            }

            $variation_id = $request->input('variation_id', null);
            $query = TransactionSellLine::join(
                'transactions as t',
                'transaction_sell_lines.transaction_id',
                '=',
                't.id'
            )
                ->join(
                    'variations as v',
                    'transaction_sell_lines.variation_id',
                    '=',
                    'v.id'
                )
                ->join('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->join('products as p', 'pv.product_id', '=', 'p.id')
                ->leftjoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->select(
                    'p.name as product_name',
                    'p.enable_stock',
                    'p.type as product_type',
                    'pv.name as product_variation',
                    'v.name as variation_name',
                    'v.sub_sku',
                    't.id as transaction_id',
                    't.transaction_date as transaction_date',
                    DB::raw('DATE_FORMAT(t.transaction_date, "%Y-%m-%d") as formated_date'),
                    DB::raw("(SELECT SUM(vld.qty_available) FROM variation_location_details as vld WHERE vld.variation_id=v.id $vld_str) as current_stock"),
                    DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as total_qty_sold'),
                    'u.short_name as unit',
                    DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as subtotal')
                )
                ->groupBy('v.id')
                ->groupBy('formated_date')
                ->get();

            if (!empty($variation_id)) {
                $query->where('transaction_sell_lines.variation_id', $variation_id);
            }
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween(DB::raw('date(transaction_date)'), [$start_date, $end_date]);
            }

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                $query->where('t.location_id', $location_id);
            }

            $customer_id = $request->input('customer_id', null);
            if (!empty($customer_id)) {
                $query->where('t.contact_id', $customer_id);
            }

            return $this->sendResponse($query, 'Product sell grouped report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving product sell grouped report.', [], 400);
        }
    }

    /**
     * Get Service Staff Line Orders
     *
     * Retrieves line orders/sales for service staff.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam service_staff_id The ID of the service staff. Example: 2
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of service staff line orders
     *   ],
     *   "message": "Service staff line orders retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving service staff line orders."
     * }
     */
    public function serviceStaffLineOrders(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $query = TransactionSellLine::leftJoin('transactions as t', 't.id', '=', 'transaction_sell_lines.transaction_id')
                ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
                ->leftJoin('products as p', 'v.product_id', '=', 'p.id')
                ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('users as ss', 'ss.id', '=', 'transaction_sell_lines.res_service_staff_id')
                ->leftjoin(
                    'business_locations AS bl',
                    't.location_id',
                    '=',
                    'bl.id'
                )
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->whereNotNull('transaction_sell_lines.res_service_staff_id');


            if (!empty($request->input('service_staff_id'))) {
                $query->where('transaction_sell_lines.res_service_staff_id', $request->input('service_staff_id'));
            }

            if ($request->has('location_id')) {
                $location_id = $request->input('location_id');
                if (!empty($location_id)) {
                    $query->where('t.location_id', $location_id);
                }
            }

            if (!empty($request->input('start_date')) && !empty($request->input('end_date'))) {
                $start = $request->input('start_date');
                $end =  $request->input('end_date');
                $query->whereDate('t.transaction_date', '>=', $start)
                    ->whereDate('t.transaction_date', '<=', $end);
            }

            $query->select(
                'p.name as product_name',
                'p.type as product_type',
                'v.name as variation_name',
                'pv.name as product_variation_name',
                'u.short_name as unit',
                't.id as transaction_id',
                'bl.name as business_location',
                't.transaction_date',
                't.invoice_no',
                'transaction_sell_lines.quantity',
                'transaction_sell_lines.unit_price_before_discount',
                'transaction_sell_lines.line_discount_type',
                'transaction_sell_lines.line_discount_amount',
                'transaction_sell_lines.item_tax',
                'transaction_sell_lines.unit_price_inc_tax',
                DB::raw('CONCAT(COALESCE(ss.first_name, ""), COALESCE(ss.last_name, "")) as service_staff')
            );

            $service_staff_line_orders = $query->get();

            return $this->sendResponse($service_staff_line_orders, 'Service staff line orders retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving service staff line orders.', [], 400);
        }
    }

    /**
     * Get Profit
     *
     * Lists profit by product, category, brand, location, invoice and date.
     *
     * @urlParam by string The grouping criteria (product, category, brand, location, invoice, date). Example: product
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of profit details
     *   ],
     *   "message": "Profit retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving profit."
     * }
     */
    public function getProfit(Request $request, $by = null)
    {
        try {
            $business_id = $request->input('business_id');

            $query = TransactionSellLine
                ::join('transactions as sale', 'transaction_sell_lines.transaction_id', '=', 'sale.id')
                ->leftjoin('transaction_sell_lines_purchase_lines as TSPL', 'transaction_sell_lines.id', '=', 'TSPL.sell_line_id')
                ->leftjoin(
                    'purchase_lines as PL',
                    'TSPL.purchase_line_id',
                    '=',
                    'PL.id'
                )
                ->join('products as P', 'transaction_sell_lines.product_id', '=', 'P.id')
                ->where('sale.business_id', $business_id)
                ->where('transaction_sell_lines.children_type', '!=', 'combo');

            $query->select(DB::raw('SUM(IF (TSPL.id IS NULL AND P.type="combo", ( 
            SELECT Sum((tspl2.quantity - tspl2.qty_returned) * (tsl.unit_price_inc_tax - pl2.purchase_price_inc_tax)) AS total
            FROM transaction_sell_lines AS tsl
                JOIN transaction_sell_lines_purchase_lines AS tspl2
            ON tsl.id=tspl2.sell_line_id 
            JOIN purchase_lines AS pl2 
            ON tspl2.purchase_line_id = pl2.id 
            WHERE tsl.parent_sell_line_id = transaction_sell_lines.id), IF(P.enable_stock=0,(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax,   
            (TSPL.quantity - TSPL.qty_returned) * (transaction_sell_lines.unit_price_inc_tax - PL.purchase_price_inc_tax)) )) AS gross_profit'));

            if (!empty($request->input('start_date')) && !empty($request->input('end_date'))) {
                $start = $request->input('start_date');
                $end =  $request->input('end_date');
                $query->whereDate('sale.transaction_date', '>=', $start)
                    ->whereDate('sale.transaction_date', '<=', $end);
            }

            if ($by == 'product') {
                $query->join('variations as V', 'transaction_sell_lines.variation_id', '=', 'V.id')
                    ->leftJoin('product_variations as PV', 'PV.id', '=', 'V.product_variation_id')
                    ->addSelect(DB::raw("IF(P.type='variable', CONCAT(P.name, ' - ', PV.name, ' - ', V.name, ' (', V.sub_sku, ')'), CONCAT(P.name, ' (', P.sku, ')')) as product"))
                    ->groupBy('V.id');
            }

            if ($by == 'category') {
                $query->join('variations as V', 'transaction_sell_lines.variation_id', '=', 'V.id')
                    ->leftJoin('categories as C', 'C.id', '=', 'P.category_id')
                    ->addSelect("C.name as category")
                    ->groupBy('C.id');
            }

            if ($by == 'brand') {
                $query->join('variations as V', 'transaction_sell_lines.variation_id', '=', 'V.id')
                    ->leftJoin('brands as B', 'B.id', '=', 'P.brand_id')
                    ->addSelect("B.name as brand")
                    ->groupBy('B.id');
            }

            if ($by == 'location') {
                $query->join('business_locations as L', 'sale.location_id', '=', 'L.id')
                    ->addSelect("L.name as location")
                    ->groupBy('L.id');
            }

            if ($by == 'invoice') {
                $query->addSelect('sale.invoice_no', 'sale.id as transaction_id')
                    ->groupBy('sale.invoice_no');
            }

            if ($by == 'date') {
                $query->addSelect("sale.transaction_date")
                    ->groupBy(DB::raw('DATE(sale.transaction_date)'));
            }

            if ($by == 'day') {
                $results = $query->addSelect(DB::raw("DAYNAME(sale.transaction_date) as day"))
                    ->groupBy(DB::raw('DAYOFWEEK(sale.transaction_date)'))
                    ->get();

                $profits = [];
                foreach ($results as $result) {
                    $profits[strtolower($result->day)] = $result->gross_profit;
                }
                $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];

                return view('report.partials.profit_by_day')->with(compact('profits', 'days'));
            }

            if ($by == 'customer') {
                $query->join('contacts as CU', 'sale.contact_id', '=', 'CU.id')
                    ->addSelect("CU.name as customer")
                    ->groupBy('sale.contact_id');
            }

            $profit = $query->get();

            return $this->sendResponse($profit, 'Profit retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving profit.', [], 400);
        }
    }

    /**
     * Get Items Report
     *
     * Shows items report from sell purchase mapping table.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam purchase_start The start date for the purchase report. Example: 2023-01-01
     * @queryParam purchase_end The end date for the purchase report. Example: 2023-12-31
     * @queryParam sale_start The start date for the sale report. Example: 2023-01-01
     * @queryParam sale_end The end date for the sale report. Example: 2023-12-31
     * @queryParam supplier_id The ID of the supplier. Example: 2
     * @queryParam customer_id The ID of the customer. Example: 3
     * @queryParam location_id The ID of the location. Example: 4
     * @queryParam only_mfg_products Filter only manufacturing products. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of items report details
     *   ],
     *   "message": "Items report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving items report."
     * }
     */
    public function itemsReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $query = TransactionSellLinesPurchaseLine::leftJoin('transaction_sell_lines 
                    as SL', 'SL.id', '=', 'transaction_sell_lines_purchase_lines.sell_line_id')
                ->leftJoin('stock_adjustment_lines 
                    as SAL', 'SAL.id', '=', 'transaction_sell_lines_purchase_lines.stock_adjustment_line_id')
                ->leftJoin('transactions as sale', 'SL.transaction_id', '=', 'sale.id')
                ->leftJoin('transactions as stock_adjustment', 'SAL.transaction_id', '=', 'stock_adjustment.id')
                ->join('purchase_lines as PL', 'PL.id', '=', 'transaction_sell_lines_purchase_lines.purchase_line_id')
                ->join('transactions as purchase', 'PL.transaction_id', '=', 'purchase.id')
                ->join('business_locations as bl', 'purchase.location_id', '=', 'bl.id')
                ->join(
                    'variations as v',
                    'PL.variation_id',
                    '=',
                    'v.id'
                )
                ->join('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->join('products as p', 'PL.product_id', '=', 'p.id')
                ->join('units as u', 'p.unit_id', '=', 'u.id')
                ->leftJoin('contacts as suppliers', 'purchase.contact_id', '=', 'suppliers.id')
                ->leftJoin('contacts as customers', 'sale.contact_id', '=', 'customers.id')
                ->where('purchase.business_id', $business_id)
                ->select(
                    'v.sub_sku as sku',
                    'p.type as product_type',
                    'p.name as product_name',
                    'v.name as variation_name',
                    'pv.name as product_variation',
                    'u.short_name as unit',
                    'purchase.transaction_date as purchase_date',
                    'purchase.ref_no as purchase_ref_no',
                    'purchase.type as purchase_type',
                    'suppliers.name as supplier',
                    'PL.purchase_price_inc_tax as purchase_price',
                    'sale.transaction_date as sell_date',
                    'stock_adjustment.transaction_date as stock_adjustment_date',
                    'sale.invoice_no as sale_invoice_no',
                    'stock_adjustment.ref_no as stock_adjustment_ref_no',
                    'customers.name as customer',
                    'transaction_sell_lines_purchase_lines.quantity as quantity',
                    'SL.unit_price_inc_tax as selling_price',
                    'SAL.unit_price as stock_adjustment_price',
                    'transaction_sell_lines_purchase_lines.stock_adjustment_line_id',
                    'transaction_sell_lines_purchase_lines.sell_line_id',
                    'transaction_sell_lines_purchase_lines.purchase_line_id',
                    'transaction_sell_lines_purchase_lines.qty_returned',
                    'bl.name as location'
                );

            if (!empty($request->input('purchase_start')) && !empty($request->input('purchase_end'))) {
                $start = $request->input('purchase_start');
                $end =  $request->input('purchase_end');
                $query->whereDate('purchase.transaction_date', '>=', $start)
                    ->whereDate('purchase.transaction_date', '<=', $end);
            }
            if (!empty($request->input('sale_start')) && !empty($request->input('sale_end'))) {
                $start = $request->input('sale_start');
                $end =  $request->input('sale_end');
                $query->where(function ($q) use ($start, $end) {
                    $q->where(function ($qr) use ($start, $end) {
                        $qr->whereDate('sale.transaction_date', '>=', $start)
                            ->whereDate('sale.transaction_date', '<=', $end);
                    })->orWhere(function ($qr) use ($start, $end) {
                        $qr->whereDate('stock_adjustment.transaction_date', '>=', $start)
                            ->whereDate('stock_adjustment.transaction_date', '<=', $end);
                    });
                });
            }

            $supplier_id = $request->input('supplier_id', null);
            if (!empty($supplier_id)) {
                $query->where('suppliers.id', $supplier_id);
            }

            $customer_id = $request->input('customer_id', null);
            if (!empty($customer_id)) {
                $query->where('customers.id', $customer_id);
            }

            $location_id = $request->input('location_id', null);
            if (!empty($location_id)) {
                $query->where('purchase.location_id', $location_id);
            }

            $only_mfg_products = $request->get('only_mfg_products', 0);
            if (!empty($only_mfg_products)) {
                $query->where('purchase.type', 'production_purchase');
            }

            $items_report = $query->get();

            return $this->sendResponse($items_report, 'Items report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving items report.', [], 400);
        }
    }

    /**
     * Get Purchase Report
     *
     * Shows purchase report.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam supplier_id The ID of the supplier. Example: 2
     * @queryParam location_id The ID of the location. Example: 3
     * @queryParam payment_status The payment status. Example: paid
     * @queryParam status The status of the purchase. Example: received
     * @queryParam start_date The start date for the report. Example: 2023-01-01
     * @queryParam end_date The end date for the report. Example: 2023-12-31
     *
     * @response {
     *   "success": true,
     *   "data": [
     *     // Array of purchase report details
     *   ],
     *   "message": "Purchase report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving purchase report."
     * }
     */
    public function purchaseReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');

            $purchases = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
                ->join(
                    'business_locations AS BS',
                    'transactions.location_id',
                    '=',
                    'BS.id'
                )
                ->leftJoin(
                    'transaction_payments AS TP',
                    'transactions.id',
                    '=',
                    'TP.transaction_id'
                )
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'purchase')
                ->with(['payment_lines'])
                ->select(
                    'transactions.id',
                    'transactions.ref_no',
                    'contacts.name',
                    'contacts.contact_id',
                    'final_total',
                    'total_before_tax',
                    'discount_amount',
                    'discount_type',
                    'tax_amount',
                    DB::raw('DATE_FORMAT(transaction_date, "%Y/%m") as purchase_year_month'),
                    DB::raw('DATE_FORMAT(transaction_date, "%d") as purchase_day')
                )
                ->groupBy('transactions.id');

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $purchases->whereIn('transactions.location_id', $permitted_locations);
            }

            if (!empty($request->input('supplier_id'))) {
                $purchases->where('contacts.id', $request->input('supplier_id'));
            }
            if (!empty($request->input('location_id'))) {
                $purchases->where('transactions.location_id', $request->input('location_id'));
            }
            if (!empty($request->input('payment_status')) && $request->input('payment_status') != 'overdue') {
                $purchases->where('transactions.payment_status', $request->input('payment_status'));
            } elseif ($request->input('payment_status') == 'overdue') {
                $purchases->whereIn('transactions.payment_status', ['due', 'partial'])
                    ->whereNotNull('transactions.pay_term_number')
                    ->whereNotNull('transactions.pay_term_type')
                    ->whereRaw("IF(transactions.pay_term_type='days', DATE_ADD(transactions.transaction_date, INTERVAL transactions.pay_term_number DAY) < CURDATE(), DATE_ADD(transactions.transaction_date, INTERVAL transactions.pay_term_number MONTH) < CURDATE())");
            }

            if (!empty($request->input('status'))) {
                $purchases->where('transactions.status', $request->input('status'));
            }

            if (!empty($request->input('start_date')) && !empty($request->input('end_date'))) {
                $start = $request->input('start_date');
                $end =  $request->input('end_date');
                $purchases->whereDate('transactions.transaction_date', '>=', $start)
                    ->whereDate('transactions.transaction_date', '<=', $end);
            }

            $purchase_report = $purchases->get();

            return $this->sendResponse($purchase_report, 'Purchase report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving purchase report.', [], 400);
        }
    }

    /**
     * Get Sale Report
     *
     * Shows sale report.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "business_locations": [
     *       // Array of business locations
     *     ],
     *     "customers": [
     *       // Array of customers
     *     ]
     *   },
     *   "message": "Sale report retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving sale report."
     * }
     */
    public function saleReport(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $business_locations = BusinessLocation::forDropdown($business_id, false);
            $customers = Contact::customersDropdown($business_id, false);

            $data = [
                'business_locations' => $business_locations,
                'customers' => $customers
            ];

            return $this->sendResponse($data, 'Sale report retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving sale report.', [], 400);
        }
    }

    /**
     * Get Stock Value
     *
     * Calculates stock values.
     *
     * @queryParam business_id required The ID of the business. Example: 1
     * @queryParam location_id The ID of the location. Example: 2
     * @queryParam category_id The ID of the category. Example: 3
     * @queryParam sub_category_id The ID of the sub-category. Example: 4
     * @queryParam brand_id The ID of the brand. Example: 5
     * @queryParam unit_id The ID of the unit. Example: 6
     *
     * @response {
     *   "success": true,
     *   "data": {
     *     "closing_stock_by_pp": 1000,
     *     "closing_stock_by_sp": 1500,
     *     "potential_profit": 500,
     *     "profit_margin": 33.33
     *   },
     *   "message": "Stock value retrieved successfully."
     * }
     * @response 400 {
     *   "success": false,
     *   "message": "Error retrieving stock value."
     * }
     */
    public function getStockValue(Request $request)
    {
        try {
            $business_id = $request->input('business_id');
            $end_date = Carbon::now()->format('Y-m-d');
            $location_id = $request->input('location_id');
            $filters = $request->only(['category_id', 'sub_category_id', 'brand_id', 'unit_id']);

            $closing_stock_by_pp = $this->transactionUtil->getOpeningClosingStock(
                $business_id,
                $end_date,
                $location_id,
                false,
                false,
                $filters
            );
            $closing_stock_by_sp = $this->transactionUtil->getOpeningClosingStock(
                $business_id,
                $end_date,
                $location_id,
                false,
                true,
                $filters
            );
            $potential_profit = $closing_stock_by_sp - $closing_stock_by_pp;
            $profit_margin = empty($closing_stock_by_sp) ? 0 : ($potential_profit / $closing_stock_by_sp) * 100;

            $data = [
                'closing_stock_by_pp' => $closing_stock_by_pp,
                'closing_stock_by_sp' => $closing_stock_by_sp,
                'potential_profit' => $potential_profit,
                'profit_margin' => $profit_margin
            ];

            return $this->sendResponse($data, 'Stock value retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving stock value.', [], 400);
        }
    }

}
