<?php

namespace App\Http\Controllers;


use App\Models\AccountTransaction;
use App\Http\Controllers\Api\BaseController;
use App\Models\PurchaseLine;
use App\Models\Transaction;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PurchaseReturnApiController extends BaseController
{
    protected $transactionUtil;
    protected $productUtil;

    public function __construct(TransactionUtil $transactionUtil, ProductUtil $productUtil)
    {
        $this->transactionUtil = $transactionUtil;
        $this->productUtil = $productUtil;
    }

    public function index(Request $request)
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $purchases_returns = Transaction::with(['contact', 'location'])
                ->where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->get();

            return $this->sendResponse($purchases_returns, 'Liste des retours d’achats récupérée avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération.', $e->getMessage(), 500);
        }
    }

    public function add($id)
    {
        try {
            if (!auth()->user()->can('purchase.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $purchase = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase')
                ->with(['purchase_lines', 'contact', 'tax', 'return_parent', 'purchase_lines.sub_unit', 'purchase_lines.product', 'purchase_lines.product.unit'])
                ->findOrFail($id);

            foreach ($purchase->purchase_lines as $key => $value) {
                if (!empty($value->sub_unit_id)) {
                    $formated_purchase_line = $this->productUtil->changePurchaseLineUnit($value, $business_id);
                    $purchase->purchase_lines[$key] = $formated_purchase_line;
                }
                $qty_available = $value->quantity - $value->quantity_sold - $value->quantity_adjusted;
                $purchase->purchase_lines[$key]->formatted_qty_available = $this->transactionUtil->num_f($qty_available);
            }

            return $this->sendResponse($purchase, 'Détails de l’achat récupérés avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération.', $e->getMessage(), 500);
        }
    }

    public function store(Request $request)
    {
        try {
            if (!auth()->user()->can('purchase.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $purchase = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase')
                ->with(['purchase_lines', 'purchase_lines.sub_unit'])
                ->findOrFail($request->transaction_id);

            $return_quantities = $request->returns;
            $return_total = 0;

            DB::beginTransaction();

            foreach ($purchase->purchase_lines as $purchase_line) {
                $old_return_qty = $purchase_line->quantity_returned;
                $return_quantity = !empty($return_quantities[$purchase_line->id])
                    ? $this->productUtil->num_uf($return_quantities[$purchase_line->id])
                    : 0;

                if (!empty($purchase_line->sub_unit->base_unit_multiplier)) {
                    $return_quantity *= $purchase_line->sub_unit->base_unit_multiplier;
                }

                $purchase_line->quantity_returned = $return_quantity;
                $purchase_line->save();

                $return_total += $purchase_line->purchase_price_inc_tax * $purchase_line->quantity_returned;

                if ($old_return_qty != $purchase_line->quantity_returned) {
                    $this->productUtil->decreaseProductQuantity(
                        $purchase_line->product_id,
                        $purchase_line->variation_id,
                        $purchase->location_id,
                        $purchase_line->quantity_returned,
                        $old_return_qty
                    );
                }
            }

            $return_total_inc_tax = $return_total + $request->tax_amount;
            $return_transaction_data = [
                'total_before_tax' => $return_total,
                'final_total' => $return_total_inc_tax,
                'tax_amount' => $request->tax_amount,
                'tax_id' => $purchase->tax_id
            ];

            if (empty($request->ref_no)) {
                $ref_count = $this->transactionUtil->setAndGetReferenceCount('purchase_return');
                $return_transaction_data['ref_no'] = $this->transactionUtil->generateReferenceNumber('purchase_return', $ref_count);
            }

            $return_transaction = Transaction::updateOrCreate(
                [
                    'business_id' => $business_id,
                    'type' => 'purchase_return',
                    'return_parent_id' => $purchase->id
                ],
                array_merge($return_transaction_data, [
                    'location_id' => $purchase->location_id,
                    'status' => 'final',
                    'contact_id' => $purchase->contact_id,
                    'transaction_date' => now(),
                    'created_by' => $request->session()->get('user.id')
                ])
            );

            $this->transactionUtil->updatePaymentStatus($return_transaction->id, $return_transaction->final_total);

            DB::commit();
            return $this->sendResponse($return_transaction, 'Retour d’achat enregistré avec succès.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Erreur lors de l’enregistrement.', $e->getMessage(), 500);
        }
    }

    public function show($id)
    {
        try {
            if (!auth()->user()->can('purchase.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $purchase = Transaction::where('business_id', $business_id)
                ->with(['return_parent', 'return_parent.tax', 'purchase_lines', 'contact', 'tax', 'purchase_lines.sub_unit', 'purchase_lines.product', 'purchase_lines.product.unit'])
                ->findOrFail($id);

            return $this->sendResponse($purchase, 'Détails du retour d’achat récupérés avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération.', $e->getMessage(), 500);
        }
    }

    public function destroy($id)
    {
        try {
            if (!auth()->user()->can('purchase.delete')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $purchase_return = Transaction::where('id', $id)
                ->where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->with(['purchase_lines'])
                ->firstOrFail();

            DB::beginTransaction();

            if (empty($purchase_return->return_parent_id)) {
                foreach ($purchase_return->purchase_lines as $purchase_line) {
                    $this->productUtil->updateProductQuantity(
                        $purchase_return->location_id,
                        $purchase_line->product_id,
                        $purchase_line->variation_id,
                        $purchase_line->quantity_returned,
                        0
                    );
                }
                PurchaseLine::where('transaction_id', $purchase_return->id)->delete();
            }

            $purchase_return->delete();
            AccountTransaction::where('transaction_id', $id)->delete();

            DB::commit();
            return $this->sendResponse([], 'Retour d’achat supprimé avec succès.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Erreur lors de la suppression.', $e->getMessage(), 500);
        }
    }
}
