<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\AccountTransaction;
use App\Models\Business;
use App\Models\BusinessLocation;
use App\Models\Contact;
use App\Models\CustomerGroup;
use App\Models\Product;
use App\Models\PurchaseLine;
use App\Models\TaxRate;
use App\Models\Transaction;
use App\Models\User;
use App\Utils\BusinessUtil;
use App\Utils\ModuleUtil;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PurchaseController extends BaseController
{
    /**
     * All Utils instance.
     */
    protected $productUtil;
    protected $transactionUtil;
    protected $businessUtil;
    protected $moduleUtil;

    /**
     * Constructor
     */
    public function __construct(ProductUtil $productUtil, TransactionUtil $transactionUtil, BusinessUtil $businessUtil, ModuleUtil $moduleUtil)
    {
        $this->productUtil = $productUtil;
        $this->transactionUtil = $transactionUtil;
        $this->businessUtil = $businessUtil;
        $this->moduleUtil = $moduleUtil;

        $this->dummyPaymentLine = [
            'method' => 'cash',
            'amount' => 0,
            'note' => '',
            'card_transaction_number' => '',
            'card_number' => '',
            'card_type' => '',
            'card_holder_name' => '',
            'card_month' => '',
            'card_year' => '',
            'card_security' => '',
            'cheque_number' => '',
            'bank_account_number' => '',
            'is_return' => 0,
            'transaction_no' => ''
        ];
    }

    /**
     * Display a listing of purchases.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create') && !auth()->user()->can('view_own_purchase')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? session('user.business_id');
            $purchases = $this->transactionUtil->getListPurchases($business_id);

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $purchases->whereIn('transactions.location_id', $permitted_locations);
            }

            // Apply filters
            if (!empty($request->supplier_id)) {
                $purchases->where('contacts.id', $request->supplier_id);
            }
            if (!empty($request->location_id)) {
                $purchases->where('transactions.location_id', $request->location_id);
            }
            if (!empty($request->payment_status) && $request->payment_status != 'overdue') {
                $purchases->where('transactions.payment_status', $request->payment_status);
            } elseif ($request->payment_status == 'overdue') {
                $purchases->whereIn('transactions.payment_status', ['due', 'partial'])
                    ->whereNotNull('transactions.pay_term_number')
                    ->whereNotNull('transactions.pay_term_type')
                    ->whereRaw("IF(transactions.pay_term_type='days', DATE_ADD(transactions.transaction_date, INTERVAL transactions.pay_term_number DAY) < CURDATE(), DATE_ADD(transactions.transaction_date, INTERVAL transactions.pay_term_number MONTH) < CURDATE())");
            }

            if (!empty($request->status)) {
                $purchases->where('transactions.status', $request->status);
            }

            if (!empty($request->start_date) && !empty($request->end_date)) {
                $purchases->whereDate('transactions.transaction_date', '>=', $request->start_date)
                    ->whereDate('transactions.transaction_date', '<=', $request->end_date);
            }

            if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
                $purchases->where('transactions.created_by', auth()->id());
            }

            $purchases = $purchases->with(['contact', 'location'])->paginate($request->per_page ?? 15);

            $business_locations = BusinessLocation::forDropdown($business_id);
            $suppliers = Contact::suppliersDropdown($business_id, false);
            $orderStatuses = $this->productUtil->orderStatuses();

            $data = [
                'purchases' => $purchases,
                'business_locations' => $business_locations,
                'suppliers' => $suppliers,
                'order_statuses' => $orderStatuses
            ];

            return $this->sendResponse($data, 'Purchases retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Get form data for creating a new purchase.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create(Request $request)
    {
        try {
            if (!auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? session('user.business_id');

            // Check if subscribed
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            }

            $taxes = TaxRate::where('business_id', $business_id)->get();
            $orderStatuses = $this->productUtil->orderStatuses();
            $business_locations = BusinessLocation::forDropdown($business_id, false, true);
            $bl_attributes = $business_locations['attributes'];
            $business_locations = $business_locations['locations'];

            $currency_details = $this->transactionUtil->purchaseCurrencyDetails($business_id);

            $default_purchase_status = null;
            if (session('business.enable_purchase_status') != 1) {
                $default_purchase_status = 'received';
            }

            $types = [];
            if (auth()->user()->can('supplier.create')) {
                $types['supplier'] = __('report.supplier');
            }
            if (auth()->user()->can('customer.create')) {
                $types['customer'] = __('report.customer');
            }
            if (auth()->user()->can('supplier.create') && auth()->user()->can('customer.create')) {
                $types['both'] = __('lang_v1.both_supplier_customer');
            }

            $customer_groups = CustomerGroup::forDropdown($business_id);
            $business_details = $this->businessUtil->getDetails($business_id);
            $shortcuts = json_decode($business_details->keyboard_shortcuts, true);
            $payment_line = $this->dummyPaymentLine;
            $payment_types = $this->productUtil->payment_types(null, true);
            $accounts = $this->moduleUtil->accountsDropdown($business_id, true);

            $data = compact(
                'taxes',
                'orderStatuses',
                'business_locations',
                'currency_details',
                'default_purchase_status',
                'customer_groups',
                'types',
                'shortcuts',
                'payment_line',
                'payment_types',
                'accounts',
                'bl_attributes'
            );

            return $this->sendResponse($data, 'Purchase form data retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Store a newly created purchase.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            if (!auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? session('user.business_id');

            // Check if subscribed
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            }

            // Validation
            $request->validate([
                'status' => 'required',
                'contact_id' => 'required',
                'transaction_date' => 'required',
                'total_before_tax' => 'required',
                'location_id' => 'required',
                'final_total' => 'required',
                'document' => 'file|max:' . (config('constants.document_size_limit') / 1000)
            ]);

            $transaction_data = $request->only([
                'ref_no',
                'status',
                'contact_id',
                'transaction_date',
                'total_before_tax',
                'location_id',
                'discount_type',
                'discount_amount',
                'tax_id',
                'tax_amount',
                'shipping_details',
                'shipping_charges',
                'final_total',
                'additional_notes',
                'exchange_rate',
                'pay_term_number',
                'pay_term_type'
            ]);

            $exchange_rate = $transaction_data['exchange_rate'];
            $user_id = auth()->id();
            $enable_product_editing = session('business.enable_editing_product_from_purchase');

            // Update business exchange rate
            Business::where('id', $business_id)->update(['p_exchange_rate' => $transaction_data['exchange_rate']]);

            $currency_details = $this->transactionUtil->purchaseCurrencyDetails($business_id);

            // Unformat input values
            $transaction_data['total_before_tax'] = $this->productUtil->num_uf($transaction_data['total_before_tax'], $currency_details) * $exchange_rate;

            if ($transaction_data['discount_type'] == 'fixed') {
                $transaction_data['discount_amount'] = $this->productUtil->num_uf($transaction_data['discount_amount'], $currency_details) * $exchange_rate;
            } elseif ($transaction_data['discount_type'] == 'percentage') {
                $transaction_data['discount_amount'] = $this->productUtil->num_uf($transaction_data['discount_amount'], $currency_details);
            } else {
                $transaction_data['discount_amount'] = 0;
            }

            $transaction_data['tax_amount'] = $this->productUtil->num_uf($transaction_data['tax_amount'], $currency_details) * $exchange_rate;
            $transaction_data['shipping_charges'] = $this->productUtil->num_uf($transaction_data['shipping_charges'], $currency_details) * $exchange_rate;
            $transaction_data['final_total'] = $this->productUtil->num_uf($transaction_data['final_total'], $currency_details) * $exchange_rate;

            $transaction_data['business_id'] = $business_id;
            $transaction_data['created_by'] = $user_id;
            $transaction_data['type'] = 'purchase';
            $transaction_data['payment_status'] = 'due';
            $transaction_data['transaction_date'] = $this->productUtil->uf_date($transaction_data['transaction_date'], true);

            // Upload document
            $transaction_data['document'] = $this->transactionUtil->uploadFile($request, 'document', 'documents');

            DB::beginTransaction();

            // Update reference count
            $ref_count = $this->productUtil->setAndGetReferenceCount($transaction_data['type']);

            // Generate reference number
            if (empty($transaction_data['ref_no'])) {
                $transaction_data['ref_no'] = $this->productUtil->generateReferenceNumber($transaction_data['type'], $ref_count);
            }

            $transaction = Transaction::create($transaction_data);

            $purchases = $request->input('purchases');
            $this->productUtil->createOrUpdatePurchaseLinesStore($transaction, $purchases, $currency_details, $enable_product_editing);

            // Add Purchase payments
            $this->transactionUtil->createOrUpdatePaymentLines($transaction, $request->input('payment'));

            // Update payment status
            $this->transactionUtil->updatePaymentStatus($transaction->id, $transaction->final_total);

            // Adjust stock over selling if found
            $this->productUtil->adjustStockOverSelling($transaction);

            DB::commit();

            return $this->sendResponse($transaction, 'Purchase created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Purchase creation failed.', $e->getMessage(), 500);
        }
    }

    /**
     * Display the specified purchase.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $business_id = session('user.business_id') ?? auth()->user()->business_id;

            $taxes = TaxRate::where('business_id', $business_id)->pluck('name', 'id');

            $purchase = Transaction::where('business_id', $business_id)
                ->where('id', $id)
                ->with([
                    'contact',
                    'purchase_lines',
                    'purchase_lines.product',
                    'purchase_lines.product.unit',
                    'purchase_lines.variations',
                    'purchase_lines.variations.product_variation',
                    'purchase_lines.sub_unit',
                    'location',
                    'payment_lines',
                    'tax'
                ])
                ->firstOrFail();

            foreach ($purchase->purchase_lines as $key => $value) {
                if (!empty($value->sub_unit_id)) {
                    $formated_purchase_line = $this->productUtil->changePurchaseLineUnit($value, $business_id);
                    $purchase->purchase_lines[$key] = $formated_purchase_line;
                }
            }

            $payment_methods = $this->productUtil->payment_types(null, true);

            $purchase_taxes = [];
            if (!empty($purchase->tax)) {
                if ($purchase->tax->is_tax_group) {
                    $purchase_taxes = $this->transactionUtil->sumGroupTaxDetails($this->transactionUtil->groupTaxDetails($purchase->tax, $purchase->tax_amount));
                } else {
                    $purchase_taxes[$purchase->tax->name] = $purchase->tax_amount;
                }
            }

            $data = [
                'purchase' => $purchase,
                'taxes' => $taxes,
                'payment_methods' => $payment_methods,
                'purchase_taxes' => $purchase_taxes
            ];

            return $this->sendResponse($data, 'Purchase retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Purchase not found.', $e->getMessage(), 404);
        }
    }

    /**
     * Get purchase data for editing.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        try {
            if (!auth()->user()->can('purchase.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = session('user.business_id') ?? auth()->user()->business_id;

            // Check if subscribed
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            }

            // Check if the transaction can be edited
            $edit_days = session('business.transaction_edit_days');
            if (!$this->transactionUtil->canBeEdited($id, $edit_days)) {
                return $this->sendError('Transaction edit not allowed after ' . $edit_days . ' days.', [], 400);
            }

            // Check if return exists
            if ($this->transactionUtil->isReturnExist($id)) {
                return $this->sendError('Cannot edit purchase with existing returns.', [], 400);
            }

            $business = Business::find($business_id);
            $currency_details = $this->transactionUtil->purchaseCurrencyDetails($business_id);

            $purchase = Transaction::where('business_id', $business_id)
                ->where('id', $id)
                ->with([
                    'contact',
                    'purchase_lines',
                    'purchase_lines.product',
                    'purchase_lines.product.unit',
                    'purchase_lines.variations',
                    'purchase_lines.variations.product_variation',
                    'location',
                    'purchase_lines.sub_unit'
                ])
                ->first();

            foreach ($purchase->purchase_lines as $key => $value) {
                if (!empty($value->sub_unit_id)) {
                    $formated_purchase_line = $this->productUtil->changePurchaseLineUnit($value, $business_id);
                    $purchase->purchase_lines[$key] = $formated_purchase_line;
                }
            }

            $taxes = TaxRate::where('business_id', $business_id)->get();
            $orderStatuses = $this->productUtil->orderStatuses();
            $business_locations = BusinessLocation::forDropdown($business_id);

            $default_purchase_status = null;
            if (session('business.enable_purchase_status') != 1) {
                $default_purchase_status = 'received';
            }

            $types = [];
            if (auth()->user()->can('supplier.create')) {
                $types['supplier'] = __('report.supplier');
            }
            if (auth()->user()->can('customer.create')) {
                $types['customer'] = __('report.customer');
            }
            if (auth()->user()->can('supplier.create') && auth()->user()->can('customer.create')) {
                $types['both'] = __('lang_v1.both_supplier_customer');
            }

            $customer_groups = CustomerGroup::forDropdown($business_id);
            $business_details = $this->businessUtil->getDetails($business_id);
            $shortcuts = json_decode($business_details->keyboard_shortcuts, true);

            $data = compact(
                'taxes',
                'purchase',
                'orderStatuses',
                'business_locations',
                'business',
                'currency_details',
                'default_purchase_status',
                'customer_groups',
                'types',
                'shortcuts'
            );

            return $this->sendResponse($data, 'Purchase edit data retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Update the specified purchase.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('purchase.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            // Validate document size
            $request->validate([
                'document' => 'file|max:' . (config('constants.document_size_limit') / 1000)
            ]);

            $transaction = Transaction::findOrFail($id);
            $before_status = $transaction->status;
            $business_id = session('user.business_id') ?? auth()->user()->business_id;
            $enable_product_editing = session('business.enable_editing_product_from_purchase');

            $currency_details = $this->transactionUtil->purchaseCurrencyDetails($business_id);

            $update_data = $request->only([
                'ref_no',
                'status',
                'contact_id',
                'transaction_date',
                'total_before_tax',
                'discount_type',
                'discount_amount',
                'tax_id',
                'tax_amount',
                'shipping_details',
                'shipping_charges',
                'final_total',
                'additional_notes',
                'exchange_rate',
                'pay_term_number',
                'pay_term_type'
            ]);

            $exchange_rate = $update_data['exchange_rate'];
            $update_data['transaction_date'] = $this->productUtil->uf_date($update_data['transaction_date'], true);

            // Unformat input values
            $update_data['total_before_tax'] = $this->productUtil->num_uf($update_data['total_before_tax'], $currency_details) * $exchange_rate;

            if ($update_data['discount_type'] == 'fixed') {
                $update_data['discount_amount'] = $this->productUtil->num_uf($update_data['discount_amount'], $currency_details) * $exchange_rate;
            } elseif ($update_data['discount_type'] == 'percentage') {
                $update_data['discount_amount'] = $this->productUtil->num_uf($update_data['discount_amount'], $currency_details);
            } else {
                $update_data['discount_amount'] = 0;
            }

            $update_data['tax_amount'] = $this->productUtil->num_uf($update_data['tax_amount'], $currency_details) * $exchange_rate;
            $update_data['shipping_charges'] = $this->productUtil->num_uf($update_data['shipping_charges'], $currency_details) * $exchange_rate;
            $update_data['final_total'] = $this->productUtil->num_uf($update_data['final_total'], $currency_details) * $exchange_rate;

            // Upload document
            $document_name = $this->transactionUtil->uploadFile($request, 'document', 'documents');
            if (!empty($document_name)) {
                $update_data['document'] = $document_name;
            }

            DB::beginTransaction();

            // Update transaction
            $transaction->update($update_data);

            // Update transaction payment status
            $this->transactionUtil->updatePaymentStatus($transaction->id);

            $purchases = $request->input('purchases');
            $delete_purchase_lines = $this->productUtil->createOrUpdatePurchaseLines($transaction, $purchases, $currency_details, $enable_product_editing, $before_status);

            // Update mapping of purchase & Sell
            $this->transactionUtil->adjustMappingPurchaseSellAfterEditingPurchase($before_status, $transaction, $delete_purchase_lines);

            // Adjust stock over selling if found
            $this->productUtil->adjustStockOverSelling($transaction);
            $this->productUtil->adjustStockOverModifyAdvance($transaction);

            DB::commit();

            return $this->sendResponse($transaction, 'Purchase updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Purchase update failed.', $e->getMessage(), 500);
        }
    }

    /**
     * Remove the specified purchase.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            if (!auth()->user()->can('purchase.delete')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = session('user.business_id') ?? auth()->user()->business_id;

            // Check if return exists
            if ($this->transactionUtil->isReturnExist($id)) {
                return $this->sendError('Cannot delete purchase with existing returns.', [], 400);
            }

            $transaction = Transaction::where('id', $id)
                ->where('business_id', $business_id)
                ->with(['purchase_lines'])
                ->first();

            if (!$transaction) {
                return $this->sendError('Purchase not found.', [], 404);
            }

            // Check if lot numbers are used in sale
            if (session('business.enable_lot_number') == 1 && $this->transactionUtil->isLotUsed($transaction)) {
                return $this->sendError('Lot numbers are used in sale.', [], 400);
            }

            $delete_purchase_lines = $transaction->purchase_lines;

            DB::beginTransaction();

            $transaction_status = $transaction->status;
            if ($transaction_status != 'received') {
                $transaction->delete();
            } else {
                // Delete purchase lines first
                $delete_purchase_line_ids = [];
                foreach ($delete_purchase_lines as $purchase_line) {
                    $delete_purchase_line_ids[] = $purchase_line->id;
                    $this->productUtil->decreaseProductQuantity(
                        $purchase_line->product_id,
                        $purchase_line->variation_id,
                        $transaction->location_id,
                        $purchase_line->quantity
                    );
                }
                PurchaseLine::where('transaction_id', $transaction->id)
                    ->whereIn('id', $delete_purchase_line_ids)
                    ->delete();

                // Update mapping of purchase & Sell
                $this->transactionUtil->adjustMappingPurchaseSellAfterEditingPurchase($transaction_status, $transaction, $delete_purchase_lines);
            }

            // Delete Transaction
            $transaction->delete();

            // Delete account transactions
            AccountTransaction::where('transaction_id', $id)->delete();

            DB::commit();

            return $this->sendResponse([], 'Purchase deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Purchase deletion failed.', $e->getMessage(), 500);
        }
    }

    /**
     * Get suppliers list.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getSuppliers(Request $request)
    {
        try {
            $term = $request->q;
            if (empty($term)) {
                return $this->sendResponse([], 'No search term provided.');
            }

            $business_id = session('user.business_id') ?? auth()->user()->business_id;
            $user_id = auth()->id();

            $query = Contact::where('business_id', $business_id)->active();

            $selected_contacts = User::isSelectedContacts($user_id);
            if ($selected_contacts) {
                $query->join('user_contact_access AS uca', 'contacts.id', 'uca.contact_id')
                    ->where('uca.user_id', $user_id);
            }

            $suppliers = $query->where(function ($query) use ($term) {
                $query->where('name', 'like', '%' . $term . '%')
                    ->orWhere('supplier_business_name', 'like', '%' . $term . '%')
                    ->orWhere('contacts.contact_id', 'like', '%' . $term . '%');
            })
                ->select(
                    'contacts.id',
                    'name as text',
                    'supplier_business_name as business_name',
                    'contact_id',
                    'contacts.pay_term_type',
                    'contacts.pay_term_number',
                    'contacts.balance'
                )
                ->onlySuppliers()
                ->get();

            return $this->sendResponse($suppliers, 'Suppliers retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Failed to retrieve suppliers.', $e->getMessage(), 500);
        }
    }

    

    /**
     * Get products list
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProducts(Request $request)
    {
        try {
            $term = $request->term;
            $check_enable_stock = $request->get('check_enable_stock', true);
            $business_id = $request->user()->business_id;

            if (empty($term)) {
                return $this->sendResponse([], 'Products retrieved successfully.');
            }

            $q = Product::leftJoin('variations', 'products.id', '=', 'variations.product_id')
                ->where(function ($query) use ($term) {
                    $query->where('products.name', 'like', '%' . $term . '%')
                          ->orWhere('sku', 'like', '%' . $term . '%')
                          ->orWhere('sub_sku', 'like', '%' . $term . '%');
                })
                ->active()
                ->where('business_id', $business_id)
                ->whereNull('variations.deleted_at')
                ->select([
                    'products.id as product_id',
                    'products.name',
                    'products.type',
                    'variations.id as variation_id',
                    'variations.name as variation',
                    'variations.sub_sku as sub_sku'
                ])
                ->groupBy('variation_id');

            if ($check_enable_stock) {
                $q->where('enable_stock', 1);
            }

            if (!empty($request->location_id)) {
                $q->ForLocation($request->location_id);
            }

            $products = $q->get();
            
            // Format products for response
            $products_array = [];
            foreach ($products as $product) {
                $products_array[$product->product_id]['name'] = $product->name;
                $products_array[$product->product_id]['sku'] = $product->sub_sku;
                $products_array[$product->product_id]['type'] = $product->type;
                $products_array[$product->product_id]['variations'][] = [
                    'variation_id' => $product->variation_id,
                    'variation_name' => $product->variation,
                    'sub_sku' => $product->sub_sku
                ];
            }

            $result = [];
            foreach ($products_array as $key => $value) {
                foreach ($value['variations'] as $variation) {
                    $text = $value['name'];
                    if ($value['type'] == 'variable') {
                        $text = $text . ' (' . $variation['variation_name'] . ')';
                    }
                    $result[] = [
                        'product_id' => $key,
                        'variation_id' => $variation['variation_id'],
                        'text' => $text . ' - ' . $variation['sub_sku'],
                        'name' => $value['name'],
                        'type' => $value['type']
                    ];
                }
            }

            return $this->sendResponse($result, 'Products retrieved successfully.');

        } catch (\Exception $e) {
            return $this->sendError('Failed to get products.', [$e->getMessage()], 500);
        }
    }

    /**
     * Check reference number uniqueness
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkRefNumber(Request $request)
    {
        try {
            $business_id = $request->user()->business_id;
            $contact_id = $request->contact_id;
            $ref_no = $request->ref_no;
            $purchase_id = $request->purchase_id;

            $count = 0;
            if (!empty($contact_id) && !empty($ref_no)) {
                $query = Transaction::where('business_id', $business_id)
                                  ->where('ref_no', $ref_no)
                                  ->where('contact_id', $contact_id);
                
                if (!empty($purchase_id)) {
                    $query->where('id', '!=', $purchase_id);
                }
                
                $count = $query->count();
            }

            $is_unique = $count == 0;
            
            return $this->sendResponse([
                'is_unique' => $is_unique,
                'count' => $count
            ], $is_unique ? 'Reference number is unique.' : 'Reference number already exists.');

        } catch (\Exception $e) {
            return $this->sendError('Failed to check reference number.', [$e->getMessage()], 500);
        }
    }

    /**
     * Update purchase status
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateStatus(Request $request)
    {
        try {
            if (!auth()->user()->can('purchase.update') && !auth()->user()->can('purchase.update_status')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $edit_days = $request->user()->business->transaction_edit_days ?? 0;
            if (!$this->transactionUtil->canBeEdited($request->purchase_id, $edit_days)) {
                return $this->sendError("Transaction cannot be edited after {$edit_days} days.", [], 400);
            }

            $business_id = $request->user()->business_id;
            
            $transaction = Transaction::where('business_id', $business_id)
                                    ->where('type', 'purchase')
                                    ->with(['purchase_lines'])
                                    ->findOrFail($request->purchase_id);

            $before_status = $transaction->status;
            $update_data['status'] = $request->status;

            DB::beginTransaction();

            $transaction->update($update_data);

            $currency_details = $this->transactionUtil->purchaseCurrencyDetails($business_id);
            foreach ($transaction->purchase_lines as $purchase_line) {
                $this->productUtil->updateProductStock($before_status, $transaction, $purchase_line->product_id, $purchase_line->variation_id, $purchase_line->quantity, $purchase_line->quantity, $currency_details);
            }

            // Update mapping of purchase & Sell
            $this->transactionUtil->adjustMappingPurchaseSellAfterEditingPurchase($before_status, $transaction, null);

            // Adjust stock over selling if found
            $this->productUtil->adjustStockOverSelling($transaction);

            DB::commit();

            return $this->sendResponse($transaction, 'Purchase status updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Failed to update purchase status.', [$e->getMessage()], 500);
        }
    }
}
