<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Brand;
use App\Models\BusinessLocation;
use App\Models\Category;
use App\Models\Product;
use App\Models\TaxRate;
use App\Models\Unit;
use App\Models\Variation;
use App\Models\VariationLocationDetail;
use App\Models\Warranty;
use App\Utils\ModuleUtil;
use App\Utils\ProductUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class ProductController extends BaseController
{
    protected $productUtil;
    protected $moduleUtil;
    private $barcode_types;

    public function __construct(ProductUtil $productUtil, ModuleUtil $moduleUtil)
    {
        $this->productUtil = $productUtil;
        $this->moduleUtil = $moduleUtil;
        $this->barcode_types = $this->productUtil->barcode_types();
    }

    /**
     * Display a listing of products
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $query = Product::leftJoin('brands', 'products.brand_id', '=', 'brands.id')
                ->join('units', 'products.unit_id', '=', 'units.id')
                ->leftJoin('categories as c1', 'products.category_id', '=', 'c1.id')
                ->leftJoin('categories as c2', 'products.sub_category_id', '=', 'c2.id')
                ->leftJoin('tax_rates', 'products.tax', '=', 'tax_rates.id')
                ->join('variations as v', 'v.product_id', '=', 'products.id')
                ->leftJoin('variation_location_details as vld', 'vld.variation_id', '=', 'v.id')
                ->where('products.business_id', $business_id)
                ->where('products.type', '!=', 'modifier');

            // Apply filters
            $this->applyFilters($query, $request);

            $products = $query->select(
                'products.id',
                'products.name as product',
                'products.type',
                'c1.name as category',
                'c2.name as sub_category',
                'units.actual_name as unit',
                'brands.name as brand',
                'tax_rates.name as tax',
                'products.sku',
                'products.image',
                'products.enable_stock',
                'products.is_inactive',
                'products.not_for_selling',
                DB::raw('SUM(vld.qty_available) as current_stock'),
                DB::raw('MAX(v.sell_price_inc_tax) as max_price'),
                DB::raw('MIN(v.sell_price_inc_tax) as min_price')
            )
                ->groupBy('products.id')
                ->paginate($request->get('per_page', 15));

            return $this->sendResponse($products, 'Products retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Product index error: ' . $e->getMessage());
            return $this->sendError('Error retrieving products', [], 500);
        }
    }

    /**
     * Store a newly created product
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'unit_id' => 'required|exists:units,id',
            'category_id' => 'nullable|exists:categories,id',
            'brand_id' => 'nullable|exists:brands,id',
            'type' => 'required|in:single,variable,combo',
            'tax_type' => 'required|in:inclusive,exclusive',
            'sku' => 'nullable|string|unique:products,sku'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        //try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            DB::beginTransaction();

            $product_details = $request->only([
                'name',
                'brand_id',
                'unit_id',
                'category_id',
                'tax',
                'type',
                'barcode_type',
                'sku',
                'alert_quantity',
                'tax_type',
                'weight',
                'product_custom_field1',
                'product_custom_field2',
                'product_custom_field3',
                'product_custom_field4',
                'product_description',
                'sub_unit_ids',
                'sub_category_id'
            ]);

            $product_details['business_id'] = $business_id;
            $product_details['created_by'] = auth()->user()->id;
            $product_details['enable_stock'] = $request->get('enable_stock', 0);
            $product_details['not_for_selling'] = $request->get('not_for_selling', 0);

            if (empty($product_details['sku'])) {
                $product_details['sku'] = ' ';
            }

            // Handle image upload
            if ($request->hasFile('image')) {
                $product_details['image'] = $this->productUtil->uploadFile(
                    $request,
                    'image',
                    config('constants.product_img_path'),
                    'image'
                );
            }

            $product = Product::create($product_details);

            if (empty(trim($request->input('sku')))) {
                $sku = $this->productUtil->generateProductSku($product->id);
                $product->sku = $sku;
                $product->save();
            }

            // Add product locations
            if ($request->has('product_locations')) {
                $product->product_locations()->sync($request->input('product_locations'));
            }

            // Create variations based on product type
            $this->createProductVariations($product, $request);

            DB::commit();

            $product->load(['variations', 'product_locations', 'category', 'brand', 'unit']);

            return $this->sendResponse($product, 'Product created successfully');
        //} catch (\Exception $e) {
        //    DB::rollBack();
        //    Log::error('Product store error: ' . $e->getMessage());
        //    return $this->sendError('Error creating product', [], 500);
        //}
    }

    /**
     * Display the specified product
     * 
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $product = Product::where('business_id', $business_id)
                ->with([
                    'brand',
                    'unit',
                    'category',
                    'sub_category',
                    'product_tax',
                    'variations',
                    'variations.product_variation',
                    'variations.group_prices',
                    'variations.media',
                    'product_locations',
                    'warranty'
                ])
                ->find($id);

            if (!$product) {
                return $this->sendError('Product not found', [], 404);
            }

            return $this->sendResponse($product, 'Product retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Product show error: ' . $e->getMessage());
            return $this->sendError('Error retrieving product', [], 500);
        }
    }

    /**
     * Update the specified product
     * 
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'unit_id' => 'sometimes|required|exists:units,id',
            'category_id' => 'nullable|exists:categories,id',
            'brand_id' => 'nullable|exists:brands,id',
            'sku' => 'nullable|string|unique:products,sku,' . $id
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            DB::beginTransaction();

            $product = Product::where('business_id', $business_id)
                ->where('id', $id)
                ->with(['product_variations'])
                ->first();

            if (!$product) {
                return $this->sendError('Product not found', [], 404);
            }

            $product_details = $request->only([
                'name',
                'brand_id',
                'unit_id',
                'category_id',
                'tax',
                'barcode_type',
                'sku',
                'alert_quantity',
                'tax_type',
                'weight',
                'product_custom_field1',
                'product_custom_field2',
                'product_custom_field3',
                'product_custom_field4',
                'product_description',
                'sub_unit_ids',
                'sub_category_id'
            ]);

            $product->update($product_details);

            // Handle image upload
            if ($request->hasFile('image')) {
                $file_name = $this->productUtil->uploadFile(
                    $request,
                    'image',
                    config('constants.product_img_path'),
                    'image'
                );
                if (!empty($file_name)) {
                    $product->image = $file_name;
                    $product->save();
                }
            }

            // Update product locations
            if ($request->has('product_locations')) {
                $product->product_locations()->sync($request->input('product_locations'));
            }

            DB::commit();

            $product->load(['variations', 'product_locations', 'category', 'brand', 'unit']);

            return $this->sendResponse($product, 'Product updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Product update error: ' . $e->getMessage());
            return $this->sendError('Error updating product', [], 500);
        }
    }

    /**
     * Remove the specified product
     * 
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $product = Product::where('id', $id)
                ->where('business_id', $business_id)
                ->with('variations')
                ->first();

            if (!$product) {
                return $this->sendError('Product not found', [], 404);
            }

            // Check if product can be deleted
            $count = \DB::table('purchase_lines')
                ->join('transactions as T', 'purchase_lines.transaction_id', '=', 'T.id')
                ->whereIn('T.type', ['purchase'])
                ->where('T.business_id', $business_id)
                ->where('purchase_lines.product_id', $id)
                ->count();

            if ($count > 0) {
                return $this->sendError('Cannot delete product - purchases exist', [], 422);
            }

            DB::beginTransaction();

            // Delete variation location details
            VariationLocationDetail::where('product_id', $id)->delete();
            $product->delete();

            DB::commit();

            return $this->sendResponse([], 'Product deleted successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Product destroy error: ' . $e->getMessage());
            return $this->sendError('Error deleting product', [], 500);
        }
    }

    /**
     * Get products for dropdown/search
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProducts(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;
            $search_term = $request->get('search', '');
            $location_id = $request->get('location_id');
            $check_qty = $request->get('check_qty', false);
            $product_types = $request->get('product_types', []);

            $search_fields = $request->get('search_fields', ['name', 'sku']);
            if (in_array('sku', $search_fields)) {
                $search_fields[] = 'sub_sku';
            }

            $result = $this->productUtil->filterProduct(
                $business_id,
                $search_term,
                $location_id,
                null,
                '',
                $product_types,
                $search_fields,
                $check_qty
            );

            return $this->sendResponse($result, 'Products retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Get products error: ' . $e->getMessage());
            return $this->sendError('Error retrieving products', [], 500);
        }
    }

    /**
     * Get subcategories for a category
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getSubCategories(Request $request)
    {
        try {
            $category_id = $request->get('category_id');
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            if (!$category_id) {
                return $this->sendError('Category ID is required', [], 422);
            }

            $sub_categories = Category::where('business_id', $business_id)
                ->where('parent_id', $category_id)
                ->select(['name', 'id'])
                ->get();

            return $this->sendResponse($sub_categories, 'Subcategories retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Get subcategories error: ' . $e->getMessage());
            return $this->sendError('Error retrieving subcategories', [], 500);
        }
    }

    /**
     * Check if product SKU exists
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkSku(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;
            $sku = $request->get('sku');
            $product_id = $request->get('product_id');

            if (!$sku) {
                return $this->sendError('SKU is required', [], 422);
            }

            $query = Product::where('business_id', $business_id)->where('sku', $sku);

            if ($product_id) {
                $query->where('id', '!=', $product_id);
            }

            $count = $query->count();

            if ($count == 0) {
                $count = Variation::where('sub_sku', $sku)
                    ->join('products', 'variations.product_id', '=', 'products.id')
                    ->where('product_id', '!=', $product_id)
                    ->where('business_id', $business_id)
                    ->count();
            }

            $available = $count == 0;

            return $this->sendResponse(['available' => $available], 'SKU check completed');
        } catch (\Exception $e) {
            Log::error('Check SKU error: ' . $e->getMessage());
            return $this->sendError('Error checking SKU', [], 500);
        }
    }

    /**
     * Apply filters to product query
     */
    private function applyFilters($query, $request)
    {
        if ($request->has('location_id') && $request->get('location_id') != 'none') {
            $location_id = $request->get('location_id');
            $query->whereHas('product_locations', function ($q) use ($location_id) {
                $q->where('product_locations.location_id', '=', $location_id);
            });
        }

        if ($request->has('type')) {
            $query->where('products.type', $request->get('type'));
        }

        if ($request->has('category_id')) {
            $query->where('products.category_id', $request->get('category_id'));
        }

        if ($request->has('brand_id')) {
            $query->where('products.brand_id', $request->get('brand_id'));
        }

        if ($request->has('unit_id')) {
            $query->where('products.unit_id', $request->get('unit_id'));
        }

        if ($request->has('active_state')) {
            $active_state = $request->get('active_state');
            if ($active_state == 'active') {
                $query->where('products.is_inactive', 0);
            } elseif ($active_state == 'inactive') {
                $query->where('products.is_inactive', 1);
            }
        }

        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('products.name', 'like', "%{$search}%")
                    ->orWhere('products.sku', 'like', "%{$search}%");
            });
        }
    }

    /**
     * Create product variations based on type
     */
    private function createProductVariations($product, $request)
    {
        if ($product->type == 'single') {
            $this->productUtil->createSingleProductVariation(
                $product->id,
                $product->sku,
                $request->input('single_dpp'),
                $request->input('single_dpp_inc_tax'),
                $request->input('profit_percent'),
                $request->input('single_dsp'),
                $request->input('single_dsp_inc_tax')
            );
        } elseif ($product->type == 'variable') {
            if ($request->has('product_variation')) {
                $input_variations = $request->input('product_variation');
                $this->productUtil->createVariableProductVariations($product->id, $input_variations);
            }
        } elseif ($product->type == 'combo') {
            $combo_variations = [];
            if ($request->has('composition_variation_id')) {
                $composition_variation_id = $request->input('composition_variation_id');
                $quantity = $request->input('quantity');
                $unit = $request->input('unit');

                foreach ($composition_variation_id as $key => $value) {
                    $combo_variations[] = [
                        'variation_id' => $value,
                        'quantity' => $this->productUtil->num_uf($quantity[$key]),
                        'unit_id' => $unit[$key]
                    ];
                }
            }

            $this->productUtil->createSingleProductVariation(
                $product->id,
                $product->sku,
                $request->input('item_level_purchase_price_total'),
                $request->input('purchase_price_inc_tax'),
                $request->input('profit_percent'),
                $request->input('selling_price'),
                $request->input('selling_price_inc_tax'),
                $combo_variations
            );
        }
    }

    /**
     * Get product types
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProductTypes()
    {
        $product_types = [
            'single' => __('lang_v1.single'),
            'variable' => __('lang_v1.variable'),
            'combo' => __('lang_v1.combo')
        ];

        return $this->sendResponse($product_types, 'Product types retrieved successfully');
    }

    /**
     * Get form data for product creation
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getFormData(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $data = [
                'categories' => Category::forDropdown($business_id, 'product'),
                'brands' => Brand::where('business_id', $business_id)->pluck('name', 'id'),
                'units' => Unit::forDropdown($business_id, true),
                'taxes' => TaxRate::forBusinessDropdown($business_id, false)['tax_rates'],
                'business_locations' => BusinessLocation::forDropdown($business_id),
                'barcode_types' => $this->barcode_types,
                'warranties' => Warranty::forDropdown($business_id)
            ];

            return $this->sendResponse($data, 'Form data retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Get form data error: ' . $e->getMessage());
            return $this->sendError('Error retrieving form data', [], 500);
        }
    }
}
