<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\NotificationTemplate;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Exception;


class NotificationTemplateController extends BaseController
{
    protected $moduleUtil;

    public function __construct(ModuleUtil $moduleUtil)
    {
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Récupérer tous les templates de notification
     * 
     * @response 200 {
     *   "success": true,
     *   "data": {
     *     "general_notifications": [...],
     *     "customer_notifications": [...],
     *     "supplier_notifications": [...]
     *   },
     *   "message": "Notification templates retrieved successfully"
     * }
     */
    public function index()
    {
        try {
            if (!auth()->user()->can('send_notification')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $general_notifications = NotificationTemplate::generalNotifications();
            $general_notifications = $this->getTemplateDetails($general_notifications);

            $customer_notifications = NotificationTemplate::customerNotifications();
            $module_customer_notifications = $this->moduleUtil->getModuleData('notification_list', ['notification_for' => 'customer']);

            if (!empty($module_customer_notifications)) {
                foreach ($module_customer_notifications as $module_customer_notification) {
                    $customer_notifications = array_merge($customer_notifications, $module_customer_notification);
                }
            }
            $customer_notifications = $this->getTemplateDetails($customer_notifications);

            $supplier_notifications = NotificationTemplate::supplierNotifications();
            $module_supplier_notifications = $this->moduleUtil->getModuleData('notification_list', ['notification_for' => 'supplier']);

            if (!empty($module_supplier_notifications)) {
                foreach ($module_supplier_notifications as $module_supplier_notification) {
                    $supplier_notifications = array_merge($supplier_notifications, $module_supplier_notification);
                }
            }
            $supplier_notifications = $this->getTemplateDetails($supplier_notifications);

            $data = [
                'general_notifications' => $general_notifications,
                'customer_notifications' => $customer_notifications,
                'supplier_notifications' => $supplier_notifications
            ];

            return $this->sendResponse($data, 'Notification templates retrieved successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to retrieve notification templates', [], 500);
        }
    }

    /**
     * Récupérer un template de notification spécifique
     * 
     * @urlParam id int required L'ID du template
     */
    public function show($id)
    {
        try {
            if (!auth()->user()->can('send_notification')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $notification_template = NotificationTemplate::where('business_id', $business_id)->findOrFail($id);

            return $this->sendResponse($notification_template, 'Notification template retrieved successfully');
        } catch (Exception $e) {
            return $this->sendError('Notification template not found', [], 404);
        }
    }

    /**
     * Créer un nouveau template de notification
     * 
     * @bodyParam template_for string required Type de template
     * @bodyParam subject string required Sujet
     * @bodyParam email_body string required Corps de l'email
     * @bodyParam sms_body string required Corps du SMS
     * @bodyParam auto_send boolean Envoi automatique des emails
     * @bodyParam auto_send_sms boolean Envoi automatique des SMS
     * @bodyParam cc string Emails en copie
     * @bodyParam bcc string Emails en copie cachée
     */
    public function store(Request $request)
    {
        try {
            if (!auth()->user()->can('send_notification')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $validatedData = $request->validate([
                'template_for' => 'required|string',
                'subject' => 'required|string',
                'email_body' => 'required|string',
                'sms_body' => 'required|string',
                'auto_send' => 'boolean',
                'auto_send_sms' => 'boolean',
                'cc' => 'nullable|string',
                'bcc' => 'nullable|string'
            ]);

            $business_id = auth()->user()->business_id;
            $validatedData['business_id'] = $business_id;

            $notification_template = NotificationTemplate::create($validatedData);

            return $this->sendResponse($notification_template, 'Notification template created successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to create notification template', [], 500);
        }
    }

    /**
     * Mettre à jour un template de notification
     * 
     * @urlParam id int required L'ID du template
     */
    public function update(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('send_notification')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $notification_template = NotificationTemplate::where('business_id', $business_id)->findOrFail($id);

            $validatedData = $request->validate([
                'template_for' => 'sometimes|string',
                'subject' => 'sometimes|string',
                'email_body' => 'sometimes|string',
                'sms_body' => 'sometimes|string',
                'auto_send' => 'sometimes|boolean',
                'auto_send_sms' => 'sometimes|boolean',
                'cc' => 'nullable|string',
                'bcc' => 'nullable|string'
            ]);

            $notification_template->update($validatedData);

            return $this->sendResponse($notification_template, 'Notification template updated successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to update notification template', [], 500);
        }
    }

    /**
     * Supprimer un template de notification
     * 
     * @urlParam id int required L'ID du template
     */
    public function destroy($id)
    {
        try {
            if (!auth()->user()->can('send_notification')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $notification_template = NotificationTemplate::where('business_id', $business_id)->findOrFail($id);
            $notification_template->delete();

            return $this->sendResponse([], 'Notification template deleted successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to delete notification template', [], 500);
        }
    }

    /**
     * Sauvegarder plusieurs templates en une fois
     * 
     * @bodyParam template_data array required Données des templates à sauvegarder
     */
    public function bulkUpdate(Request $request)
    {
        try {
            if (!auth()->user()->can('send_notification')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $template_data = $request->input('template_data');
            $business_id = auth()->user()->business_id;

            foreach ($template_data as $key => $value) {
                NotificationTemplate::updateOrCreate(
                    [
                        'business_id' => $business_id,
                        'template_for' => $key
                    ],
                    [
                        'subject' => $value['subject'],
                        'email_body' => $value['email_body'],
                        'sms_body' => $value['sms_body'],
                        'auto_send' => !empty($value['auto_send']) ? 1 : 0,
                        'auto_send_sms' => !empty($value['auto_send_sms']) ? 1 : 0,
                        'cc' => $value['cc'] ?? null,
                        'bcc' => $value['bcc'] ?? null,
                    ]
                );
            }

            return $this->sendResponse([], 'Notification templates updated successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to update notification templates', [], 500);
        }
    }

    /**
     * Méthode privée pour récupérer les détails des templates
     */
    private function getTemplateDetails($notifications)
    {
        $business_id = auth()->user()->business_id;
        foreach ($notifications as $key => $value) {
            $notification_template = NotificationTemplate::getTemplate($business_id, $key);
            $notifications[$key]['subject'] = $notification_template['subject'];
            $notifications[$key]['email_body'] = $notification_template['email_body'];
            $notifications[$key]['sms_body'] = $notification_template['sms_body'];
            $notifications[$key]['auto_send'] = $notification_template['auto_send'];
            $notifications[$key]['auto_send_sms'] = $notification_template['auto_send_sms'];
            $notifications[$key]['cc'] = $notification_template['cc'];
            $notifications[$key]['bcc'] = $notification_template['bcc'];
        }
        return $notifications;
    }
}
