<?php

namespace App\Http\Controllers;

use App\Models\Contact;
use App\Notifications\CustomerNotification;
use App\Notifications\SupplierNotification;
use App\Models\NotificationTemplate;
use App\Restaurant\Booking;
use App\Models\Transaction;
use App\Utils\NotificationUtil;
use Illuminate\Http\Request;
use Exception;
use Illuminate\Support\Facades\Notification as FacadesNotification;
use Notification;

class NotificationController extends Controller
{
    protected $notificationUtil;

    public function __construct(NotificationUtil $notificationUtil)
    {
        $this->notificationUtil = $notificationUtil;
    }

    /**
     * Récupérer toutes les notifications
     * 
     * @response 200 {
     *   "success": true,
     *   "data": {
     *     "notifications": [...],
     *     "customer_notifications": [...],
     *     "supplier_notifications": [...],
     *     "general_notifications": [...]
     *   },
     *   "message": "Notifications retrieved successfully"
     * }
     */
    public function index()
    {
        try {
            $customer_notifications = NotificationTemplate::customerNotifications();
            $supplier_notifications = NotificationTemplate::supplierNotifications();
            $general_notifications = NotificationTemplate::generalNotifications();

            $data = [
                'customer_notifications' => $customer_notifications,
                'supplier_notifications' => $supplier_notifications,
                'general_notifications' => $general_notifications
            ];

            return $this->sendResponse($data, 'Notifications retrieved successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to retrieve notifications', [], 500);
        }
    }

    /**
     * Récupérer un template de notification
     * 
     * @urlParam id int required L'ID de l'élément (transaction, contact, booking)
     * @urlParam template_for string required Type de template (new_booking, send_ledger, etc.)
     */
    public function show($id, $template_for)
    {
        try {
            $business_id = auth()->user()->business_id;
            $notification_template = NotificationTemplate::getTemplate($business_id, $template_for);

            $contact = null;
            $transaction = null;

            if ($template_for == 'new_booking') {
                $transaction = Booking::where('business_id', $business_id)
                    ->with(['customer'])
                    ->find($id);
                $contact = $transaction ? $transaction->customer : null;
            } elseif ($template_for == 'send_ledger') {
                $contact = Contact::find($id);
            } else {
                $transaction = Transaction::where('business_id', $business_id)
                    ->with(['contact'])
                    ->find($id);
                $contact = $transaction ? $transaction->contact : null;
            }

            $customer_notifications = NotificationTemplate::customerNotifications();
            $supplier_notifications = NotificationTemplate::supplierNotifications();
            $general_notifications = NotificationTemplate::generalNotifications();

            $template_name = '';
            $tags = [];

            if (array_key_exists($template_for, $customer_notifications)) {
                $template_name = $customer_notifications[$template_for]['name'];
                $tags = $customer_notifications[$template_for]['extra_tags'];
            } elseif (array_key_exists($template_for, $supplier_notifications)) {
                $template_name = $supplier_notifications[$template_for]['name'];
                $tags = $supplier_notifications[$template_for]['extra_tags'];
            } elseif (array_key_exists($template_for, $general_notifications)) {
                $template_name = $general_notifications[$template_for]['name'];
                $tags = $general_notifications[$template_for]['extra_tags'];
            }

            $data = [
                'notification_template' => $notification_template,
                'transaction' => $transaction,
                'contact' => $contact,
                'template_name' => $template_name,
                'tags' => $tags
            ];

            return $this->sendResponse($data, 'Notification template retrieved successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to retrieve notification template', [], 500);
        }
    }

    /**
     * Créer un nouveau template de notification
     * 
     * @bodyParam template_for string required Type de template
     * @bodyParam subject string required Sujet de la notification
     * @bodyParam email_body string required Corps de l'email
     * @bodyParam sms_body string required Corps du SMS
     * @bodyParam auto_send boolean Envoi automatique des emails
     * @bodyParam auto_send_sms boolean Envoi automatique des SMS
     */
    public function store(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'template_for' => 'required|string',
                'subject' => 'required|string',
                'email_body' => 'required|string',
                'sms_body' => 'required|string',
                'auto_send' => 'boolean',
                'auto_send_sms' => 'boolean',
                'cc' => 'nullable|string',
                'bcc' => 'nullable|string'
            ]);

            $business_id = auth()->user()->business_id;

            $notification_template = NotificationTemplate::create([
                'business_id' => $business_id,
                'template_for' => $validatedData['template_for'],
                'subject' => $validatedData['subject'],
                'email_body' => $validatedData['email_body'],
                'sms_body' => $validatedData['sms_body'],
                'auto_send' => $validatedData['auto_send'] ?? false,
                'auto_send_sms' => $validatedData['auto_send_sms'] ?? false,
                'cc' => $validatedData['cc'] ?? null,
                'bcc' => $validatedData['bcc'] ?? null
            ]);

            return $this->sendResponse($notification_template, 'Notification template created successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to create notification template', [], 500);
        }
    }

    /**
     * Mettre à jour un template de notification
     * 
     * @urlParam id int required L'ID du template
     */
    public function update(Request $request, $id)
    {
        try {
            $business_id = auth()->user()->business_id;
            $notification_template = NotificationTemplate::where('business_id', $business_id)->findOrFail($id);

            $validatedData = $request->validate([
                'subject' => 'sometimes|string',
                'email_body' => 'sometimes|string',
                'sms_body' => 'sometimes|string',
                'auto_send' => 'sometimes|boolean',
                'auto_send_sms' => 'sometimes|boolean',
                'cc' => 'nullable|string',
                'bcc' => 'nullable|string'
            ]);

            $notification_template->update($validatedData);

            return $this->sendResponse($notification_template, 'Notification template updated successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to update notification template', [], 500);
        }
    }

    /**
     * Supprimer un template de notification
     * 
     * @urlParam id int required L'ID du template
     */
    public function destroy($id)
    {
        try {
            $business_id = auth()->user()->business_id;
            $notification_template = NotificationTemplate::where('business_id', $business_id)->findOrFail($id);
            $notification_template->delete();

            return $this->sendResponse([], 'Notification template deleted successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to delete notification template', [], 500);
        }
    }

    /**
     * Envoyer une notification
     * 
     * @bodyParam to_email string required Email(s) de destination (séparés par des virgules)
     * @bodyParam subject string required Sujet
     * @bodyParam email_body string required Corps de l'email
     * @bodyParam mobile_number string Numéro de téléphone pour SMS
     * @bodyParam sms_body string Corps du SMS
     * @bodyParam notification_type string required Type: email_only, sms_only, both
     * @bodyParam template_for string required Type de template
     * @bodyParam transaction_id int ID de la transaction (optionnel)
     * @bodyParam cc string Emails en copie
     * @bodyParam bcc string Emails en copie cachée
     */
    public function send(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'to_email' => 'required|string',
                'subject' => 'required|string',
                'email_body' => 'required|string',
                'mobile_number' => 'nullable|string',
                'sms_body' => 'nullable|string',
                'notification_type' => 'required|in:email_only,sms_only,both',
                'template_for' => 'required|string',
                'transaction_id' => 'nullable|integer',
                'cc' => 'nullable|string',
                'bcc' => 'nullable|string'
            ]);

            $notAllowed = $this->notificationUtil->notAllowedInDemo();
            if (!empty($notAllowed)) {
                return $this->sendError('Demo restriction', [], 403);
            }

            $customer_notifications = NotificationTemplate::customerNotifications();
            $supplier_notifications = NotificationTemplate::supplierNotifications();

            $data = $validatedData;
            $emails_array = array_map('trim', explode(',', $data['to_email']));
            $transaction_id = $request->input('transaction_id');
            $business_id = auth()->user()->business_id;

            $orig_data = [
                'email_body' => $data['email_body'],
                'sms_body' => $data['sms_body'],
                'subject' => $data['subject']
            ];

            if ($request->input('template_for') == 'new_booking') {
                $tag_replaced_data = $this->notificationUtil->replaceBookingTags($business_id, $orig_data, $transaction_id);
            } else {
                $tag_replaced_data = $this->notificationUtil->replaceTags($business_id, $orig_data, $transaction_id);
            }

            $data['email_body'] = $tag_replaced_data['email_body'];
            $data['sms_body'] = $tag_replaced_data['sms_body'];
            $data['subject'] = $tag_replaced_data['subject'];
            $data['email_settings'] = auth()->user()->business->email_settings ?? [];
            $data['sms_settings'] = auth()->user()->business->sms_settings ?? [];

            $notification_type = $request->input('notification_type');
            $template_for = $request->input('template_for');

            if (array_key_exists($template_for, $customer_notifications)) {
                if ($notification_type == 'email_only') {
                    FacadesNotification::route('mail', $emails_array)
                        ->notify(new CustomerNotification($data));
                } elseif ($notification_type == 'sms_only') {
                    $this->notificationUtil->sendSms($data);
                } elseif ($notification_type == 'both') {
                    FacadesNotification::route('mail', $emails_array)
                        ->notify(new CustomerNotification($data));
                    $this->notificationUtil->sendSms($data);
                }
            } elseif (array_key_exists($template_for, $supplier_notifications)) {
                if ($notification_type == 'email_only') {
                    FacadesNotification::route('mail', $emails_array)
                        ->notify(new SupplierNotification($data));
                } elseif ($notification_type == 'sms_only') {
                    $this->notificationUtil->sendSms($data);
                } elseif ($notification_type == 'both') {
                    FacadesNotification::route('mail', $emails_array)
                        ->notify(new SupplierNotification($data));
                    $this->notificationUtil->sendSms($data);
                }
            }

            return $this->sendResponse([], 'Notification sent successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to send notification', [], 500);
        }
    }
}
