<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController as ApiBaseController;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Models\User;
use App\Models\Contact;
use App\Models\BusinessLocation;
use App\Models\System;
use Spatie\Permission\Models\Role;
use App\Utils\ModuleUtil;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
class ManageUserController extends ApiBaseController
{
    /**
     * All Utils instance.
     *
     */
    protected $moduleUtil;

    /**
     * Constructor
     *
     * @param ModuleUtil $moduleUtil
     * @return void
     */
    public function __construct(ModuleUtil $moduleUtil)
    {
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        
        if (!auth()->user()->can('user.view')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }
        try {
            // Vérifier l'authentification Sanctum
            $user = auth('sanctum')->user();

            if (!$user) {
                return $this->sendError('Non authentifié', [], 401);
            }
            
            

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;
            $user_id = $request->user()->id ?? $request->session()->get('user.id');

            $users = User::where('business_id', $business_id)
                ->user()
                ->where('is_cmmsn_agnt', 0)
                ->select(['id', 'username', 'surname', 'first_name', 'last_name', 'email', 'allow_login', 'status'])
                ->get()
                ->map(function ($user) {
                    $full_name = trim($user->surname . ' ' . $user->first_name . ' ' . $user->last_name);
                    $role_name = $this->moduleUtil->getUserRoleName($user->id);

                    // Determine available actions
                    $actions = [];
                    if (auth()->user()->can('user.update')) {
                        $actions[] = 'edit';
                    }
                    if (auth()->user()->can('user.view')) {
                        $actions[] = 'view';
                    }
                    if (auth()->user()->can('user.delete')) {
                        $actions[] = 'delete';
                    }

                    return [
                        'id' => $user->id,
                        'username' => $user->username,
                        'full_name' => $full_name,
                        'email' => $user->email,
                        'allow_login' => $user->allow_login,
                        'status' => $user->status,
                        'role' => $role_name,
                        'actions' => $actions
                    ];
                });

            return $this->sendResponse($users, 'Users retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Get data needed for creating a new user.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function create(Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('user.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            // Check subscription and quota
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            }

            if (!$this->moduleUtil->isQuotaAvailable('users', $business_id)) {
                return $this->sendError('User quota exceeded.', [], 402);
            }

            $roles = $this->getRolesArray($business_id);
            $username_ext = $this->getUsernameExtension();
            $contacts = Contact::contactDropdown($business_id, true, false);
            $locations = BusinessLocation::where('business_id', $business_id)
                ->active()
                ->select(['id', 'name'])
                ->get();

            // Get user form part from modules
            $form_partials = $this->moduleUtil->getModuleData('moduleViewPartials', ['view' => 'manage_user.create']);

            $data = [
                'roles' => $roles,
                'username_extension' => $username_ext,
                'contacts' => $contacts,
                'locations' => $locations,
                'form_partials' => $form_partials
            ];

            return $this->sendResponse($data, 'User creation data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function store(Request $request): JsonResponse
    {
        if (!auth()->user()->can('user.create')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }
        try {
            // Validation
            $validator = Validator::make($request->all(), [
                'surname' => 'nullable|string|max:255',
                'first_name' => 'required|string|max:255',
                'last_name' => 'nullable|string|max:255',
                'username' => 'nullable|string|max:255|unique:users',
                'email' => 'nullable|email|unique:users',
                'password' => 'nullable|string|min:6',
                'role' => 'required|exists:roles,id',
                'contact_number' => 'nullable|string|max:20',
                'allow_login' => 'boolean',
                'is_active' => 'boolean',
                'selected_contacts' => 'boolean',
                'selected_contact_ids' => 'array',
                'location_permissions' => 'array',
                'access_all_locations' => 'nullable|string',
                'dob' => 'nullable|date',
                'cmmsn_percent' => 'nullable|numeric|min:0|max:100',
                'max_sales_discount_percent' => 'nullable|numeric|min:0|max:100'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            // Check subscription and quota
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            }

            if (!$this->moduleUtil->isQuotaAvailable('users', $business_id)) {
                return $this->sendError('User quota exceeded.', [], 402);
            }

            $user_details = $request->only([
                'surname',
                'first_name',
                'last_name',
                'username',
                'email',
                'password',
                'selected_contacts',
                'marital_status',
                'blood_group',
                'contact_number',
                'fb_link',
                'twitter_link',
                'social_media_1',
                'social_media_2',
                'permanent_address',
                'current_address',
                'guardian_name',
                'custom_field_1',
                'custom_field_2',
                'custom_field_3',
                'custom_field_4',
                'id_proof_name',
                'id_proof_number',
                'cmmsn_percent',
                'gender',
                'max_sales_discount_percent'
            ]);

            $user_details['status'] = $request->input('is_active', true) ? 'active' : 'inactive';
            $user_details['user_type'] = 'user';
            $user_details['business_id'] = $business_id;

            // Handle login permissions
            if (!$request->input('allow_login', false)) {
                unset($user_details['username']);
                unset($user_details['password']);
                $user_details['allow_login'] = 0;
            } else {
                $user_details['allow_login'] = 1;
                if (!empty($user_details['password'])) {
                    $user_details['password'] = Hash::make($user_details['password']);
                }

                // Generate username if not provided
                if (empty($user_details['username'])) {
                    $ref_count = $this->moduleUtil->setAndGetReferenceCount('username');
                    $user_details['username'] = $this->moduleUtil->generateReferenceNumber('username', $ref_count);
                }

                $username_ext = $this->getUsernameExtension();
                if (!empty($username_ext)) {
                    $user_details['username'] .= $username_ext;
                }
            }

            // Handle date of birth
            if (!empty($request->input('dob'))) {
                $user_details['dob'] = Carbon::createFromFormat('Y-m-d', $request->input('dob'));
            }

            // Handle bank details
            if (!empty($request->input('bank_details'))) {
                $user_details['bank_details'] = json_encode($request->input('bank_details'));
            }

            // Handle commission percentage
            $user_details['cmmsn_percent'] = !empty($user_details['cmmsn_percent'])
                ? $this->moduleUtil->num_uf($user_details['cmmsn_percent']) : 0;

            $user_details['max_sales_discount_percent'] = !is_null($user_details['max_sales_discount_percent'])
                ? $this->moduleUtil->num_uf($user_details['max_sales_discount_percent']) : null;

            if (!isset($user_details['selected_contacts'])) {
                $user_details['selected_contacts'] = false;
            }

            // Create the user
            $user = User::create($user_details);

            // Assign role
            $role_id = $request->input('role');
            $role = Role::findOrFail($role_id);
            $user->assignRole($role->name);

            // Grant location permissions
            $this->giveLocationPermissions($user, $request);

            // Assign selected contacts
            if ($user_details['selected_contacts'] == 1) {
                $contact_ids = $request->get('selected_contact_ids', []);
                $user->contactAccess()->sync($contact_ids);
            }

            // Save module fields for user
            $this->moduleUtil->getModuleData('afterModelSaved', [
                'event' => 'user_saved',
                'model_instance' => $user
            ]);

            $user_data = [
                'id' => $user->id,
                'full_name' => trim($user->surname . ' ' . $user->first_name . ' ' . $user->last_name),
                'username' => $user->username,
                'email' => $user->email,
                'status' => $user->status,
                'allow_login' => $user->allow_login,
                'role' => $role->name
            ];

            return $this->sendResponse($user_data, 'User created successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @param Request $request
     * @return JsonResponse
     */
    public function show($id, Request $request): JsonResponse
    {
        //try {


        if (!auth()->user()->can('user.view')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $user = User::where('business_id', $business_id)
                ->with(['contactAccess', 'roles'])
                ->find($id);

            if (!$user) {
                return $this->sendError('User not found.', [], 404);
            }

            // Get user view part from modules
            $view_partials = $this->moduleUtil->getModuleData('moduleViewPartials', [
                'view' => 'manage_user.show',
                'user' => $user
            ]);

            $permitted_locations = $user->permitted_locations();
            $role_name = $this->moduleUtil->getUserRoleName($user->id);

            $user_data = [
                'id' => $user->id,
                'surname' => $user->surname,
                'first_name' => $user->first_name,
                'last_name' => $user->last_name,
                'full_name' => trim($user->surname . ' ' . $user->first_name . ' ' . $user->last_name),
                'username' => $user->username,
                'email' => $user->email,
                'contact_number' => $user->contact_number,
                'status' => $user->status,
                'allow_login' => $user->allow_login,
                'dob' => $user->dob ? $user->dob->format('Y-m-d') : null,
                'gender' => $user->gender,
                'marital_status' => $user->marital_status,
                'blood_group' => $user->blood_group,
                'contact_access' => $user->contactAccess->pluck('id')->toArray(),
                'permitted_locations' => $permitted_locations,
                'role' => $role_name,
                'cmmsn_percent' => $user->cmmsn_percent,
                'max_sales_discount_percent' => $user->max_sales_discount_percent,
                'view_partials' => $view_partials
            ];

            return $this->sendResponse($user_data, 'User retrieved successfully.');
        
    }

    /**
     * Get data needed for editing a user.
     *
     * @param int $id
     * @param Request $request
     * @return JsonResponse
     */
    public function edit($id, Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('user.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $user = User::where('business_id', $business_id)
                ->with(['contactAccess', 'roles'])
                ->findOrFail($id);

            $roles = $this->getRolesArray($business_id);
            $contact_access = $user->contactAccess->pluck('id')->toArray();
            $contacts = Contact::contactDropdown($business_id, true, false);
            $locations = BusinessLocation::where('business_id', $business_id)->get();
            $permitted_locations = $user->permitted_locations();
            $username_ext = $this->getUsernameExtension();

            // Get user form part from modules
            $form_partials = $this->moduleUtil->getModuleData('moduleViewPartials', [
                'view' => 'manage_user.edit',
                'user' => $user
            ]);

            $user_data = [
                'id' => $user->id,
                'surname' => $user->surname,
                'first_name' => $user->first_name,
                'last_name' => $user->last_name,
                'username' => $user->username,
                'email' => $user->email,
                'contact_number' => $user->contact_number,
                'status' => $user->status,
                'is_active' => $user->status == 'active',
                'allow_login' => $user->allow_login,
                'dob' => $user->dob ? $user->dob->format('Y-m-d') : null,
                'gender' => $user->gender,
                'marital_status' => $user->marital_status,
                'blood_group' => $user->blood_group,
                'fb_link' => $user->fb_link,
                'twitter_link' => $user->twitter_link,
                'social_media_1' => $user->social_media_1,
                'social_media_2' => $user->social_media_2,
                'permanent_address' => $user->permanent_address,
                'current_address' => $user->current_address,
                'guardian_name' => $user->guardian_name,
                'custom_field_1' => $user->custom_field_1,
                'custom_field_2' => $user->custom_field_2,
                'custom_field_3' => $user->custom_field_3,
                'custom_field_4' => $user->custom_field_4,
                'id_proof_name' => $user->id_proof_name,
                'id_proof_number' => $user->id_proof_number,
                'cmmsn_percent' => $user->cmmsn_percent,
                'max_sales_discount_percent' => $user->max_sales_discount_percent,
                'selected_contacts' => !empty($contact_access) ? 1 : 0,
                'contact_access' => $contact_access,
                'permitted_locations' => $permitted_locations,
                'current_role_id' => $user->roles->first()->id ?? null,
                'bank_details' => $user->bank_details ? json_decode($user->bank_details, true) : null
            ];

            $data = [
                'user' => $user_data,
                'roles' => $roles,
                'contacts' => $contacts,
                'locations' => $locations,
                'username_extension' => $username_ext,
                'form_partials' => $form_partials
            ];

            return $this->sendResponse($data, 'User edit data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function update(Request $request, $id): JsonResponse
    {
        try {
            if (!auth()->user()->can('user.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            // Validation
            $validator = Validator::make($request->all(), [
                'surname' => 'nullable|string|max:255',
                'first_name' => 'required|string|max:255',
                'last_name' => 'nullable|string|max:255',
                'username' => 'nullable|string|max:255|unique:users,username,' . $id,
                'email' => 'nullable|email|unique:users,email,' . $id,
                'password' => 'nullable|string|min:6',
                'role' => 'required|exists:roles,id',
                'contact_number' => 'nullable|string|max:20',
                'allow_login' => 'boolean',
                'is_active' => 'boolean',
                'selected_contacts' => 'boolean',
                'selected_contact_ids' => 'array',
                'location_permissions' => 'array',
                'access_all_locations' => 'nullable|string',
                'dob' => 'nullable|date',
                'cmmsn_percent' => 'nullable|numeric|min:0|max:100',
                'max_sales_discount_percent' => 'nullable|numeric|min:0|max:100'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $user_data = $request->only([
                'surname',
                'first_name',
                'last_name',
                'email',
                'selected_contacts',
                'marital_status',
                'blood_group',
                'contact_number',
                'fb_link',
                'twitter_link',
                'social_media_1',
                'social_media_2',
                'permanent_address',
                'current_address',
                'guardian_name',
                'custom_field_1',
                'custom_field_2',
                'custom_field_3',
                'custom_field_4',
                'id_proof_name',
                'id_proof_number',
                'cmmsn_percent',
                'gender',
                'max_sales_discount_percent'
            ]);

            $user_data['status'] = $request->input('is_active', true) ? 'active' : 'inactive';

            if (!isset($user_data['selected_contacts'])) {
                $user_data['selected_contacts'] = 0;
            }

            // Handle login permissions
            if (!$request->input('allow_login', false)) {
                $user_data['username'] = null;
                $user_data['password'] = null;
                $user_data['allow_login'] = 0;
            } else {
                $user_data['allow_login'] = 1;

                if ($request->has('username')) {
                    $user_data['username'] = $request->input('username');
                    if (empty($user_data['username'])) {
                        $ref_count = $this->moduleUtil->setAndGetReferenceCount('username');
                        $user_data['username'] = $this->moduleUtil->generateReferenceNumber('username', $ref_count);
                    }

                    $username_ext = $this->getUsernameExtension();
                    if (!empty($username_ext)) {
                        $user_data['username'] .= $username_ext;
                    }
                }
            }

            // Handle password update
            if (!empty($request->input('password'))) {
                $user_data['password'] = $user_data['allow_login'] == 1
                    ? Hash::make($request->input('password')) : null;
            }

            // Handle date of birth
            if (!empty($request->input('dob'))) {
                $user_data['dob'] = Carbon::createFromFormat('Y-m-d', $request->input('dob'));
            }

            // Handle bank details
            if (!empty($request->input('bank_details'))) {
                $user_data['bank_details'] = json_encode($request->input('bank_details'));
            }

            // Handle commission percentage
            $user_data['cmmsn_percent'] = !empty($user_data['cmmsn_percent'])
                ? $this->moduleUtil->num_uf($user_data['cmmsn_percent']) : 0;

            $user_data['max_sales_discount_percent'] = !is_null($user_data['max_sales_discount_percent'])
                ? $this->moduleUtil->num_uf($user_data['max_sales_discount_percent']) : null;

            $user = User::where('business_id', $business_id)->findOrFail($id);
            $user->update($user_data);

            // Update role
            $role_id = $request->input('role');
            $user_role = $user->roles->first();
            $previous_role = !empty($user_role->id) ? $user_role->id : 0;

            if ($previous_role != $role_id) {
                if (!empty($previous_role)) {
                    $user->removeRole($user_role->name);
                }

                $role = Role::findOrFail($role_id);
                $user->assignRole($role->name);
            }

            // Grant location permissions
            $this->giveLocationPermissions($user, $request);

            // Assign selected contacts
            $contact_ids = $user_data['selected_contacts'] == 1
                ? $request->get('selected_contact_ids', []) : [];
            $user->contactAccess()->sync($contact_ids);

            // Update module fields for user
            $this->moduleUtil->getModuleData('afterModelSaved', [
                'event' => 'user_saved',
                'model_instance' => $user
            ]);

            $user_response = [
                'id' => $user->id,
                'full_name' => trim($user->surname . ' ' . $user->first_name . ' ' . $user->last_name),
                'username' => $user->username,
                'email' => $user->email,
                'status' => $user->status,
                'allow_login' => $user->allow_login
            ];

            return $this->sendResponse($user_response, 'User updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @param Request $request
     * @return JsonResponse
     */
    public function destroy($id, Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('user.delete')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $user = User::where('business_id', $business_id)->where('id', $id)->first();

            if (!$user) {
                return $this->sendError('User not found.', [], 404);
            }

            $user->delete();

            return $this->sendResponse([], 'User deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Get username extension based on business settings
     *
     * @return string|null
     */
    private function getUsernameExtension()
    {
        $extension = !empty(System::getProperty('enable_business_based_username'))
            ? '-' . str_pad(auth()->user()->business_id, 2, 0, STR_PAD_LEFT) : null;
        return $extension;
    }

    /**
     * Retrieves roles array (Hides admin role from non admin users)
     *
     * @param int $business_id
     * @return array $roles
     */
    private function getRolesArray($business_id)
    {
        $roles_array = Role::where('business_id', $business_id)->get()->pluck('name', 'id');
        $roles = [];

        $is_admin = $this->moduleUtil->is_admin(auth()->user(), $business_id);

        foreach ($roles_array as $key => $value) {
            if (!$is_admin && $value == 'Admin#' . $business_id) {
                continue;
            }
            $roles[$key] = str_replace('#' . $business_id, '', $value);
        }
        return $roles;
    }

    /**
     * Adds or updates location permissions of a user
     *
     * @param User $user
     * @param Request $request
     * @return void
     */
    private function giveLocationPermissions($user, $request)
    {
        $permitted_locations = $user->permitted_locations();
        $permissions = $request->input('access_all_locations');
        $revoked_permissions = [];

        // If not access all location then revoke permission
        if ($permitted_locations == 'all' && $permissions != 'access_all_locations') {
            $user->revokePermissionTo('access_all_locations');
        }

        // Include location permissions
        $location_permissions = $request->input('location_permissions', []);
        if (empty($permissions) && !empty($location_permissions)) {
            $permissions = [];
            foreach ($location_permissions as $location_permission) {
                $permissions[] = $location_permission;
            }

            if (is_array($permitted_locations)) {
                foreach ($permitted_locations as $key => $value) {
                    if (!in_array('location.' . $value, $permissions)) {
                        $revoked_permissions[] = 'location.' . $value;
                    }
                }
            }
        }

        if (!empty($revoked_permissions)) {
            $user->revokePermissionTo($revoked_permissions);
        }

        if (!empty($permissions)) {
            $user->givePermissionTo($permissions);
        } else {
            // if no location permission given revoke previous permissions
            if (!empty($permitted_locations)) {
                $revoke_permissions = [];
                foreach ($permitted_locations as $key => $value) {
                    $revoke_permissions[] = 'location.' . $value;
                }

                if (!empty($revoke_permissions)) {
                    $user->revokePermissionTo($revoke_permissions);
                }
            }
        }
    }

    /**
     * Get users dropdown for business
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getUsersDropdown(Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('user.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id ?? auth()->user()->business_id;

            $users = User::forDropdown($business_id, false);

            return $this->sendResponse($users, 'Users dropdown retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }
}
