<?php

namespace App\Http\Controllers;


use App\Models\BusinessLocation;
use App\Http\Controllers\Api\BaseController as ApiBaseController;
use App\Models\Printer;
use App\Models\InvoiceLayout;
use App\Models\InvoiceScheme;
use Illuminate\Http\Request;
use Exception;

/**
 * @group Location Settings Management
 * 
 * APIs pour gérer les paramètres de localisation des points de vente
 */

class LocationSettingsController extends ApiBaseController
{
    protected $printReceiptOnInvoice;
    protected $receiptPrinterType;

    public function __construct()
    {
        $this->printReceiptOnInvoice = ['1' => 'Yes', '0' => 'No'];
        $this->receiptPrinterType = ['browser' => 'Browser Based Printing', 'printer' => 'Use Configured Receipt Printer'];
    }

    /**
     * Récupérer tous les paramètres de localisation
     * 
     * @response 200 {
     *   "success": true,
     *   "data": {
     *     "locations": [...],
     *     "printers": [...],
     *     "invoice_layouts": [...],
     *     "invoice_schemes": [...]
     *   },
     *   "message": "Location settings retrieved successfully"
     * }
     */
    public function index(Request $request)
    {
        try {
            if (!auth()->user()->can('business_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $locations = BusinessLocation::where('business_id', $business_id)->get();
            $printers = Printer::forDropdown($business_id);
            $invoice_layouts = InvoiceLayout::where('business_id', $business_id)->get()->pluck('name', 'id');
            $invoice_schemes = InvoiceScheme::where('business_id', $business_id)->get()->pluck('name', 'id');

            $data = [
                'locations' => $locations,
                'printers' => $printers,
                'invoice_layouts' => $invoice_layouts,
                'invoice_schemes' => $invoice_schemes,
                'print_receipt_options' => $this->printReceiptOnInvoice,
                'receipt_printer_types' => $this->receiptPrinterType
            ];

            return $this->sendResponse($data, 'Location settings retrieved successfully');
        } catch (Exception $e) {
            return $this->sendError('Something went wrong', [], 500);
        }
    }

    /**
     * Récupérer les paramètres d'une localisation spécifique
     * 
     * @urlParam location_id int required L'ID de la localisation
     */
    public function show($location_id)
    {
        try {
            if (
                !auth()->user()->can('business_settings.access') ||
                !auth()->user()->can_access_this_location($location_id)
            ) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $location = BusinessLocation::where('business_id', $business_id)->findOrFail($location_id);
            $printers = Printer::forDropdown($business_id);
            $invoice_layouts = InvoiceLayout::where('business_id', $business_id)->get()->pluck('name', 'id');
            $invoice_schemes = InvoiceScheme::where('business_id', $business_id)->get()->pluck('name', 'id');

            $data = [
                'location' => $location,
                'printers' => $printers,
                'invoice_layouts' => $invoice_layouts,
                'invoice_schemes' => $invoice_schemes,
                'print_receipt_options' => $this->printReceiptOnInvoice,
                'receipt_printer_types' => $this->receiptPrinterType
            ];

            return $this->sendResponse($data, 'Location settings retrieved successfully');
        } catch (Exception $e) {
            return $this->sendError('Location not found', [], 404);
        }
    }

    /**
     * Créer une nouvelle localisation
     * 
     * @bodyParam name string required Nom de la localisation
     * @bodyParam landmark string Repère géographique
     * @bodyParam country string required Pays
     * @bodyParam state string required État/Province
     * @bodyParam city string required Ville
     * @bodyParam zip_code string Code postal
     * @bodyParam print_receipt_on_invoice string Options: 0,1
     * @bodyParam receipt_printer_type string Options: browser,printer
     * @bodyParam printer_id int ID de l'imprimante
     * @bodyParam invoice_layout_id int ID du layout de facture
     * @bodyParam invoice_scheme_id int ID du schéma de facture
     */
    public function store(Request $request)
    {
        try {
            if (!auth()->user()->can('business_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'landmark' => 'nullable|string|max:255',
                'country' => 'required|string|max:255',
                'state' => 'required|string|max:255',
                'city' => 'required|string|max:255',
                'zip_code' => 'nullable|string|max:20',
                'print_receipt_on_invoice' => 'nullable|in:0,1',
                'receipt_printer_type' => 'nullable|in:browser,printer',
                'printer_id' => 'nullable|integer|exists:printers,id',
                'invoice_layout_id' => 'nullable|integer|exists:invoice_layouts,id',
                'invoice_scheme_id' => 'nullable|integer|exists:invoice_schemes,id'
            ]);

            $business_id = auth()->user()->business_id;
            $validatedData['business_id'] = $business_id;

            if (config('app.env') == 'demo') {
                $validatedData['receipt_printer_type'] = 'browser';
            }

            $location = BusinessLocation::create($validatedData);

            return $this->sendResponse($location, 'Location created successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to create location', [], 500);
        }
    }

    /**
     * Mettre à jour les paramètres d'une localisation
     * 
     * @urlParam location_id int required L'ID de la localisation
     * @bodyParam print_receipt_on_invoice string Options: 0,1
     * @bodyParam receipt_printer_type string Options: browser,printer
     * @bodyParam printer_id int ID de l'imprimante
     * @bodyParam invoice_layout_id int ID du layout de facture
     * @bodyParam invoice_scheme_id int ID du schéma de facture
     */
    public function update($location_id, Request $request)
    {
        try {
            if (
                !auth()->user()->can('business_settings.access') ||
                !auth()->user()->can_access_this_location($location_id)
            ) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $input = $request->only([
                'name',
                'landmark',
                'country',
                'state',
                'city',
                'zip_code',
                'print_receipt_on_invoice',
                'receipt_printer_type',
                'printer_id',
                'invoice_layout_id',
                'invoice_scheme_id'
            ]);

            if (config('app.env') == 'demo') {
                $input['receipt_printer_type'] = 'browser';
            }

            $business_id = auth()->user()->business_id;
            $location = BusinessLocation::where('business_id', $business_id)->findOrFail($location_id);
            $location->fill($input);
            $location->update();

            return $this->sendResponse($location, 'Location settings updated successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to update location settings', [], 500);
        }
    }

    /**
     * Supprimer une localisation
     * 
     * @urlParam location_id int required L'ID de la localisation
     */
    public function destroy($location_id)
    {
        try {
            if (
                !auth()->user()->can('business_settings.access') ||
                !auth()->user()->can_access_this_location($location_id)
            ) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $location = BusinessLocation::where('business_id', $business_id)->findOrFail($location_id);
            $location->delete();

            return $this->sendResponse([], 'Location deleted successfully');
        } catch (Exception $e) {
            return $this->sendError('Failed to delete location', [], 500);
        }
    }
}
