<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Barcode;
use App\Models\Product;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class LabelsController extends BaseController
{
    /**
     * All Utils instance.
     */
    protected $transactionUtil;
    protected $productUtil;

    /**
     * Constructor
     *
     * @param TransactionUtil $transactionUtil
     * @param ProductUtil $productUtil
     * @return void
     */
    public function __construct(TransactionUtil $transactionUtil, ProductUtil $productUtil)
    {
        $this->transactionUtil = $transactionUtil;
        $this->productUtil = $productUtil;
    }

    /**
     * Display a listing of labels.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $business_id = Auth::user()->business_id;
            $purchase_id = $request->get('purchase_id', false);
            $product_id = $request->get('product_id', false);

            // Get products for the business
            $products = [];
            if ($purchase_id) {
                $products = $this->transactionUtil->getPurchaseProducts($business_id, $purchase_id);
            } elseif ($product_id) {
                $products = $this->productUtil->getDetailsFromProduct($business_id, $product_id);
            }

            $barcode_settings = Barcode::where('business_id', $business_id)
                ->orWhereNull('business_id')
                ->pluck('name', 'id');

            $data = [
                'products' => $products,
                'barcode_settings' => $barcode_settings
            ];

            return $this->sendResponse($data, 'Labels retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving labels', [], 500);
        }
    }

    /**
     * Show the form for creating a new label.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create()
    {
        try {
            $business_id = Auth::user()->business_id;

            $barcode_settings = Barcode::where('business_id', $business_id)
                ->orWhereNull('business_id')
                ->get(['id', 'name']);

            $data = [
                'barcode_settings' => $barcode_settings
            ];

            return $this->sendResponse($data, 'Create label form data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving create form data', [], 500);
        }
    }

    /**
     * Store a newly created label in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'products' => 'required|array',
                'barcode_setting' => 'required|exists:barcodes,id'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $business_id = Auth::user()->business_id;
            $products = $request->get('products');
            $barcode_setting = $request->get('barcode_setting');

            // Process and store label data
            $product_details = [];
            $total_qty = 0;
            foreach ($products as $value) {
                $details = $this->productUtil->getDetailsFromVariation($value['variation_id'], $business_id, null, false);
                $product_details[] = ['details' => $details, 'qty' => $value['quantity']];
                $total_qty += $value['quantity'];
            }

            $data = [
                'product_details' => $product_details,
                'total_quantity' => $total_qty,
                'barcode_setting_id' => $barcode_setting
            ];

            return $this->sendResponse($data, 'Label created successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error creating label', [], 500);
        }
    }

    /**
     * Display the specified label.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request)
    {
        try {
            $business_id = Auth::user()->business_id;
            $purchase_id = $request->get('purchase_id', false);
            $product_id = $request->get('product_id', false);

            // Get products for the business
            $products = [];
            if ($purchase_id) {
                $products = $this->transactionUtil->getPurchaseProducts($business_id, $purchase_id);
            } elseif ($product_id) {
                $products = $this->productUtil->getDetailsFromProduct($business_id, $product_id);
            }

            $barcode_settings = Barcode::where('business_id', $business_id)
                ->orWhereNull('business_id')
                ->pluck('name', 'id');

            $data = [
                'products' => $products,
                'barcode_settings' => $barcode_settings
            ];

            return $this->sendResponse($data, 'Label details retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving label details', [], 500);
        }
    }

    /**
     * Show the form for editing the specified label.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        try {
            $business_id = Auth::user()->business_id;

            $barcode_settings = Barcode::where('business_id', $business_id)
                ->orWhereNull('business_id')
                ->get(['id', 'name']);

            $data = [
                'label_id' => $id,
                'barcode_settings' => $barcode_settings
            ];

            return $this->sendResponse($data, 'Edit label form data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving edit form data', [], 500);
        }
    }

    /**
     * Update the specified label in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'products' => 'required|array',
                'barcode_setting' => 'required|exists:barcodes,id'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $business_id = Auth::user()->business_id;
            $products = $request->get('products');
            $barcode_setting = $request->get('barcode_setting');

            // Process and update label data
            $product_details = [];
            $total_qty = 0;
            foreach ($products as $value) {
                $details = $this->productUtil->getDetailsFromVariation($value['variation_id'], $business_id, null, false);
                $product_details[] = ['details' => $details, 'qty' => $value['quantity']];
                $total_qty += $value['quantity'];
            }

            $data = [
                'label_id' => $id,
                'product_details' => $product_details,
                'total_quantity' => $total_qty,
                'barcode_setting_id' => $barcode_setting
            ];

            return $this->sendResponse($data, 'Label updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error updating label', [], 500);
        }
    }

    /**
     * Remove the specified label from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            // Logic to delete label if needed
            return $this->sendResponse([], 'Label deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error deleting label', [], 500);
        }
    }

    /**
     * Add product row for labels
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addProductRow(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'product_id' => 'required|exists:products,id',
                'variation_id' => 'nullable|exists:variations,id',
                'row_count' => 'required|integer'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $product_id = $request->input('product_id');
            $variation_id = $request->input('variation_id');
            $business_id = Auth::user()->business_id;
            $index = $request->input('row_count');

            $products = $this->productUtil->getDetailsFromProduct($business_id, $product_id, $variation_id);

            $data = [
                'products' => $products,
                'index' => $index
            ];

            return $this->sendResponse($data, 'Product row added successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error adding product row', [], 500);
        }
    }

    /**
     * Generate labels preview
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function preview(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'products' => 'required|array',
                'barcode_setting' => 'required|exists:barcodes,id'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $products = $request->get('products');
            $print = $request->get('print', false);
            $barcode_setting = $request->get('barcode_setting');
            $business_id = Auth::user()->business_id;

            $barcode_details = Barcode::find($barcode_setting);
            $business_name = Auth::user()->business->name;

            $product_details = [];
            $total_qty = 0;
            foreach ($products as $value) {
                $details = $this->productUtil->getDetailsFromVariation($value['variation_id'], $business_id, null, false);
                $product_details[] = ['details' => $details, 'qty' => $value['quantity']];
                $total_qty += $value['quantity'];
            }

            $page_height = null;
            if ($barcode_details->is_continuous) {
                $rows = ceil($total_qty / $barcode_details->stickers_in_one_row) + 0.4;
                $barcode_details->paper_height = $barcode_details->top_margin + ($rows * $barcode_details->height) + ($rows * $barcode_details->row_distance);
            }

            $data = [
                'product_details' => $product_details,
                'business_name' => $business_name,
                'barcode_details' => $barcode_details,
                'page_height' => $page_height,
                'total_quantity' => $total_qty,
                'print' => $print
            ];

            return $this->sendResponse($data, 'Labels preview generated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error generating labels preview', [], 500);
        }
    }
}
