<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\InvoiceScheme;
use App\Models\InvoiceLayout;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class InvoiceSchemeController extends BaseController
{
    /**
     * Display a listing of the invoice schemes.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = Auth::user()->business_id;

            $schemes = InvoiceScheme::where('business_id', $business_id)
                ->select(['id', 'name', 'scheme_type', 'prefix', 'start_number', 'invoice_count', 'total_digits', 'is_default'])
                ->get();

            // Transform data for API response
            $schemes_data = $schemes->map(function ($scheme) {
                return [
                    'id' => $scheme->id,
                    'name' => $scheme->name,
                    'scheme_type' => $scheme->scheme_type,
                    'prefix' => $scheme->scheme_type == 'year' ? date('Y') . '-' : $scheme->prefix,
                    'start_number' => $scheme->start_number,
                    'invoice_count' => $scheme->invoice_count,
                    'total_digits' => $scheme->total_digits,
                    'is_default' => $scheme->is_default,
                    'display_name' => $scheme->is_default ? $scheme->name . ' (Default)' : $scheme->name
                ];
            });

            $invoice_layouts = InvoiceLayout::where('business_id', $business_id)
                ->with(['locations'])
                ->get();

            $data = [
                'schemes' => $schemes_data,
                'invoice_layouts' => $invoice_layouts
            ];

            return $this->sendResponse($data, 'Invoice schemes retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving invoice schemes', [], 500);
        }
    }

    /**
     * Show the form for creating a new invoice scheme.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create()
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $data = [
                'scheme_types' => [
                    'blank' => 'Blank',
                    'year' => 'Year'
                ]
            ];

            return $this->sendResponse($data, 'Create invoice scheme form data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving create form data', [], 500);
        }
    }

    /**
     * Store a newly created invoice scheme in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'scheme_type' => 'required|in:blank,year',
                'prefix' => 'nullable|string|max:10',
                'start_number' => 'required|integer|min:1',
                'total_digits' => 'required|integer|min:1|max:10',
                'is_default' => 'boolean'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $input = $request->only(['name', 'scheme_type', 'prefix', 'start_number', 'total_digits']);
            $business_id = Auth::user()->business_id;
            $input['business_id'] = $business_id;
            $input['invoice_count'] = 0;

            if (!empty($request->input('is_default'))) {
                // Remove default from existing schemes
                InvoiceScheme::where('business_id', $business_id)
                    ->where('is_default', 1)
                    ->update(['is_default' => 0]);
                $input['is_default'] = 1;
            } else {
                $input['is_default'] = 0;
            }

            $invoice_scheme = InvoiceScheme::create($input);

            return $this->sendResponse($invoice_scheme, 'Invoice scheme created successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error creating invoice scheme', [], 500);
        }
    }

    /**
     * Display the specified invoice scheme.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = Auth::user()->business_id;
            $invoice_scheme = InvoiceScheme::where('business_id', $business_id)->find($id);

            if (!$invoice_scheme) {
                return $this->sendError('Invoice scheme not found.', [], 404);
            }

            // Transform data for API response
            $scheme_data = [
                'id' => $invoice_scheme->id,
                'name' => $invoice_scheme->name,
                'scheme_type' => $invoice_scheme->scheme_type,
                'prefix' => $invoice_scheme->scheme_type == 'year' ? date('Y') . '-' : $invoice_scheme->prefix,
                'start_number' => $invoice_scheme->start_number,
                'invoice_count' => $invoice_scheme->invoice_count,
                'total_digits' => $invoice_scheme->total_digits,
                'is_default' => $invoice_scheme->is_default,
                'display_name' => $invoice_scheme->is_default ? $invoice_scheme->name . ' (Default)' : $invoice_scheme->name
            ];

            return $this->sendResponse($scheme_data, 'Invoice scheme details retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving invoice scheme details', [], 500);
        }
    }

    /**
     * Show the form for editing the specified invoice scheme.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = Auth::user()->business_id;
            $invoice_scheme = InvoiceScheme::where('business_id', $business_id)->find($id);

            if (!$invoice_scheme) {
                return $this->sendError('Invoice scheme not found.', [], 404);
            }

            $data = [
                'invoice_scheme' => $invoice_scheme,
                'scheme_types' => [
                    'blank' => 'Blank',
                    'year' => 'Year'
                ]
            ];

            return $this->sendResponse($data, 'Edit invoice scheme form data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving edit form data', [], 500);
        }
    }

    /**
     * Update the specified invoice scheme in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'scheme_type' => 'required|in:blank,year',
                'prefix' => 'nullable|string|max:10',
                'start_number' => 'required|integer|min:1',
                'total_digits' => 'required|integer|min:1|max:10'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $business_id = Auth::user()->business_id;
            $invoice_scheme = InvoiceScheme::where('business_id', $business_id)->find($id);

            if (!$invoice_scheme) {
                return $this->sendError('Invoice scheme not found.', [], 404);
            }

            $input = $request->only(['name', 'scheme_type', 'prefix', 'start_number', 'total_digits']);
            $invoice_scheme->update($input);

            return $this->sendResponse($invoice_scheme->fresh(), 'Invoice scheme updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error updating invoice scheme', [], 500);
        }
    }

    /**
     * Remove the specified invoice scheme from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = Auth::user()->business_id;
            $invoice_scheme = InvoiceScheme::where('business_id', $business_id)->find($id);

            if (!$invoice_scheme) {
                return $this->sendError('Invoice scheme not found.', [], 404);
            }

            if ($invoice_scheme->is_default == 1) {
                return $this->sendError('Cannot delete default invoice scheme.', [], 422);
            }

            $invoice_scheme->delete();

            return $this->sendResponse([], 'Invoice scheme deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error deleting invoice scheme', [], 500);
        }
    }

    /**
     * Set invoice scheme as default
     *
     * @param  int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function setDefault($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = Auth::user()->business_id;
            $invoice_scheme = InvoiceScheme::where('business_id', $business_id)->find($id);

            if (!$invoice_scheme) {
                return $this->sendError('Invoice scheme not found.', [], 404);
            }

            // Remove default from all schemes
            InvoiceScheme::where('business_id', $business_id)
                ->where('is_default', 1)
                ->update(['is_default' => 0]);

            // Set current scheme as default
            $invoice_scheme->is_default = 1;
            $invoice_scheme->save();

            return $this->sendResponse($invoice_scheme->fresh(), 'Default invoice scheme set successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error setting default invoice scheme', [], 500);
        }
    }

    /**
     * Get default invoice scheme
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDefault()
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = Auth::user()->business_id;
            $default_scheme = InvoiceScheme::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();

            if (!$default_scheme) {
                return $this->sendError('No default invoice scheme found.', [], 404);
            }

            return $this->sendResponse($default_scheme, 'Default invoice scheme retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error retrieving default invoice scheme', [], 500);
        }
    }
}
