<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\InvoiceLayout;
use App\Utils\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class InvoiceLayoutController extends BaseController
{
    protected $designs;
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;

        $this->designs = [
            'classic' => 'Classic (For normal printer)',
            'elegant' => 'Elegant (For normal printer)',
            'detailed' => 'Detailed (For normal printer)',
            'columnize-taxes' => 'Columnize Taxes (For normal printer)',
            'slim' => 'Slim (For thermal line receipt printer)'
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $query = InvoiceLayout::where('business_id', $business_id);

            // Search functionality
            if ($request->has('search') && !empty($request->search)) {
                $search = $request->search;
                $query->where('name', 'like', "%{$search}%");
            }

            // Filter by design
            if ($request->has('design') && !empty($request->design)) {
                $query->where('design', $request->design);
            }

            // Filter by default status
            if ($request->has('is_default')) {
                $query->where('is_default', $request->is_default);
            }

            $perPage = $request->get('per_page', 10);
            $layouts = $query->orderBy('created_at', 'desc')->paginate($perPage);

            // Decode JSON fields for each layout
            $layouts->getCollection()->transform(function ($layout) {
                $layout->module_info = json_decode($layout->module_info, true);
                $layout->table_tax_headings = !empty($layout->table_tax_headings) ?
                    json_decode($layout->table_tax_headings, true) : ['', '', '', ''];
                $layout->product_custom_fields = !empty($layout->product_custom_fields) ?
                    json_decode($layout->product_custom_fields, true) : null;
                $layout->contact_custom_fields = !empty($layout->contact_custom_fields) ?
                    json_decode($layout->contact_custom_fields, true) : null;
                $layout->location_custom_fields = !empty($layout->location_custom_fields) ?
                    json_decode($layout->location_custom_fields, true) : null;
                $layout->common_settings = !empty($layout->common_settings) ?
                    json_decode($layout->common_settings, true) : null;
                return $layout;
            });

            return $this->sendResponse($layouts, 'Layouts de facture récupérés avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération des layouts', [$e->getMessage()], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create()
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $designs = $this->designs;
            $common_settings = session()->get('business.common_settings');
            $is_warranty_enabled = !empty($common_settings['enable_product_warranty']) ? true : false;

            // Get available checkboxes options
            $checkboxes = [
                'show_business_name' => 'Afficher le nom de l\'entreprise',
                'show_location_name' => 'Afficher le nom de la location',
                'show_landmark' => 'Afficher le point de repère',
                'show_city' => 'Afficher la ville',
                'show_state' => 'Afficher l\'état/région',
                'show_country' => 'Afficher le pays',
                'show_zip_code' => 'Afficher le code postal',
                'show_mobile_number' => 'Afficher le numéro de mobile',
                'show_alternate_number' => 'Afficher le numéro alternatif',
                'show_email' => 'Afficher l\'email',
                'show_tax_1' => 'Afficher la taxe 1',
                'show_tax_2' => 'Afficher la taxe 2',
                'show_logo' => 'Afficher le logo',
                'show_barcode' => 'Afficher le code-barres',
                'show_payments' => 'Afficher les paiements',
                'show_customer' => 'Afficher le client',
                'show_client_id' => 'Afficher l\'ID client',
                'show_brand' => 'Afficher la marque',
                'show_sku' => 'Afficher le SKU',
                'show_cat_code' => 'Afficher le code catégorie',
                'show_sale_description' => 'Afficher la description de vente',
                'show_sales_person' => 'Afficher le vendeur',
                'show_expiry' => 'Afficher la date d\'expiration',
                'show_lot' => 'Afficher le lot',
                'show_previous_bal' => 'Afficher le solde précédent',
                'show_image' => 'Afficher l\'image',
                'show_reward_point' => 'Afficher les points de récompense'
            ];

            $data = [
                'designs' => $designs,
                'is_warranty_enabled' => $is_warranty_enabled,
                'available_checkboxes' => $checkboxes
            ];

            return $this->sendResponse($data, 'Données pour création de layout récupérées avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération des données', [$e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'design' => 'required|string',
                'logo' => 'nullable|mimes:jpeg,gif,png|max:1000',
                'header_text' => 'nullable|string',
                'footer_text' => 'nullable|string',
                'highlight_color' => 'nullable|string|max:7', // Format hexadécimal
            ]);

            if ($validator->fails()) {
                return $this->sendError('Données de validation incorrectes', $validator->errors(), 422);
            }

            $input = $request->only([
                'name',
                'header_text',
                'invoice_no_prefix',
                'invoice_heading',
                'sub_total_label',
                'discount_label',
                'tax_label',
                'total_label',
                'highlight_color',
                'footer_text',
                'invoice_heading_not_paid',
                'invoice_heading_paid',
                'total_due_label',
                'customer_label',
                'paid_label',
                'sub_heading_line1',
                'sub_heading_line2',
                'sub_heading_line3',
                'sub_heading_line4',
                'sub_heading_line5',
                'table_product_label',
                'table_qty_label',
                'table_unit_price_label',
                'table_subtotal_label',
                'client_id_label',
                'date_label',
                'quotation_heading',
                'quotation_no_prefix',
                'design',
                'client_tax_label',
                'cat_code_label',
                'cn_heading',
                'cn_no_label',
                'cn_amount_label',
                'sales_person_label',
                'prev_bal_label',
                'date_time_format',
                'common_settings',
                'change_return_label',
                'round_off_label'
            ]);

            $business_id = auth()->user()->business_id;
            $input['business_id'] = $business_id;

            // Set value for checkboxes
            $checkboxes = [
                'show_business_name',
                'show_location_name',
                'show_landmark',
                'show_city',
                'show_state',
                'show_country',
                'show_zip_code',
                'show_mobile_number',
                'show_alternate_number',
                'show_email',
                'show_tax_1',
                'show_tax_2',
                'show_logo',
                'show_barcode',
                'show_payments',
                'show_customer',
                'show_client_id',
                'show_brand',
                'show_sku',
                'show_cat_code',
                'show_sale_description',
                'show_sales_person',
                'show_expiry',
                'show_lot',
                'show_previous_bal',
                'show_image',
                'show_reward_point'
            ];

            foreach ($checkboxes as $name) {
                $input[$name] = !empty($request->input($name)) ? 1 : 0;
            }

            // Upload Logo
            if ($request->hasFile('logo')) {
                $logo_name = $this->commonUtil->uploadFile($request, 'logo', 'invoice_logos', 'image');
                if (!empty($logo_name)) {
                    $input['logo'] = $logo_name;
                }
            }

            if (!empty($request->input('is_default'))) {
                // Reset other default layouts
                InvoiceLayout::where('business_id', $business_id)
                    ->where('is_default', 1)
                    ->update(['is_default' => 0]);
                $input['is_default'] = 1;
            } else {
                $input['is_default'] = 0;
            }

            // Module info
            if ($request->has('module_info')) {
                $input['module_info'] = json_encode($request->input('module_info'));
            }

            if (!empty($request->input('table_tax_headings'))) {
                $input['table_tax_headings'] = json_encode($request->input('table_tax_headings'));
            }

            $input['product_custom_fields'] = !empty($request->input('product_custom_fields')) ?
                json_encode($request->input('product_custom_fields')) : null;
            $input['contact_custom_fields'] = !empty($request->input('contact_custom_fields')) ?
                json_encode($request->input('contact_custom_fields')) : null;
            $input['location_custom_fields'] = !empty($request->input('location_custom_fields')) ?
                json_encode($request->input('location_custom_fields')) : null;

            $layout = InvoiceLayout::create($input);

            // Decode JSON fields for response
            $layout->module_info = json_decode($layout->module_info, true);
            $layout->table_tax_headings = !empty($layout->table_tax_headings) ?
                json_decode($layout->table_tax_headings, true) : ['', '', '', ''];
            $layout->product_custom_fields = !empty($layout->product_custom_fields) ?
                json_decode($layout->product_custom_fields, true) : null;
            $layout->contact_custom_fields = !empty($layout->contact_custom_fields) ?
                json_decode($layout->contact_custom_fields, true) : null;
            $layout->location_custom_fields = !empty($layout->location_custom_fields) ?
                json_decode($layout->location_custom_fields, true) : null;

            return $this->sendResponse($layout, 'Layout de facture créé avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la création du layout', [$e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return $this->sendError('Layout non trouvé', [], 404);
            }

            // Decode JSON fields
            $layout->module_info = json_decode($layout->module_info, true);
            $layout->table_tax_headings = !empty($layout->table_tax_headings) ?
                json_decode($layout->table_tax_headings, true) : ['', '', '', ''];
            $layout->product_custom_fields = !empty($layout->product_custom_fields) ?
                json_decode($layout->product_custom_fields, true) : null;
            $layout->contact_custom_fields = !empty($layout->contact_custom_fields) ?
                json_decode($layout->contact_custom_fields, true) : null;
            $layout->location_custom_fields = !empty($layout->location_custom_fields) ?
                json_decode($layout->location_custom_fields, true) : null;
            $layout->common_settings = !empty($layout->common_settings) ?
                json_decode($layout->common_settings, true) : null;

            return $this->sendResponse($layout, 'Layout récupéré avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération du layout', [$e->getMessage()], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return $this->sendError('Layout non trouvé', [], 404);
            }

            // Decode JSON fields for editing
            $layout->module_info = json_decode($layout->module_info, true);
            $layout->table_tax_headings = !empty($layout->table_tax_headings) ?
                json_decode($layout->table_tax_headings, true) : ['', '', '', ''];
            $layout->product_custom_fields = !empty($layout->product_custom_fields) ?
                json_decode($layout->product_custom_fields, true) : null;
            $layout->contact_custom_fields = !empty($layout->contact_custom_fields) ?
                json_decode($layout->contact_custom_fields, true) : null;
            $layout->location_custom_fields = !empty($layout->location_custom_fields) ?
                json_decode($layout->location_custom_fields, true) : null;
            $layout->common_settings = !empty($layout->common_settings) ?
                json_decode($layout->common_settings, true) : null;

            // Get available checkboxes options
            $checkboxes = [
                'show_business_name' => 'Afficher le nom de l\'entreprise',
                'show_location_name' => 'Afficher le nom de la location',
                'show_landmark' => 'Afficher le point de repère',
                'show_city' => 'Afficher la ville',
                'show_state' => 'Afficher l\'état/région',
                'show_country' => 'Afficher le pays',
                'show_zip_code' => 'Afficher le code postal',
                'show_mobile_number' => 'Afficher le numéro de mobile',
                'show_alternate_number' => 'Afficher le numéro alternatif',
                'show_email' => 'Afficher l\'email',
                'show_tax_1' => 'Afficher la taxe 1',
                'show_tax_2' => 'Afficher la taxe 2',
                'show_logo' => 'Afficher le logo',
                'show_barcode' => 'Afficher le code-barres',
                'show_payments' => 'Afficher les paiements',
                'show_customer' => 'Afficher le client',
                'show_client_id' => 'Afficher l\'ID client',
                'show_brand' => 'Afficher la marque',
                'show_sku' => 'Afficher le SKU',
                'show_cat_code' => 'Afficher le code catégorie',
                'show_sale_description' => 'Afficher la description de vente',
                'show_sales_person' => 'Afficher le vendeur',
                'show_expiry' => 'Afficher la date d\'expiration',
                'show_lot' => 'Afficher le lot',
                'show_previous_bal' => 'Afficher le solde précédent',
                'show_image' => 'Afficher l\'image',
                'show_reward_point' => 'Afficher les points de récompense'
            ];

            $data = [
                'layout' => $layout,
                'designs' => $this->designs,
                'available_checkboxes' => $checkboxes
            ];

            return $this->sendResponse($data, 'Données pour édition récupérées avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération des données', [$e->getMessage()], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return $this->sendError('Layout non trouvé', [], 404);
            }

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'design' => 'required|string',
                'logo' => 'nullable|mimes:jpeg,gif,png|max:1000',
                'header_text' => 'nullable|string',
                'footer_text' => 'nullable|string',
                'highlight_color' => 'nullable|string|max:7', // Format hexadécimal
            ]);

            if ($validator->fails()) {
                return $this->sendError('Données de validation incorrectes', $validator->errors(), 422);
            }

            $input = $request->only([
                'name',
                'header_text',
                'invoice_no_prefix',
                'invoice_heading',
                'sub_total_label',
                'discount_label',
                'tax_label',
                'total_label',
                'highlight_color',
                'footer_text',
                'invoice_heading_not_paid',
                'invoice_heading_paid',
                'total_due_label',
                'customer_label',
                'paid_label',
                'sub_heading_line1',
                'sub_heading_line2',
                'sub_heading_line3',
                'sub_heading_line4',
                'sub_heading_line5',
                'table_product_label',
                'table_qty_label',
                'table_unit_price_label',
                'table_subtotal_label',
                'client_id_label',
                'date_label',
                'quotation_heading',
                'quotation_no_prefix',
                'design',
                'client_tax_label',
                'cat_code_label',
                'cn_heading',
                'cn_no_label',
                'cn_amount_label',
                'sales_person_label',
                'prev_bal_label',
                'date_time_format',
                'change_return_label',
                'round_off_label'
            ]);

            // Set value for checkboxes
            $checkboxes = [
                'show_business_name',
                'show_location_name',
                'show_landmark',
                'show_city',
                'show_state',
                'show_country',
                'show_zip_code',
                'show_mobile_number',
                'show_alternate_number',
                'show_email',
                'show_tax_1',
                'show_tax_2',
                'show_logo',
                'show_barcode',
                'show_payments',
                'show_customer',
                'show_client_id',
                'show_brand',
                'show_sku',
                'show_cat_code',
                'show_sale_description',
                'show_sales_person',
                'show_expiry',
                'show_lot',
                'show_previous_bal',
                'show_image',
                'show_reward_point'
            ];

            foreach ($checkboxes as $name) {
                $input[$name] = !empty($request->input($name)) ? 1 : 0;
            }

            // Upload Logo
            if ($request->hasFile('logo')) {
                $logo_name = $this->commonUtil->uploadFile($request, 'logo', 'invoice_logos', 'image');
                if (!empty($logo_name)) {
                    $input['logo'] = $logo_name;
                }
            }

            if (!empty($request->input('is_default'))) {
                // Reset other default layouts
                InvoiceLayout::where('business_id', $business_id)
                    ->where('is_default', 1)
                    ->where('id', '!=', $id)
                    ->update(['is_default' => 0]);
                $input['is_default'] = 1;
            }

            // Module info
            if ($request->has('module_info')) {
                $input['module_info'] = json_encode($request->input('module_info'));
            }

            if (!empty($request->input('table_tax_headings'))) {
                $input['table_tax_headings'] = json_encode($request->input('table_tax_headings'));
            }

            $input['product_custom_fields'] = !empty($request->input('product_custom_fields')) ?
                json_encode($request->input('product_custom_fields')) : null;
            $input['contact_custom_fields'] = !empty($request->input('contact_custom_fields')) ?
                json_encode($request->input('contact_custom_fields')) : null;
            $input['location_custom_fields'] = !empty($request->input('location_custom_fields')) ?
                json_encode($request->input('location_custom_fields')) : null;
            $input['common_settings'] = !empty($request->input('common_settings')) ?
                json_encode($request->input('common_settings')) : null;

            $layout->update($input);

            // Refresh model and decode JSON fields for response
            $layout->refresh();
            $layout->module_info = json_decode($layout->module_info, true);
            $layout->table_tax_headings = !empty($layout->table_tax_headings) ?
                json_decode($layout->table_tax_headings, true) : ['', '', '', ''];
            $layout->product_custom_fields = !empty($layout->product_custom_fields) ?
                json_decode($layout->product_custom_fields, true) : null;
            $layout->contact_custom_fields = !empty($layout->contact_custom_fields) ?
                json_decode($layout->contact_custom_fields, true) : null;
            $layout->location_custom_fields = !empty($layout->location_custom_fields) ?
                json_decode($layout->location_custom_fields, true) : null;
            $layout->common_settings = !empty($layout->common_settings) ?
                json_decode($layout->common_settings, true) : null;

            return $this->sendResponse($layout, 'Layout mis à jour avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la mise à jour du layout', [$e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return $this->sendError('Layout non trouvé', [], 404);
            }

            // Don't allow deletion of default layout
            if ($layout->is_default) {
                return $this->sendError('Impossible de supprimer le layout par défaut', [], 422);
            }

            // Delete logo file if exists
            if (!empty($layout->logo)) {
                $logoPath = public_path('storage/invoice_logos/' . $layout->logo);
                if (file_exists($logoPath)) {
                    unlink($logoPath);
                }
            }

            $layout->delete();

            return $this->sendResponse(null, 'Layout supprimé avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la suppression du layout', [$e->getMessage()], 500);
        }
    }

    /**
     * Set layout as default
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function setDefault($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return $this->sendError('Layout non trouvé', [], 404);
            }

            // Reset all other layouts
            InvoiceLayout::where('business_id', $business_id)
                ->update(['is_default' => 0]);

            // Set this layout as default
            $layout->update(['is_default' => 1]);

            return $this->sendResponse($layout, 'Layout défini par défaut avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la définition du layout par défaut', [$e->getMessage()], 500);
        }
    }

    /**
     * Get available designs
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDesigns()
    {
        try {
            return $this->sendResponse($this->designs, 'Designs disponibles récupérés avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération des designs', [$e->getMessage()], 500);
        }
    }

    /**
     * Duplicate a layout
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function duplicate($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $original = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$original) {
                return $this->sendError('Layout non trouvé', [], 404);
            }

            $duplicate = $original->replicate();
            $duplicate->name = $original->name . ' - Copie';
            $duplicate->is_default = 0;
            $duplicate->save();

            // Decode JSON fields for response
            $duplicate->module_info = json_decode($duplicate->module_info, true);
            $duplicate->table_tax_headings = !empty($duplicate->table_tax_headings) ?
                json_decode($duplicate->table_tax_headings, true) : ['', '', '', ''];
            $duplicate->product_custom_fields = !empty($duplicate->product_custom_fields) ?
                json_decode($duplicate->product_custom_fields, true) : null;
            $duplicate->contact_custom_fields = !empty($duplicate->contact_custom_fields) ?
                json_decode($duplicate->contact_custom_fields, true) : null;
            $duplicate->location_custom_fields = !empty($duplicate->location_custom_fields) ?
                json_decode($duplicate->location_custom_fields, true) : null;

            return $this->sendResponse($duplicate, 'Layout dupliqué avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la duplication du layout', [$e->getMessage()], 500);
        }
    }

    /**
     * Preview a layout
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function preview($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return $this->sendError('Layout non trouvé', [], 404);
            }

            // Decode JSON fields for preview
            $layout->module_info = json_decode($layout->module_info, true);
            $layout->table_tax_headings = !empty($layout->table_tax_headings) ?
                json_decode($layout->table_tax_headings, true) : ['', '', '', ''];
            $layout->product_custom_fields = !empty($layout->product_custom_fields) ?
                json_decode($layout->product_custom_fields, true) : null;
            $layout->contact_custom_fields = !empty($layout->contact_custom_fields) ?
                json_decode($layout->contact_custom_fields, true) : null;
            $layout->location_custom_fields = !empty($layout->location_custom_fields) ?
                json_decode($layout->location_custom_fields, true) : null;

            // Create sample data for preview
            $sampleData = [
                'business_name' => 'Mon Entreprise SARL',
                'business_address' => '123 Rue de la Paix, Douala, Cameroun',
                'invoice_number' => 'INV-2025-001',
                'date' => date('d/m/Y'),
                'customer_name' => 'Client Exemple',
                'customer_address' => '456 Avenue des Fleurs, Yaoundé',
                'items' => [
                    [
                        'product_name' => 'Produit A',
                        'quantity' => 2,
                        'unit_price' => 5000,
                        'total' => 10000
                    ],
                    [
                        'product_name' => 'Produit B',
                        'quantity' => 1,
                        'unit_price' => 15000,
                        'total' => 15000
                    ]
                ],
                'subtotal' => 25000,
                'tax' => 4750,
                'total' => 29750
            ];

            $previewData = [
                'layout' => $layout,
                'sample_data' => $sampleData,
                'preview_url' => route('api.invoice-layouts.generate-preview', $id)
            ];

            return $this->sendResponse($previewData, 'Prévisualisation générée avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la génération de la prévisualisation', [$e->getMessage()], 500);
        }
    }

    /**
     * Get layout statistics
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function statistics()
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $stats = [
                'total_layouts' => InvoiceLayout::where('business_id', $business_id)->count(),
                'default_layout' => InvoiceLayout::where('business_id', $business_id)
                    ->where('is_default', 1)
                    ->first(['id', 'name', 'design']),
                'layouts_by_design' => InvoiceLayout::where('business_id', $business_id)
                    ->select('design', DB::raw('count(*) as total'))
                    ->groupBy('design')
                    ->get(),
                'recently_created' => InvoiceLayout::where('business_id', $business_id)
                    ->orderBy('created_at', 'desc')
                    ->limit(5)
                    ->get(['id', 'name', 'design', 'created_at']),
                'most_used_checkboxes' => $this->getMostUsedCheckboxes($business_id)
            ];

            return $this->sendResponse($stats, 'Statistiques récupérées avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération des statistiques', [$e->getMessage()], 500);
        }
    }

    /**
     * Get most used checkboxes across all layouts
     *
     * @param int $business_id
     * @return array
     */
    private function getMostUsedCheckboxes($business_id)
    {
        $layouts = InvoiceLayout::where('business_id', $business_id)->get();

        $checkboxes = [
            'show_business_name',
            'show_location_name',
            'show_landmark',
            'show_city',
            'show_state',
            'show_country',
            'show_zip_code',
            'show_mobile_number',
            'show_alternate_number',
            'show_email',
            'show_tax_1',
            'show_tax_2',
            'show_logo',
            'show_barcode',
            'show_payments',
            'show_customer',
            'show_client_id',
            'show_brand',
            'show_sku',
            'show_cat_code',
            'show_sale_description',
            'show_sales_person',
            'show_expiry',
            'show_lot',
            'show_previous_bal',
            'show_image',
            'show_reward_point'
        ];

        $checkboxStats = [];

        foreach ($checkboxes as $checkbox) {
            $count = $layouts->where($checkbox, 1)->count();
            $checkboxStats[$checkbox] = [
                'count' => $count,
                'percentage' => $layouts->count() > 0 ? round(($count / $layouts->count()) * 100, 2) : 0
            ];
        }

        // Sort by usage count
        arsort($checkboxStats);

        return array_slice($checkboxStats, 0, 10); // Return top 10
    }

    /**
     * Export layout configuration
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function exportConfig($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return $this->sendError('Layout non trouvé', [], 404);
            }

            // Prepare export data (remove business-specific and ID fields)
            $exportData = $layout->toArray();
            unset($exportData['id'], $exportData['business_id'], $exportData['created_at'], $exportData['updated_at']);

            // Add export metadata
            $exportData['export_info'] = [
                'exported_at' => now()->toISOString(),
                'exported_by' => auth()->user()->name ?? auth()->user()->email,
                'version' => '1.0'
            ];

            $fileName = 'invoice_layout_' . $layout->name . '_' . date('Y-m-d_H-i-s') . '.json';

            return response()->json($exportData)
                ->header('Content-Type', 'application/json')
                ->header('Content-Disposition', 'attachment; filename="' . $fileName . '"');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de l\'export de la configuration', [$e->getMessage()], 500);
        }
    }

    /**
     * Import layout configuration
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function importConfig(Request $request)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $validator = Validator::make($request->all(), [
                'config_file' => 'required|file|mimes:json',
                'layout_name' => 'nullable|string|max:255'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Données de validation incorrectes', $validator->errors(), 422);
            }

            $business_id = auth()->user()->business_id;

            // Read and decode the JSON file
            $fileContent = file_get_contents($request->file('config_file')->path());
            $configData = json_decode($fileContent, true);

            if (!$configData) {
                return $this->sendError('Fichier de configuration invalide', [], 422);
            }

            // Remove export info and set business data
            unset($configData['export_info']);
            $configData['business_id'] = $business_id;
            $configData['is_default'] = 0; // Never import as default

            // Use provided name or add suffix to original name
            if ($request->layout_name) {
                $configData['name'] = $request->layout_name;
            } else {
                $configData['name'] = ($configData['name'] ?? 'Layout importé') . ' - Importé';
            }

            // Ensure unique name
            $baseName = $configData['name'];
            $counter = 1;
            while (InvoiceLayout::where('business_id', $business_id)->where('name', $configData['name'])->exists()) {
                $configData['name'] = $baseName . ' (' . $counter . ')';
                $counter++;
            }

            $layout = InvoiceLayout::create($configData);

            return $this->sendResponse($layout, 'Configuration importée avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de l\'import de la configuration', [$e->getMessage()], 500);
        }
    }

    /**
     * Validate layout configuration
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function validateConfig(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'design' => 'required|string|in:' . implode(',', array_keys($this->designs)),
                'highlight_color' => 'nullable|regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/',
            ]);

            $errors = [];
            $warnings = [];

            if ($validator->fails()) {
                $errors = $validator->errors()->toArray();
            }

            // Additional custom validations
            if ($request->has('table_tax_headings')) {
                $taxHeadings = $request->table_tax_headings;
                if (is_array($taxHeadings) && count($taxHeadings) > 4) {
                    $warnings[] = 'Plus de 4 en-têtes de taxe peuvent affecter l\'affichage';
                }
            }

            // Check for potential issues
            if (!$request->show_business_name && !$request->show_logo) {
                $warnings[] = 'Ni le nom de l\'entreprise ni le logo ne sont affichés';
            }

            if (!$request->show_customer) {
                $warnings[] = 'Les informations client ne seront pas affichées';
            }

            $result = [
                'is_valid' => empty($errors),
                'errors' => $errors,
                'warnings' => $warnings,
                'suggestions' => $this->getConfigSuggestions($request)
            ];

            return $this->sendResponse($result, 'Validation de la configuration terminée.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la validation', [$e->getMessage()], 500);
        }
    }

    /**
     * Get configuration suggestions
     *
     * @param \Illuminate\Http\Request $request
     * @return array
     */
    private function getConfigSuggestions($request)
    {
        $suggestions = [];

        if ($request->design === 'slim' && $request->show_logo) {
            $suggestions[] = 'Pour les imprimantes thermiques (slim), considérez désactiver le logo pour optimiser l\'espace';
        }

        if ($request->design !== 'slim' && !$request->show_logo) {
            $suggestions[] = 'Pour les imprimantes normales, l\'affichage du logo améliore l\'apparence professionnelle';
        }

        if (!$request->footer_text) {
            $suggestions[] = 'Ajoutez un texte de pied de page pour inclure des informations de contact ou des conditions';
        }

        return $suggestions;
    }

    /**
     * Reset layout to default configuration
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function resetToDefault($id)
    {
        try {
            if (!auth()->user()->can('invoice_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $layout = InvoiceLayout::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$layout) {
                return $this->sendError('Layout non trouvé', [], 404);
            }

            // Default configuration based on design
            $defaultConfig = $this->getDefaultConfigForDesign($layout->design);
            $defaultConfig['business_id'] = $business_id;
            $defaultConfig['name'] = $layout->name; // Keep the name
            $defaultConfig['is_default'] = $layout->is_default; // Keep default status

            $layout->update($defaultConfig);

            return $this->sendResponse($layout, 'Layout réinitialisé aux valeurs par défaut avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la réinitialisation', [$e->getMessage()], 500);
        }
    }

    /**
     * Get default configuration for a design
     *
     * @param string $design
     * @return array
     */
    private function getDefaultConfigForDesign($design)
    {
        $commonDefaults = [
            'header_text' => '',
            'footer_text' => 'Merci pour votre confiance',
            'invoice_no_prefix' => 'INV-',
            'invoice_heading' => 'Facture',
            'sub_total_label' => 'Sous-total',
            'discount_label' => 'Remise',
            'tax_label' => 'Taxe',
            'total_label' => 'Total',
            'customer_label' => 'Client',
            'date_label' => 'Date',
            'table_product_label' => 'Produit',
            'table_qty_label' => 'Qté',
            'table_unit_price_label' => 'Prix unitaire',
            'table_subtotal_label' => 'Sous-total',
            'show_business_name' => 1,
            'show_customer' => 1,
            'show_payments' => 1,
            'show_sku' => 1,
        ];

        $designSpecific = match ($design) {
            'slim' => [
                'show_logo' => 0,
                'show_barcode' => 0,
                'show_image' => 0,
            ],
            'elegant' => [
                'show_logo' => 1,
                'show_business_name' => 1,
                'highlight_color' => '#3498db',
            ],
            'detailed' => [
                'show_logo' => 1,
                'show_brand' => 1,
                'show_sku' => 1,
                'show_expiry' => 1,
            ],
            default => []
        };

        return array_merge($commonDefaults, $designSpecific, ['design' => $design]);
    }
}
