<?php

namespace App\Http\Controllers;

use App\Models\Business;
use App\Models\BusinessLocation;
use App\Exports\InventoryExport;
use App\Http\Controllers\Api\BaseController;
use App\Imports\InventoryImport;
use App\Models\Inventory;
use App\Models\InventoryProduct;
use App\Models\Product;
use App\Models\User;
use App\Models\VariationLocationDetail;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Validator;

class InventoryController extends BaseController
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id;

            $query = Inventory::where('inventories.business_id', $business_id)
                ->leftjoin('business_locations as bl', 'inventories.location_id', '=', 'bl.id')
                ->leftjoin('users as u', 'inventories.created_by', '=', 'u.id')
                ->select([
                    'inventories.id as id',
                    'inventories.reference as reference',
                    'inventories.location_id',
                    'bl.name as location_name',
                    'inventories.status as status',
                    'u.last_name as last',
                    'u.first_name as first',
                    'inventories.created_at as created_at',
                    'inventories.sell_theoric_total as theoric_total',
                    'inventories.sell_physic_total as physic_total'
                ]);

            // Filter by creator
            if ($request->has('created_by') && !empty($request->created_by)) {
                $query->where('inventories.created_by', $request->created_by);
            }

            // Filter by location
            if ($request->has('location_id') && !empty($request->location_id)) {
                $query->where('inventories.location_id', $request->location_id);
            }

            // Filter by date range
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $query->whereDate('inventories.created_at', '>=', $request->start_date)
                    ->whereDate('inventories.created_at', '<=', $request->end_date);
            }

            // Apply location permissions
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('inventories.location_id', $permitted_locations);
            }

            // Search functionality
            if ($request->has('search') && !empty($request->search)) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('inventories.reference', 'like', "%{$search}%")
                        ->orWhere('bl.name', 'like', "%{$search}%")
                        ->orWhere('u.last_name', 'like', "%{$search}%")
                        ->orWhere('u.first_name', 'like', "%{$search}%");
                });
            }

            $perPage = $request->get('per_page', 10);
            $inventories = $query->orderBy('inventories.created_at', 'desc')->paginate($perPage);

            return $this->sendResponse($inventories, 'Inventaires récupérés avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération des inventaires', [$e->getMessage()], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create()
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $user = auth()->user();

            $permitted_locations = $user->permitted_locations();
            $business_locations = BusinessLocation::where('business_id', $business_id)->get();

            $products = Product::join('variation_location_details as vld', 'vld.product_id', '=', 'products.id')
                ->select(["products.id", "products.name", "vld.qty_available", "vld.location_id"])
                ->get();

            $data = [
                'permitted_locations' => $permitted_locations,
                'business_locations' => $business_locations,
                'products' => $products,
                'user' => $user
            ];

            return $this->sendResponse($data, 'Données pour création d\'inventaire récupérées avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération des données', [$e->getMessage()], 500);
        }
    }

    /**
     * Get products by location
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function products(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'location_id' => 'required|integer|exists:business_locations,id'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Données de validation incorrectes', $validator->errors(), 422);
            }

            $products = Product::join('variation_location_details as vld', 'vld.product_id', '=', 'products.id')
                ->where("vld.location_id", $request->location_id)
                ->select(["products.name", "products.id", "vld.qty_available"])
                ->get();

            return $this->sendResponse($products, 'Produits récupérés avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération des produits', [$e->getMessage()], 500);
        }
    }

    /**
     * Show products for specific location
     *
     * @param int $location_id
     * @return \Illuminate\Http\JsonResponse
     */
    public function createprod($location_id)
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $bl = BusinessLocation::find($location_id);
            if (!$bl) {
                return $this->sendError('Location non trouvée', [], 404);
            }

            $business_id = auth()->user()->business_id;
            $user = auth()->user();
            $permitted_locations = $user->permitted_locations();

            $products = Product::join('variation_location_details as vld', 'vld.product_id', '=', 'products.id')
                ->join('variations as v', 'v.product_id', '=', 'products.id')
                ->where('vld.location_id', $location_id)
                ->select([
                    "products.id",
                    "products.name",
                    "vld.qty_available",
                    "vld.location_id",
                    "v.default_sell_price as sell_price"
                ])
                ->get();

            $data = [
                'permitted_locations' => $permitted_locations,
                'business_location' => $bl,
                'products' => $products,
                'user' => $user
            ];

            return $this->sendResponse($data, 'Produits pour la location récupérés avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération des produits', [$e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $location_id
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request, $location_id)
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $validator = Validator::make($request->all(), [
                'product' => 'required|array',
                'theorique' => 'required|array',
                'physique' => 'required|array',
                'sell_price' => 'required|array',
                'sell_theory_total' => 'required|array',
                'sell_physic_total' => 'required|array',
                'observation' => 'array'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Données de validation incorrectes', $validator->errors(), 422);
            }

            $business_id = auth()->user()->business_id;
            $location = BusinessLocation::find($location_id);

            if (!$location) {
                return $this->sendError('Location non trouvée', [], 404);
            }

            DB::transaction(function () use ($request, $business_id, $location) {
                $trans = new Inventory();
                $trans->business_id = $business_id;
                $trans->location_id = $location->id;
                $trans->inv_create_at = Carbon::now();
                $trans->reference = 'INVENT' . '-' . date('y') . substr(number_format(time() * mt_rand(), 0, '', ''), 0, 5);
                $trans->status = 'pending';
                $trans->invent = 'INVENT' . '-' . date('y') . substr(number_format(time() * mt_rand(), 0, '', ''), 0, 5);
                $trans->created_by = Auth::user()->id;

                // Calculate totals
                $total_sell_theoric = 0;
                $total_sell_physic = 0;

                for ($product_id = 0; $product_id < count($request->product); $product_id++) {
                    $unit_price = $request->sell_price[$product_id];
                    $sell_theoric = $request->theorique[$product_id];
                    $sell_physic = $request->physique[$product_id];

                    $total_sell_theoric += $unit_price * $sell_theoric;
                    $total_sell_physic += $unit_price * $sell_physic;
                }

                $trans->sell_theoric_total = $total_sell_theoric;
                $trans->sell_physic_total = $total_sell_physic;
                $trans->save();

                // Create inventory products
                for ($product_id = 0; $product_id < count($request->product); $product_id++) {
                    $sell = new InventoryProduct();
                    $sell->inventaire_id = $trans->id;
                    $sell->product_id = $request->product[$product_id];
                    $sell->stock_theorique = $request->theorique[$product_id];
                    $sell->stock_physique = $request->physique[$product_id];
                    $sell->ecart = $request->physique[$product_id] - $request->theorique[$product_id];
                    $sell->unit_price = $request->sell_price[$product_id];
                    $sell->sell_theoric = $request->sell_theory_total[$product_id];
                    $sell->sell_physic = $request->sell_physic_total[$product_id];
                    $sell->observation = $request->observation[$product_id] ?? '';
                    $sell->save();

                    // Update stock quantities
                    $product_qty = VariationLocationDetail::where('location_id', $trans->location_id)
                        ->where('product_id', $sell->product_id)
                        ->first();

                    if ($product_qty) {
                        $product_qty->qty_available = $sell->stock_physique;
                        $product_qty->update();
                    }
                }
            });

            return $this->sendResponse(null, 'Inventaire créé avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la création de l\'inventaire', [$e->getMessage()], 500);
        }
    }

    /**
     * Import inventory from Excel file
     *
     * @param Request $request
     * @param int $location_id
     * @return \Illuminate\Http\JsonResponse
     */
    public function import(Request $request, $location_id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'excel_file' => 'required|mimes:xlsx,xls'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Fichier Excel requis', $validator->errors(), 422);
            }

            Excel::import(new InventoryImport($location_id), $request->file('excel_file'));

            return $this->sendResponse(null, 'Inventaire importé avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de l\'importation', [$e->getMessage()], 500);
        }
    }

    /**
     * Export inventory to Excel
     *
     * @param int $location_id
     * @return \Illuminate\Http\JsonResponse
     */
    public function export($location_id)
    {
        try {
            $location = BusinessLocation::findOrFail($location_id);
            $fileName = 'inventaire-' . $location->name . '-' . date('Y-m-d-His') . '.xlsx';

            return Excel::download(new InventoryExport($location_id), $fileName);
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de l\'export', [$e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $business_id = auth()->user()->business_id;

            $inventory = Inventory::with(['location', 'creator'])
                ->where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$inventory) {
                return $this->sendError('Inventaire non trouvé', [], 404);
            }

            $location_details = Business::where('id', $business_id)->first();
            $invProducts = InventoryProduct::with('product')
                ->where('inventaire_id', $inventory->id)
                ->get();

            $data = [
                'inventory' => $inventory,
                'location_details' => $location_details,
                'inventory_products' => $invProducts
            ];

            return $this->sendResponse($data, 'Inventaire récupéré avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération de l\'inventaire', [$e->getMessage()], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $inventory = Inventory::with(['location', 'creator'])
                ->where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$inventory) {
                return $this->sendError('Inventaire non trouvé', [], 404);
            }

            $invProducts = InventoryProduct::with('product')
                ->where('inventaire_id', $inventory->id)
                ->get();

            $business_locations = BusinessLocation::where('business_id', $business_id)->get();
            $users = User::where('business_id', $business_id)->get();

            $data = [
                'inventory' => $inventory,
                'inventory_products' => $invProducts,
                'business_locations' => $business_locations,
                'users' => $users
            ];

            return $this->sendResponse($data, 'Données pour édition récupérées avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la récupération des données', [$e->getMessage()], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $inventory = Inventory::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$inventory) {
                return $this->sendError('Inventaire non trouvé', [], 404);
            }

            $validator = Validator::make($request->all(), [
                'status' => 'string|in:pending,confirmed,cancelled',
                'location_id' => 'integer|exists:business_locations,id'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Données de validation incorrectes', $validator->errors(), 422);
            }

            DB::transaction(function () use ($request, $inventory) {
                // Update inventory
                $inventory->update($request->only(['status', 'location_id']));

                // Update products if provided
                if ($request->has('products')) {
                    // Delete existing products
                    InventoryProduct::where('inventaire_id', $inventory->id)->delete();

                    // Add new products
                    foreach ($request->products as $productData) {
                        InventoryProduct::create(array_merge($productData, [
                            'inventaire_id' => $inventory->id
                        ]));
                    }
                }
            });

            return $this->sendResponse($inventory, 'Inventaire mis à jour avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la mise à jour', [$e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $inventory = Inventory::where('id', $id)
                ->where('business_id', $business_id)
                ->first();

            if (!$inventory) {
                return $this->sendError('Inventaire non trouvé', [], 404);
            }

            DB::transaction(function () use ($inventory) {
                // Delete related products
                InventoryProduct::where('inventaire_id', $inventory->id)->delete();

                // Delete inventory
                $inventory->delete();
            });

            return $this->sendResponse(null, 'Inventaire supprimé avec succès.');
        } catch (\Exception $e) {
            return $this->sendError('Erreur lors de la suppression', [$e->getMessage()], 500);
        }
    }
}
