<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\BusinessLocation;
use App\Models\Product;
use App\Models\Transaction;
use App\Utils\ProductUtil;
use App\Models\Variation;
use Carbon\Carbon;
use Excel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class ImportOpeningStockController extends BaseController
{
    protected $productUtil;

    public function __construct(ProductUtil $productUtil)
    {
        $this->productUtil = $productUtil;
    }

    /**
     * Get import opening stock form data
     * @OA\Get(
     *     path="/api/import-opening-stock",
     *     summary="Get import opening stock form",
     *     tags={"Import Opening Stock"},
     *     @OA\Response(response=200, description="Success")
     * )
     */
    public function index()
    {
        try {
            if (!auth()->user()->can('product.opening_stock')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $zip_loaded = extension_loaded('zip');
            $date_formats = ['d/m/Y', 'm/d/Y', 'Y/m/d']; // Simplified date formats

            $response_data = [
                'zip_loaded' => $zip_loaded,
                'date_formats' => $date_formats,
                'sample_file_url' => asset('files/import_opening_stock_csv_template.csv')
            ];

            if (!$zip_loaded) {
                $response_data['warning'] = 'Please install/enable PHP Zip archive for import';
            }

            return $this->sendResponse($response_data, 'Import opening stock form data retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving form data', [$e->getMessage()], 500);
        }
    }

    /**
     * Create/Import opening stock
     * @OA\Post(
     *     path="/api/import-opening-stock",
     *     summary="Import opening stock from CSV",
     *     tags={"Import Opening Stock"},
     *     @OA\Response(response=200, description="Success")
     * )
     */
    public function store(Request $request)
    {
        try {
            if (!auth()->user()->can('product.opening_stock')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $notAllowed = $this->productUtil->notAllowedInDemo();
            if (!empty($notAllowed)) {
                return $this->sendError('Not allowed in demo', [], 403);
            }

            $request->validate([
                'products_csv' => 'required|file|mimes:csv,txt,xlsx'
            ]);

            // Set maximum php execution time
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', -1);

            if ($request->hasFile('products_csv')) {
                $file = $request->file('products_csv');

                $parsed_array = Excel::toArray([], $file);
                // Remove header row
                $imported_data = array_splice($parsed_array[0], 1);

                $business_id = auth()->user()->business_id;
                $user_id = auth()->user()->id;

                $is_valid = true;
                $error_msg = '';
                $processed_count = 0;

                DB::beginTransaction();

                foreach ($imported_data as $key => $value) {
                    $row_no = $key + 1;

                    // Check for product SKU, get product id, variation id.
                    if (!empty($value[0])) {
                        $sku = $value[0];
                        $product_info = Variation::where('sub_sku', $sku)
                            ->join('products AS P', 'variations.product_id', '=', 'P.id')
                            ->leftjoin('tax_rates AS TR', 'P.tax', 'TR.id')
                            ->where('P.business_id', $business_id)
                            ->select([
                                'P.id',
                                'variations.id as variation_id',
                                'P.enable_stock',
                                'TR.amount as tax_percent',
                                'TR.id as tax_id'
                            ])
                            ->first();

                        if (empty($product_info)) {
                            $is_valid = false;
                            $error_msg = "Product with sku $sku not found in row no. $row_no";
                            break;
                        } elseif ($product_info->enable_stock == 0) {
                            $is_valid = false;
                            $error_msg = "Manage Stock not enabled for the product with $sku in row no. $row_no";
                            break;
                        }
                    } else {
                        $is_valid = false;
                        $error_msg = "PRODUCT SKU is required in row no. $row_no";
                        break;
                    }

                    // Get location details.
                    if (!empty(trim($value[1]))) {
                        $location_name = trim($value[1]);
                        $location = BusinessLocation::where('name', $location_name)
                            ->where('business_id', $business_id)
                            ->first();
                        if (empty($location)) {
                            $is_valid = false;
                            $error_msg = "Location with name '$location_name' not found in row no. $row_no";
                            break;
                        }
                    } else {
                        $location = BusinessLocation::where('business_id', $business_id)->first();
                    }

                    $opening_stock = [
                        'quantity' => trim($value[2]),
                        'location_id' => $location->id,
                        'lot_number' => trim($value[4]),
                    ];

                    if (!empty(trim($value[5]))) {
                        $opening_stock['exp_date'] = $this->productUtil->uf_date($value[5]);
                    }

                    if (!empty(trim($value[3]))) {
                        $unit_cost_before_tax = trim($value[3]);
                    } else {
                        $is_valid = false;
                        $error_msg = "Invalid UNIT COST in row no. $row_no";
                        break;
                    }

                    if (!is_numeric(trim($value[2]))) {
                        $is_valid = false;
                        $error_msg = "Invalid quantity $value[2] in row no. $row_no";
                        break;
                    }

                    // Check for existing opening stock transaction
                    $os_transaction = Transaction::where('business_id', $business_id)
                        ->where('location_id', $location->id)
                        ->where('type', 'opening_stock')
                        ->where('opening_stock_product_id', $product_info->id)
                        ->first();

                    $this->addOpeningStock($opening_stock, $product_info, $business_id, $unit_cost_before_tax, $os_transaction);
                    $processed_count++;
                }

                if (!$is_valid) {
                    throw new \Exception($error_msg);
                }

                DB::commit();

                return $this->sendResponse([
                    'processed_count' => $processed_count
                ], 'Opening stock imported successfully');
            }

            return $this->sendError('No file uploaded', [], 400);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Import failed', [$e->getMessage()], 500);
        }
    }

    /**
     * Show the form for creating opening stock
     */
    public function create()
    {
        try {
            if (!auth()->user()->can('product.opening_stock')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $locations = BusinessLocation::forDropdown($business_id);
            $products = Product::where('business_id', $business_id)
                ->where('enable_stock', 1)
                ->where('is_inactive', 0)
                ->with('variations')
                ->get();

            return $this->sendResponse([
                'locations' => $locations,
                'products' => $products
            ], 'Opening stock form data retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving form data', [$e->getMessage()], 500);
        }
    }

    /**
     * Show opening stock details
     */
    public function show($id)
    {
        try {
            if (!auth()->user()->can('product.opening_stock')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $transaction = Transaction::where('business_id', $business_id)
                ->where('type', 'opening_stock')
                ->where('id', $id)
                ->with(['purchase_lines.product', 'purchase_lines.variation', 'location'])
                ->first();

            if (!$transaction) {
                return $this->sendError('Opening stock not found', [], 404);
            }

            return $this->sendResponse($transaction, 'Opening stock details retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving opening stock details', [$e->getMessage()], 500);
        }
    }

    /**
     * Show the form for editing opening stock
     */
    public function edit($id)
    {
        try {
            if (!auth()->user()->can('product.opening_stock')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $transaction = Transaction::where('business_id', $business_id)
                ->where('type', 'opening_stock')
                ->where('id', $id)
                ->with(['purchase_lines.product', 'purchase_lines.variation'])
                ->first();

            if (!$transaction) {
                return $this->sendError('Opening stock not found', [], 404);
            }

            $locations = BusinessLocation::forDropdown($business_id);

            return $this->sendResponse([
                'transaction' => $transaction,
                'locations' => $locations
            ], 'Opening stock edit form retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving edit form', [$e->getMessage()], 500);
        }
    }

    /**
     * Update opening stock
     */
    public function update(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('product.opening_stock')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $transaction = Transaction::where('business_id', $business_id)
                ->where('type', 'opening_stock')
                ->where('id', $id)
                ->first();

            if (!$transaction) {
                return $this->sendError('Opening stock not found', [], 404);
            }

            DB::beginTransaction();

            // Update transaction details
            $transaction->update([
                'location_id' => $request->location_id,
                'transaction_date' => $request->transaction_date
            ]);

            DB::commit();

            return $this->sendResponse($transaction, 'Opening stock updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Error updating opening stock', [$e->getMessage()], 500);
        }
    }

    /**
     * Delete opening stock
     */
    public function destroy($id)
    {
        try {
            if (!auth()->user()->can('product.opening_stock')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $transaction = Transaction::where('business_id', $business_id)
                ->where('type', 'opening_stock')
                ->where('id', $id)
                ->first();

            if (!$transaction) {
                return $this->sendError('Opening stock not found', [], 404);
            }

            DB::beginTransaction();

            // Delete purchase lines and adjust stock
            foreach ($transaction->purchase_lines as $line) {
                $this->productUtil->updateProductQuantity(
                    $transaction->location_id,
                    $line->product_id,
                    $line->variation_id,
                    -$line->quantity
                );
            }

            $transaction->purchase_lines()->delete();
            $transaction->delete();

            DB::commit();

            return $this->sendResponse([], 'Opening stock deleted successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->sendError('Error deleting opening stock', [$e->getMessage()], 500);
        }
    }

    /**
     * Adds opening stock of a single product
     */
    private function addOpeningStock($opening_stock, $product, $business_id, $unit_cost_before_tax, $transaction = null)
    {
        $user_id = auth()->user()->id;
        $transaction_date = auth()->user()->business->financial_year_start ?? Carbon::now()->format('Y-m-d');
        $transaction_date = Carbon::createFromFormat('Y-m-d', $transaction_date)->toDateTimeString();

        // Get product tax
        $tax_percent = !empty($product->tax_percent) ? $product->tax_percent : 0;
        $tax_id = !empty($product->tax_id) ? $product->tax_id : null;

        $item_tax = $this->productUtil->calc_percentage($unit_cost_before_tax, $tax_percent);

        // Total before transaction tax
        $total_before_trans_tax = $opening_stock['quantity'] * ($unit_cost_before_tax + $item_tax);

        // Add opening stock transaction
        if (empty($transaction)) {
            $transaction = new Transaction();
            $transaction->type = 'opening_stock';
            $transaction->opening_stock_product_id = $product->id;
            $transaction->business_id = $business_id;
            $transaction->transaction_date = $transaction_date;
            $transaction->location_id = $opening_stock['location_id'];
            $transaction->payment_status = 'paid';
            $transaction->created_by = $user_id;
            $transaction->total_before_tax = 0;
            $transaction->final_total = 0;
        }

        $transaction->total_before_tax += $total_before_trans_tax;
        $transaction->final_total += $total_before_trans_tax;
        $transaction->save();

        // Create purchase line
        $transaction->purchase_lines()->create([
            'product_id' => $product->id,
            'variation_id' => $product->variation_id,
            'quantity' => $opening_stock['quantity'],
            'pp_without_discount' => $unit_cost_before_tax,
            'item_tax' => $item_tax,
            'tax_id' => $tax_id,
            'purchase_price' => $unit_cost_before_tax,
            'purchase_price_inc_tax' => $unit_cost_before_tax + $item_tax,
            'exp_date' => !empty($opening_stock['exp_date']) ? $opening_stock['exp_date'] : null,
            'lot_number' => !empty($opening_stock['lot_number']) ? $opening_stock['lot_number'] : null,
        ]);

        // Update variation location details
        $this->productUtil->updateProductQuantity($opening_stock['location_id'], $product->id, $product->variation_id, $opening_stock['quantity']);
    }
}
