<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Charts\CommonChart;
use App\Models\Contact;
use App\Models\Currency;
use App\Models\CustomerGroup;
use App\Models\Transaction;
use App\Models\BusinessLocation;
use App\Utils\BusinessUtil;
use App\Utils\ContactUtil;
use App\Utils\ModuleUtil;
use App\Utils\TransactionUtil;
use App\Models\VariationLocationDetail;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class HomeController extends BaseController
{
    protected $businessUtil;
    protected $transactionUtil;
    protected $moduleUtil;
    protected $contactUtil;

    public function __construct(
        BusinessUtil $businessUtil,
        TransactionUtil $transactionUtil,
        ModuleUtil $moduleUtil,
        ContactUtil $contactUtil
    ) {
        $this->businessUtil = $businessUtil;
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
        $this->contactUtil = $contactUtil;
    }

    /**
     * Dashboard data
     * @OA\Get(
     *     path="/api/dashboard",
     *     summary="Get dashboard data",
     *     tags={"Dashboard"},
     *     @OA\Response(response=200, description="Success")
     * )
     */
    public function index(Request $request)
    {
        try {
            $business_id = auth()->user()->business_id;

            if (!auth()->user()->can('dashboard.data')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $dat = $request->get('date-filter');

            // Date parsing logic
            $d = [];
            if ($dat) {
                $an_start = substr($dat, 6, 4);
                $jour_start = substr($dat, 3, 2);
                $mois_start = substr($dat, 0, 2);
                $start_date = "$an_start-$mois_start-$jour_start";
                $start_date1 = "$jour_start-$mois_start-$an_start";

                $an_end = substr($dat, 19, 4);
                $jour_end = substr($dat, 16, 2);
                $mois_end = substr($dat, 13, 2);
                $end_date = "$an_end-$mois_end-$jour_end";
                $end_date1 = "$jour_end-$mois_end-$an_end";

                $d = [
                    "start" => $start_date,
                    "end" => $end_date
                ];
            }

            $fy = [
                "start" => "2021-02-2",
                "end" => "2021-12-31"
            ];

            if (empty($d)) {
                $date_filters['this_fy'] = $fy;
                $date_filters['this_date']['start'] = date('Y-m-d');
                $date_filters['this_date']['end'] = date('Y-m-d');
            } else {
                $date_filters['this_fy'] = $d;
                $date_filters['this_date']['start'] = $start_date1;
                $date_filters['this_date']['end'] = $end_date1;
            }

            $date_filters['this_month']['start'] = date('Y-m-01');
            $date_filters['this_month']['end'] = date('Y-m-t');
            $date_filters['this_week']['start'] = date('Y-m-d', strtotime('monday this week'));
            $date_filters['this_week']['end'] = date('Y-m-d', strtotime('sunday this week'));

            $start_date = $d['start'] ?? date('Y-m-d');
            $end_date = $d['end'] ?? date('Y-m-d');

            $date = $this->transactionUtil->getSellsBetweenDate($business_id, $start_date, $end_date);
            $date_filters['date_bey'] = $date;

            $currency = Currency::where('id', auth()->user()->business->currency_id)->first();

            // Get sells last 30 days
            $sells_last_30_days = $this->transactionUtil->getSellsLast30Days($business_id);
            $labels = [];
            $all_sell_values = [];
            $dates = [];

            for ($i = 29; $i >= 0; $i--) {
                $date = Carbon::now()->subDays($i)->format('Y-m-d');
                $dates[] = $date;
                $labels[] = date('j M Y', strtotime($date));

                if (!empty($sells_last_30_days[$date])) {
                    $all_sell_values[] = (float) $sells_last_30_days[$date];
                } else {
                    $all_sell_values[] = 0;
                }
            }

            $all_locations = BusinessLocation::forDropdown($business_id)->toArray();

            $response_data = [
                'date_filters' => $date_filters,
                'currency' => $currency,
                'sells_chart_data' => [
                    'labels' => $labels,
                    'values' => $all_sell_values
                ],
                'locations' => $all_locations
            ];

            return $this->sendResponse($response_data, 'Dashboard data retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving dashboard data', [$e->getMessage()], 500);
        }
    }

    /**
     * Get due contacts
     */
    public function getDueContacts(Request $request)
    {
        try {
            $business_id = auth()->user()->business_id;

            if (!auth()->user()->can('dashboard.data')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $contacts = Contact::where('contacts.business_id', $business_id)
                ->join('transactions AS t', 'contacts.id', '=', 't.contact_id')
                ->active()
                ->groupBy('contacts.id')
                ->select(
                    DB::raw("SUM(IF(t.type = 'purchase', final_total, 0)) as total_purchase"),
                    DB::raw("SUM(IF(t.type = 'purchase_return', final_total, 0)) as total_purchase_return"),
                    DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', final_total, 0)) as total_invoice"),
                    DB::raw("SUM(IF(t.type = 'purchase', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as purchase_paid"),
                    DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', (SELECT SUM(IF(is_return = 1,-1*amount,amount)) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as invoice_received"),
                    DB::raw("SUM(IF(t.type = 'sell_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as sell_return_paid"),
                    DB::raw("SUM(IF(t.type = 'purchase_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as purchase_return_received"),
                    DB::raw("SUM(IF(t.type = 'sell_return', final_total, 0)) as total_sell_return"),
                    DB::raw("SUM(IF(t.type = 'opening_balance', final_total, 0)) as opening_balance"),
                    DB::raw("SUM(IF(t.type = 'opening_balance', (SELECT SUM(IF(is_return = 1,-1*amount,amount)) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as opening_balance_paid"),
                    'contacts.supplier_business_name',
                    'contacts.name',
                    'contacts.id',
                    'contacts.type as contact_type'
                );

            $permitted_locations = auth()->user()->permitted_locations();

            if ($permitted_locations != 'all') {
                $contacts->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->input('customer_group_id'))) {
                $contacts->where('contacts.customer_group_id', $request->input('customer_group_id'));
            }

            if (!empty($request->input('contact_type'))) {
                $contacts->whereIn('contacts.type', [$request->input('contact_type'), 'both']);
            }

            $contacts_data = $contacts->get()->map(function ($row) {
                $due = ($row->total_invoice - $row->invoice_received - $row->total_sell_return + $row->sell_return_paid) - ($row->total_purchase - $row->total_purchase_return + $row->purchase_return_received - $row->purchase_paid);

                if ($row->contact_type == 'supplier') {
                    $due -= $row->opening_balance - $row->opening_balance_paid;
                } else {
                    $due += $row->opening_balance - $row->opening_balance_paid;
                }

                return [
                    'id' => $row->id,
                    'name' => $row->name,
                    'supplier_business_name' => $row->supplier_business_name,
                    'contact_type' => $row->contact_type,
                    'due' => $due
                ];
            });

            return $this->sendResponse($contacts_data, 'Due contacts retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving due contacts', [$e->getMessage()], 500);
        }
    }

    /**
     * Get totals for dashboard
     */
    public function getTotals(Request $request)
    {
        try {
            $start = $request->start;
            $end = $request->end;
            $location_id = $request->location_id;
            $business_id = auth()->user()->business_id;

            $purchase_details = $this->transactionUtil->getPurchaseTotals($business_id, $start, $end, $location_id);
            $sell_details = $this->transactionUtil->getSellTotals($business_id, $start, $end, $location_id);

            $transaction_types = ['purchase_return', 'sell_return', 'expense'];
            $transaction_totals = $this->transactionUtil->getTransactionTotals($business_id, $transaction_types, $start, $end);

            $total_purchase_inc_tax = !empty($purchase_details['total_purchase_inc_tax']) ? $purchase_details['total_purchase_inc_tax'] : 0;
            $total_purchase_return_inc_tax = $transaction_totals['total_purchase_return_inc_tax'];
            $total_purchase = $total_purchase_inc_tax - $total_purchase_return_inc_tax;

            $total_sell_inc_tax = !empty($sell_details['total_sell_inc_tax']) ? $sell_details['total_sell_inc_tax'] : 0;
            $total_sell_return_inc_tax = !empty($transaction_totals['total_sell_return_inc_tax']) ? $transaction_totals['total_sell_return_inc_tax'] : 0;

            $output = $purchase_details;
            $output['total_purchase'] = $total_purchase;
            $output['total_sell'] = $total_sell_inc_tax - $total_sell_return_inc_tax;
            $output['invoice_due'] = $sell_details['invoice_due'];
            $output['total_expense'] = $transaction_totals['total_expense'];

            return $this->sendResponse($output, 'Totals retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving totals', [$e->getMessage()], 500);
        }
    }

    /**
     * Get product stock alerts
     */
    public function getProductStockAlert(Request $request)
    {
        try {
            $business_id = auth()->user()->business_id;

            $query = VariationLocationDetail::join('product_variations as pv', 'variation_location_details.product_variation_id', '=', 'pv.id')
                ->join('variations as v', 'variation_location_details.variation_id', '=', 'v.id')
                ->join('products as p', 'variation_location_details.product_id', '=', 'p.id')
                ->leftjoin('business_locations as l', 'variation_location_details.location_id', '=', 'l.id')
                ->leftjoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('p.business_id', $business_id)
                ->where('p.enable_stock', 1)
                ->where('p.is_inactive', 0)
                ->whereRaw('variation_location_details.qty_available <= p.alert_quantity');

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('variation_location_details.location_id', $permitted_locations);
            }

            $products = $query->select(
                'p.name as product',
                'p.type',
                'pv.name as product_variation',
                'v.name as variation',
                'l.name as location',
                'variation_location_details.qty_available as stock',
                'u.short_name as unit'
            )->groupBy('variation_location_details.id')
                ->orderBy('stock', 'asc')
                ->get();

            $formatted_products = $products->map(function ($row) {
                $product_name = $row->type == 'single' ? $row->product : $row->product . ' - ' . $row->product_variation . ' - ' . $row->variation;
                $stock = $row->stock ? $row->stock : 0;

                return [
                    'product' => $product_name,
                    'location' => $row->location,
                    'stock' => $stock,
                    'unit' => $row->unit
                ];
            });

            return $this->sendResponse($formatted_products, 'Stock alerts retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving stock alerts', [$e->getMessage()], 500);
        }
    }

    /**
     * Get purchase payment dues
     */
    public function getPurchasePaymentDues(Request $request)
    {
        try {
            $business_id = auth()->user()->business_id;
            $today = Carbon::now()->format("Y-m-d H:i:s");

            $query = Transaction::join('contacts as c', 'transactions.contact_id', '=', 'c.id')
                ->leftJoin('transaction_payments as tp', 'transactions.id', '=', 'tp.transaction_id')
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'purchase')
                ->where('transactions.payment_status', '!=', 'paid')
                ->whereRaw("DATEDIFF( DATE_ADD( transaction_date, INTERVAL IF(c.pay_term_type = 'days', c.pay_term_number, 30 * c.pay_term_number) DAY), '$today') <= 7");

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('transactions.location_id', $permitted_locations);
            }

            $dues = $query->select(
                'transactions.id as id',
                'c.name as supplier',
                'ref_no',
                'final_total',
                DB::raw('SUM(tp.amount) as total_paid')
            )->groupBy('transactions.id')->get();

            $formatted_dues = $dues->map(function ($row) {
                $total_paid = !empty($row->total_paid) ? $row->total_paid : 0;
                $due = $row->final_total - $total_paid;

                return [
                    'id' => $row->id,
                    'supplier' => $row->supplier,
                    'ref_no' => $row->ref_no,
                    'final_total' => $row->final_total,
                    'total_paid' => $total_paid,
                    'due' => $due
                ];
            });

            return $this->sendResponse($formatted_dues, 'Purchase payment dues retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving purchase payment dues', [$e->getMessage()], 500);
        }
    }

    /**
     * Get sales payment dues
     */
    public function getSalesPaymentDues(Request $request)
    {
        try {
            $business_id = auth()->user()->business_id;
            $today = Carbon::now()->format("Y-m-d H:i:s");

            $query = Transaction::join('contacts as c', 'transactions.contact_id', '=', 'c.id')
                ->leftJoin('transaction_payments as tp', 'transactions.id', '=', 'tp.transaction_id')
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'sell')
                ->where('transactions.payment_status', '!=', 'paid')
                ->where('transactions.payment_status', '!=', 'settle')
                ->whereNotNull('transactions.pay_term_number')
                ->whereNotNull('transactions.pay_term_type')
                ->whereRaw("DATEDIFF( DATE_ADD( transaction_date, INTERVAL IF(transactions.pay_term_type = 'days', transactions.pay_term_number, 30 * transactions.pay_term_number) DAY), '$today') <= 7");

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('transactions.location_id', $permitted_locations);
            }

            $dues = $query->select(
                'transactions.id as id',
                'c.name as customer',
                'transactions.invoice_no',
                'final_total',
                DB::raw('SUM(tp.amount) as total_paid')
            )->groupBy('transactions.id')->get();

            $formatted_dues = $dues->map(function ($row) {
                $total_paid = !empty($row->total_paid) ? $row->total_paid : 0;
                $due = $row->final_total - $total_paid;

                return [
                    'id' => $row->id,
                    'customer' => $row->customer,
                    'invoice_no' => $row->invoice_no,
                    'final_total' => $row->final_total,
                    'total_paid' => $total_paid,
                    'due' => $due
                ];
            });

            return $this->sendResponse($formatted_dues, 'Sales payment dues retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving sales payment dues', [$e->getMessage()], 500);
        }
    }

    /**
     * Load more notifications
     */
    public function loadMoreNotifications(Request $request)
    {
        try {
            $page = $request->input('page', 1);
            $notifications = auth()->user()->notifications()->orderBy('created_at', 'DESC')->paginate(10, ['*'], 'page', $page);

            if ($page == 1) {
                auth()->user()->unreadNotifications->markAsRead();
            }

            $notifications_data = [];
            foreach ($notifications as $notification) {
                $data = $notification->data;
                if (in_array($notification->type, [\App\Notifications\RecurringInvoiceNotification::class, \App\Notifications\RecurringExpenseNotification::class])) {
                    $msg = '';
                    $icon_class = '';
                    $link = '';

                    if ($notification->type == \App\Notifications\RecurringInvoiceNotification::class) {
                        $msg = !empty($data['invoice_status']) && $data['invoice_status'] == 'draft' ?
                            'Recurring invoice error: ' . $data['out_of_stock_product'] :
                            'Recurring invoice created: ' . $data['invoice_no'];
                        $icon_class = !empty($data['invoice_status']) && $data['invoice_status'] == 'draft' ? "fas fa-exclamation-triangle bg-yellow" : "fas fa-recycle bg-green";
                        $link = '/subscriptions';
                    } else if ($notification->type == \App\Notifications\RecurringExpenseNotification::class) {
                        $msg = 'Recurring expense created: ' . $data['ref_no'];
                        $icon_class = "fas fa-recycle bg-green";
                        $link = '/expenses';
                    }

                    $notifications_data[] = [
                        'msg' => $msg,
                        'icon_class' => $icon_class,
                        'link' => $link,
                        'read_at' => $notification->read_at,
                        'created_at' => $notification->created_at->diffForHumans()
                    ];
                }
            }

            return $this->sendResponse([
                'notifications' => $notifications_data,
                'pagination' => [
                    'current_page' => $notifications->currentPage(),
                    'last_page' => $notifications->lastPage(),
                    'per_page' => $notifications->perPage(),
                    'total' => $notifications->total()
                ]
            ], 'Notifications loaded successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error loading notifications', [$e->getMessage()], 500);
        }
    }

    /**
     * Get total unread notifications count
     */
    public function getTotalUnreadNotifications(Request $request)
    {
        try {
            $total_unread = auth()->user()->unreadNotifications->count();

            return $this->sendResponse(['total_unread' => $total_unread], 'Unread notifications count retrieved successfully');
        } catch (\Exception $e) {
            return $this->sendError('Error getting unread notifications count', [$e->getMessage()], 500);
        }
    }
}
