<?php

namespace App\Http\Controllers;


use App\Http\Controllers\Controller;
use App\Http\Controllers\Api\BaseController;
use Illuminate\Http\Request;
use App\Models\TaxRate;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class GroupTaxController extends BaseController
{
    /**
     * Display a listing of group taxes
     *
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $tax_rates = TaxRate::where('business_id', $business_id)
                ->where('is_tax_group', '1')
                ->with(['sub_taxes'])
                ->get();

            // Format sub_taxes for each tax rate
            $tax_rates->transform(function ($tax_rate) {
                $sub_taxes_names = [];
                foreach ($tax_rate->sub_taxes as $sub_tax) {
                    $sub_taxes_names[] = $sub_tax->name;
                }
                $tax_rate->sub_taxes_formatted = implode(' + ', $sub_taxes_names);
                return $tax_rate;
            });

            return $this->sendResponse($tax_rates, 'Liste des groupes de taxes récupérée avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Get form data for creating group tax
     *
     * @return JsonResponse
     */
    public function create(): JsonResponse
    {
        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $taxes = TaxRate::where('business_id', $business_id)
                ->where('is_tax_group', '0')
                ->get(['id', 'name', 'amount']);

            $sub_taxes = [];
            foreach ($tax_rate->sub_taxes as $sub_tax) {
                $sub_taxes[] = $sub_tax->id;
            }

            return $this->sendResponse([
                'tax_rate' => $tax_rate,
                'taxes' => $taxes,
                'sub_taxes' => $sub_taxes
            ], 'Données de modification récupérées avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Groupe de taxes non trouvé', [], 404);
        }
    }

    /**
     * Update the specified group tax
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function update(Request $request, int $id): JsonResponse
    {
        try {
            $business_id = auth()->user()->business_id;
            $sub_tax_ids = $request->input('taxes');

            $sub_taxes = TaxRate::whereIn('id', $sub_tax_ids)->get();
            $amount = 0;
            foreach ($sub_taxes as $sub_tax) {
                $amount += $sub_tax->amount;
            }

            $tax_rate = TaxRate::where('business_id', $business_id)
                ->where('is_tax_group', '1')
                ->findOrFail($id);

            $tax_rate->name = $request->input('name');
            $tax_rate->amount = $amount;
            $tax_rate->save();
            $tax_rate->sub_taxes()->sync($sub_tax_ids);

            $tax_rate->load('sub_taxes');

            return $this->sendResponse($tax_rate, 'Groupe de taxes mis à jour avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Remove the specified group tax
     *
     * @param int $id
     * @return JsonResponse
     */
    public function destroy(int $id): JsonResponse
    {
        try {
            $business_id = request()->user()->business_id;

            $tax_rate = TaxRate::where('business_id', $business_id)
                ->where('is_tax_group', '1')
                ->findOrFail($id);
            $tax_rate->delete();

            return $this->sendResponse(null, 'Groupe de taxes supprimé avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }
}
