<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Transaction;
use App\Models\ExpenseCategory;
use App\Models\BusinessLocation;
use App\Models\User;
use App\Models\Contact;
use App\Models\TaxRate;
use App\Models\Account;
use App\Models\AccountTransaction;
use App\Utils\TransactionUtil;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class ExpenseController extends BaseController
{
    /**
     * Constructor
     *
     * @param TransactionUtil $transactionUtil
     * @return void
     */
    protected $transactionUtil;
    protected $moduleUtil;
    protected $dummyPaymentLine;
    public function __construct(TransactionUtil $transactionUtil, ModuleUtil $moduleUtil)
    {
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
        $this->dummyPaymentLine = [
            'method' => 'cash',
            'amount' => 0,
            'note' => '',
            'card_transaction_number' => '',
            'card_number' => '',
            'card_type' => '',
            'card_holder_name' => '',
            'card_month' => '',
            'card_year' => '',
            'card_security' => '',
            'cheque_number' => '',
            'bank_account_number' => '',
            'is_return' => 0,
            'transaction_no' => ''
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('expense.access') && !auth()->user()->can('view_own_expense')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = auth()->user()->business_id;

            $expenses = Transaction::leftJoin('expense_categories AS ec', 'transactions.expense_category_id', '=', 'ec.id')
                ->join(
                    'business_locations AS bl',
                    'transactions.location_id',
                    '=',
                    'bl.id'
                )
                ->leftJoin('tax_rates as tr', 'transactions.tax_id', '=', 'tr.id')
                ->leftJoin('users AS U', 'transactions.expense_for', '=', 'U.id')
                ->leftJoin('users AS usr', 'transactions.created_by', '=', 'usr.id')
                ->leftJoin('contacts AS c', 'transactions.contact_id', '=', 'c.id')
                ->leftJoin(
                    'transaction_payments AS TP',
                    'transactions.id',
                    '=',
                    'TP.transaction_id'
                )
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'expense')
                ->select(
                    'transactions.id',
                    'transactions.document',
                    'transaction_date',
                    'ref_no',
                    'ec.name as category',
                    'payment_status',
                    'additional_notes',
                    'final_total',
                    'transactions.is_recurring',
                    'transactions.recur_interval',
                    'transactions.recur_interval_type',
                    'transactions.recur_repetitions',
                    'transactions.subscription_repeat_on',
                    'bl.name as location_name',
                    DB::raw("CONCAT(COALESCE(U.surname, ''),' ',COALESCE(U.first_name, ''),' ',COALESCE(U.last_name,'')) as expense_for"),
                    DB::raw("CONCAT(tr.name ,' (', tr.amount ,' )') as tax"),
                    DB::raw('SUM(TP.amount) as amount_paid'),
                    DB::raw("CONCAT(COALESCE(usr.surname, ''),' ',COALESCE(usr.first_name, ''),' ',COALESCE(usr.last_name,'')) as added_by"),
                    'transactions.recur_parent_id',
                    'c.name as contact_name'
                )
                ->with(['recurring_parent'])
                ->groupBy('transactions.id');

            // Apply filters
            if ($request->has('expense_for') && !empty($request->get('expense_for'))) {
                $expenses->where('transactions.expense_for', $request->get('expense_for'));
            }

            if ($request->has('contact_id') && !empty($request->get('contact_id'))) {
                $expenses->where('transactions.contact_id', $request->get('contact_id'));
            }

            if ($request->has('location_id') && !empty($request->get('location_id'))) {
                $expenses->where('transactions.location_id', $request->get('location_id'));
            }

            if ($request->has('expense_category_id') && !empty($request->get('expense_category_id'))) {
                $expenses->where('transactions.expense_category_id', $request->get('expense_category_id'));
            }

            if (!empty($request->start_date) && !empty($request->end_date)) {
                $expenses->whereDate('transaction_date', '>=', $request->start_date)
                    ->whereDate('transaction_date', '<=', $request->end_date);
            }

            if ($request->has('payment_status') && !empty($request->get('payment_status'))) {
                $expenses->where('transactions.payment_status', $request->get('payment_status'));
            }

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $expenses->whereIn('transactions.location_id', $permitted_locations);
            }

            $is_admin = $this->moduleUtil->is_admin(auth()->user(), $business_id);
            if (!$is_admin && auth()->user()->can('view_own_expense')) {
                $expenses->where('transactions.created_by', $request->session()->get('user.id'));
            }

            // Pagination
            $per_page = $request->get('per_page', 15);
            $page = $request->get('page', 1);

            $result = $expenses->paginate($per_page, ['*'], 'page', $page);

            // Transform data
            $result->getCollection()->transform(function ($item) {
                $item->formatted_transaction_date = format_datetime($item->transaction_date);
                $item->formatted_final_total = number_format($item->final_total, 2);
                $item->payment_due = $item->final_total - ($item->amount_paid ?? 0);
                $item->formatted_payment_due = number_format($item->payment_due, 2);
                $item->payment_status_label = __('lang_v1.' . $item->payment_status);

                // Recurring details
                $item->recur_details = $this->getRecurringDetails($item);
                $item->ref_no_formatted = $this->formatRefNo($item);

                return $item;
            });

            return $this->sendResponse($result, 'Expenses retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while retrieving expenses.', [], 500);
        }
    }

    /**
     * Get data for creating a new expense.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create(Request $request)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = auth()->user()->business_id;

            // Check if subscribed or not
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 403);
            }

            $business_locations = BusinessLocation::forDropdown($business_id, false, true);
            $bl_attributes = $business_locations['attributes'];
            $business_locations = $business_locations['locations'];

            $expense_categories = ExpenseCategory::where('business_id', $business_id)->pluck('name', 'id');
            $users = User::forDropdown($business_id, true, true);
            $taxes = TaxRate::forBusinessDropdown($business_id, true, true);
            $payment_line = $this->dummyPaymentLine;
            $payment_types = $this->transactionUtil->payment_types();
            $contacts = Contact::contactDropdown($business_id, false, false);

            // Accounts
            $accounts = [];
            if ($this->moduleUtil->isModuleEnabled('account')) {
                $accounts = Account::forDropdown($business_id, true, false, true);
            }

            $data = [
                'expense_categories' => $expense_categories,
                'business_locations' => $business_locations,
                'bl_attributes' => $bl_attributes,
                'users' => $users,
                'taxes' => $taxes,
                'payment_line' => $payment_line,
                'payment_types' => $payment_types,
                'contacts' => $contacts,
                'accounts' => $accounts
            ];

            return $this->sendResponse($data, 'Expense creation data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = auth()->user()->business_id;

            // Check if subscribed or not
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 403);
            }

            // Validate document size
            $request->validate([
                'document' => 'file|max:' . (config('constants.document_size_limit') / 1000)
            ]);

            $user_id = $user_id = auth()->user()->id;

            DB::beginTransaction();

            $expense = $this->transactionUtil->createExpense($request, $business_id, $user_id);

            DB::commit();

            return $this->sendResponse($expense, __('expense.expense_add_success'));
        } catch (\Exception $e) {
            DB::rollBack();

            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            return $this->sendError(__('messages.something_went_wrong'), [], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        if (!auth()->user()->can('expense.access') && !auth()->user()->can('view_own_expense')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $expense = Transaction::with(['expense_category', 'location', 'tax_rate', 'contact'])
                ->where('business_id', $business_id)
                ->where('id', $id)
                ->where('type', 'expense')
                ->first();

            if (!$expense) {
                return $this->sendError('Expense not found.', [], 404);
            }

            // Check if user can view this expense
            $is_admin = $this->moduleUtil->is_admin(auth()->user(), $business_id);
            if (!$is_admin && auth()->user()->can('view_own_expense') && $expense->created_by != request()->session()->get('user.id')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            return $this->sendResponse($expense, 'Expense details retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while retrieving expense details.', [], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            // Check if subscribed or not
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 403);
            }

            $business_locations = BusinessLocation::forDropdown($business_id);
            $expense_categories = ExpenseCategory::where('business_id', $business_id)->pluck('name', 'id');
            $expense = Transaction::where('business_id', $business_id)->where('id', $id)->first();

            if (!$expense) {
                return $this->sendError('Expense not found.', [], 404);
            }

            $users = User::forDropdown($business_id, true, true);
            $taxes = TaxRate::forBusinessDropdown($business_id, true, true);
            $contacts = Contact::contactDropdown($business_id, false, false);

            $data = [
                'expense' => $expense,
                'expense_categories' => $expense_categories,
                'business_locations' => $business_locations,
                'users' => $users,
                'taxes' => $taxes,
                'contacts' => $contacts
            ];

            return $this->sendResponse($data, 'Expense edit data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            // Validate document size
            $request->validate([
                'document' => 'file|max:' . (config('constants.document_size_limit') / 1000)
            ]);

            $business_id = auth()->user()->business_id;

            // Check if subscribed or not
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 403);
            }

            $expense = $this->transactionUtil->updateExpense($request, $id, $business_id);

            return $this->sendResponse($expense, __('expense.expense_update_success'));
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            return $this->sendError(__('messages.something_went_wrong'), [], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $expense = Transaction::where('business_id', $business_id)
                ->where('type', 'expense')
                ->where('id', $id)
                ->first();

            if (!$expense) {
                return $this->sendError('Expense not found.', [], 404);
            }

            $expense->delete();

            // Delete account transactions
            AccountTransaction::where('transaction_id', $expense->id)->delete();

            return $this->sendResponse(null, __("expense.expense_delete_success"));
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            return $this->sendError(__("messages.something_went_wrong"), [], 500);
        }
    }

    /**
     * Get recurring details for expense
     *
     * @param object $row
     * @return string
     */
    private function getRecurringDetails($row)
    {
        $details = '';
        if ($row->is_recurring == 1) {
            $type = $row->recur_interval == 1 ? Str::singular(__('lang_v1.' . $row->recur_interval_type)) : __('lang_v1.' . $row->recur_interval_type);
            $recur_interval = $row->recur_interval . $type;

            $details .= __('lang_v1.recur_interval') . ': ' . $recur_interval;
            if (!empty($row->recur_repetitions)) {
                $details .= ', ' . __('lang_v1.no_of_repetitions') . ': ' . $row->recur_repetitions;
            }
            if ($row->recur_interval_type == 'months' && !empty($row->subscription_repeat_on)) {
                $details .= ' ' . __('lang_v1.repeat_on') . ': ' . str_ordinal($row->subscription_repeat_on);
            }
        } elseif (!empty($row->recur_parent_id)) {
            $details .= __('lang_v1.recurred_from') . ': ' . $row->recurring_parent->ref_no;
        }
        return $details;
    }

    /**
     * Format reference number with recurring indicators
     *
     * @param object $row
     * @return string
     */
    private function formatRefNo($row)
    {
        $ref_no = $row->ref_no;
        $badges = [];

        if (!empty($row->is_recurring)) {
            $badges[] = [
                'type' => 'recurring',
                'title' => __('lang_v1.recurring_expense')
            ];
        }

        if (!empty($row->recur_parent_id)) {
            $badges[] = [
                'type' => 'generated_recurring',
                'title' => __('lang_v1.generated_recurring_expense')
            ];
        }

        return [
            'ref_no' => $ref_no,
            'badges' => $badges
        ];
    }
}
