<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\ExpenseCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ExpenseCategoryController extends BaseController
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = auth()->user()->business_id;

            $expense_categories = ExpenseCategory::where('business_id', $business_id)
                ->select(['name', 'code', 'id']);

            // Pagination
            $per_page = $request->get('per_page', 15);
            $page = $request->get('page', 1);

            $result = $expense_categories->paginate($per_page, ['*'], 'page', $page);

            return $this->sendResponse($result, 'Expense categories retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong while retrieving expense categories.', [], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create()
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        return $this->sendResponse([], 'Expense category creation form data.');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $input = $request->only(['name', 'code']);
            $input['business_id'] = auth()->user()->business_id;

            $expense_category = ExpenseCategory::create($input);

            return $this->sendResponse($expense_category, __("expense.added_success"));
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            return $this->sendError(__("messages.something_went_wrong"), [], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $expense_category = ExpenseCategory::where('business_id', $business_id)->find($id);

            if (!$expense_category) {
                return $this->sendError('Expense category not found.', [], 404);
            }

            return $this->sendResponse($expense_category, 'Expense category retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $expense_category = ExpenseCategory::where('business_id', $business_id)->find($id);

            if (!$expense_category) {
                return $this->sendError('Expense category not found.', [], 404);
            }

            return $this->sendResponse($expense_category, 'Expense category edit data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $input = $request->only(['name', 'code']);
            $business_id = auth()->user()->business_id;

            $expense_category = ExpenseCategory::where('business_id', $business_id)->findOrFail($id);
            $expense_category->name = $input['name'];
            $expense_category->code = $input['code'];
            $expense_category->save();

            return $this->sendResponse($expense_category, __("expense.updated_success"));
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            return $this->sendError(__("messages.something_went_wrong"), [], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('expense.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $expense_category = ExpenseCategory::where('business_id', $business_id)->findOrFail($id);
            $expense_category->delete();

            return $this->sendResponse(null, __("expense.deleted_success"));
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            return $this->sendError(__("messages.something_went_wrong"), [], 500);
        }
    }
}
