<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController as ApiBaseController;
use Illuminate\Http\Request;
use App\Models\DocumentAndNote;
use App\Models\Media;
use App\Utils\ModuleUtil;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\JsonResponse;
use Exception;
use Illuminate\Support\Facades\Log;

class DocumentAndNoteController extends ApiBaseController
{
    protected $moduleUtil;

    public function __construct(ModuleUtil $moduleUtil)
    {
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of documents and notes
     *
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $user_id = $user_id = auth()->user()->id;
            $notable_id = request()->get('notable_id');
            $notable_type = request()->get('notable_type');

            $permissions = $this->getPermission($business_id, $notable_id, $notable_type);

            if (!empty($permissions) && in_array('view', $permissions)) {
                $documents_notes = DocumentAndNote::where('business_id', $business_id)
                    ->where('notable_id', $notable_id)
                    ->where(function ($query) use ($user_id) {
                        $query->where('is_private', 0)
                            ->orWhere(function ($q) use ($user_id) {
                                $q->where('is_private', 1)
                                    ->where('created_by', $user_id);
                            });
                    })
                    ->where('notable_type', $notable_type)
                    ->with('createdBy', 'media')
                    ->get();

                return $this->sendResponse([
                    'documents_notes' => $documents_notes,
                    'permissions' => $permissions
                ], 'Liste des documents et notes récupérée avec succès');
            }

            return $this->sendError('Permission refusée', [], 403);
        } catch (Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Get form data for creating document/note
     *
     * @return JsonResponse
     */
    public function create(): JsonResponse
    {
        try {
            $notable_id = request()->get('notable_id');
            $notable_type = request()->get('notable_type');

            return $this->sendResponse([
                'notable_id' => $notable_id,
                'notable_type' => $notable_type
            ], 'Données de formulaire récupérées avec succès');
        } catch (Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Store a newly created document/note
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $notable_id = request()->get('notable_id');
            $notable_type = request()->get('notable_type');

            $input = $request->only('heading', 'description', 'is_private');
            $input['business_id'] = auth()->user()->business_id;
            $input['created_by'] = $user_id = auth()->user()->id;

            DB::beginTransaction();

            if (!empty($input['is_private'])) {
                activity()->disableLogging();
            }

            $model = $notable_type::where('business_id', $input['business_id'])
                ->findOrFail($notable_id);

            $model_note = $model->documentsAndnote()->create($input);

            if (!empty($request->get('file_names'))) {
                $file_names = is_array($request->get('file_names'))
                    ? $request->get('file_names')
                    : explode(',', $request->get('file_names')[0]);

                $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
                Media::attachMediaToModel($model_note, $business_id, $file_names);
            }

            DB::commit();

            return $this->sendResponse($model_note, 'Document/Note créé avec succès');
        } catch (Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Display the specified document/note
     *
     * @param int $id
     * @return JsonResponse
     */
    public function show(int $id): JsonResponse
    {
        try {
            $notable_id = request()->get('notable_id');
            $notable_type = request()->get('notable_type');

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $document_note = DocumentAndNote::where('business_id', $business_id)
                ->where('notable_id', $notable_id)
                ->where('notable_type', $notable_type)
                ->with('media', 'createdBy')
                ->findOrFail($id);

            return $this->sendResponse($document_note, 'Document/Note récupéré avec succès');
        } catch (Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Document/Note non trouvé', [], 404);
        }
    }

    /**
     * Get form data for editing document/note
     *
     * @param int $id
     * @return JsonResponse
     */
    public function edit(int $id): JsonResponse
    {
        try {
            $notable_id = request()->get('notable_id');
            $notable_type = request()->get('notable_type');

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $document_note = DocumentAndNote::where('business_id', $business_id)
                ->where('notable_id', $notable_id)
                ->where('notable_type', $notable_type)
                ->with('media')
                ->findOrFail($id);

            return $this->sendResponse([
                'document_note' => $document_note,
                'notable_id' => $notable_id,
                'notable_type' => $notable_type
            ], 'Données de modification récupérées avec succès');
        } catch (Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Document/Note non trouvé', [], 404);
        }
    }

    /**
     * Update the specified document/note
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function update(Request $request, int $id): JsonResponse
    {
        try {
            $notable_id = request()->get('notable_id');
            $notable_type = request()->get('notable_type');

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $input = $request->only('heading', 'description');
            $input['is_private'] = !empty($request->get('is_private')) ? 1 : 0;

            $document_note = DocumentAndNote::where('business_id', $business_id)
                ->where('notable_id', $notable_id)
                ->where('notable_type', $notable_type)
                ->findOrFail($id);

            DB::beginTransaction();

            if ($input['is_private']) {
                $document_note->disableLogging();
            }

            $document_note->update($input);

            if (!empty($request->get('file_names'))) {
                $file_names = is_array($request->get('file_names'))
                    ? $request->get('file_names')
                    : explode(',', $request->get('file_names')[0]);

                $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
                Media::attachMediaToModel($document_note, $business_id, $file_names);
            }

            DB::commit();

            return $this->sendResponse($document_note, 'Document/Note mis à jour avec succès');
        } catch (Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Remove the specified document/note
     *
     * @param int $id
     * @return JsonResponse
     */
    public function destroy(int $id): JsonResponse
    {
        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $notable_id = request()->get('notable_id');
            $notable_type = request()->get('notable_type');

            $document_note = DocumentAndNote::where('business_id', $business_id)
                ->where('notable_id', $notable_id)
                ->where('notable_type', $notable_type)
                ->findOrFail($id);

            DB::beginTransaction();

            $document_note->delete();
            $document_note->media()->delete();

            DB::commit();

            return $this->sendResponse(null, 'Document/Note supprimé avec succès');
        } catch (Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Upload media files
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function uploadMedia(Request $request): JsonResponse
    {
        try {
            $file = $request->file('file')[0];
            $file_name = Media::uploadFile($file);

            return $this->sendResponse([
                'file_name' => $file_name
            ], 'Fichier uploadé avec succès');
        } catch (Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Erreur lors de l\'upload', [], 500);
        }
    }

    /**
     * Get permissions for notable
     *
     * @param int $business_id
     * @param int $notable_id
     * @param string $notable_type
     * @return array
     */
    private function getPermission($business_id, $notable_id, $notable_type): array
    {
        $app_notable = [
            'App\User' => [
                'permissions' => ['view', 'create', 'delete']
            ],
            'App\Contact' => [
                'permissions' => ['view', 'create', 'delete']
            ]
        ];

        if (isset($app_notable[$notable_type])) {
            return $app_notable[$notable_type]['permissions'];
        } else {
            $module_parameters = [
                'business_id' => $business_id,
                'notable_id' => $notable_id,
                'notable_type' => $notable_type
            ];
            $module_data = $this->moduleUtil->getModuleData('addDocumentAndNotes', $module_parameters);

            foreach ($module_data as $module => $data) {
                if (isset($data[$notable_type])) {
                    return $data[$notable_type]['permissions'];
                }
            }
        }

        return [];
    }
}
