<?php

namespace App\Http\Controllers;


use App\Http\Controllers\Api\BaseController;
use App\Models\Brand;
use Illuminate\Http\Request;
use App\Models\Discount;
use App\Models\Category;
use App\Models\Brands;
use App\Models\BusinessLocation;
use App\Utils\Util;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class DiscountController extends BaseController
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of discounts
     *
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        try {
            if (!auth()->user()->can('discount.access')) {
                return $this->sendError('Action non autorisée', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $discounts = Discount::where('discounts.business_id', $business_id)
                ->leftjoin('brands as b', 'discounts.brand_id', '=', 'b.id')
                ->leftjoin('categories as c', 'discounts.category_id', '=', 'c.id')
                ->leftjoin('business_locations as l', 'discounts.location_id', '=', 'l.id')
                ->select([
                    'discounts.id',
                    'discounts.name',
                    'starts_at',
                    'ends_at',
                    'priority',
                    'discount_type',
                    'discount_amount',
                    'b.name as brand',
                    'c.name as category',
                    'l.name as location',
                    'discounts.is_active',
                    'discounts.applicable_in_spg',
                    'discounts.applicable_in_cg'
                ])
                ->get();

            // Format dates
            $discounts->transform(function ($discount) {
                $discount->starts_at = !empty($discount->starts_at)
                    ? $this->commonUtil->format_date($discount->starts_at, true)
                    : null;
                $discount->ends_at = !empty($discount->ends_at)
                    ? $this->commonUtil->format_date($discount->ends_at, true)
                    : null;
                return $discount;
            });

            return $this->sendResponse($discounts, 'Liste des remises récupérée avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Get form data for creating discount
     *
     * @return JsonResponse
     */
    public function create(): JsonResponse
    {
        try {
            if (!auth()->user()->can('discount.access')) {
                return $this->sendError('Action non autorisée', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $categories = Category::where('business_id', $business_id)
                ->where('parent_id', 0)
                ->pluck('name', 'id');

            $brands = Brand::forDropdown($business_id);
            $locations = BusinessLocation::forDropdown($business_id);

            $data = [
                'categories' => $categories,
                'brands' => $brands,
                'locations' => $locations
            ];

            return $this->sendResponse($data, 'Données de formulaire récupérées avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Store a newly created discount
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function store(Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('discount.access')) {
                return $this->sendError('Action non autorisée', [], 403);
            }

            $input = $request->only([
                'name',
                'brand_id',
                'category_id',
                'location_id',
                'priority',
                'discount_type',
                'discount_amount'
            ]);

            $business_id = auth()->user()->business_id;
            $input['business_id'] = $business_id;

            $input['starts_at'] = $request->has('starts_at')
                ? $this->commonUtil->uf_date($request->input('starts_at'), true)
                : null;
            $input['ends_at'] = $request->has('ends_at')
                ? $this->commonUtil->uf_date($request->input('ends_at'), true)
                : null;

            $checkboxes = ['is_active', 'applicable_in_spg', 'applicable_in_cg'];
            foreach ($checkboxes as $checkbox) {
                $input[$checkbox] = $request->has($checkbox) ? 1 : 0;
            }

            $discount = Discount::create($input);

            return $this->sendResponse($discount, 'Remise créée avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Display the specified discount
     *
     * @param int $id
     * @return JsonResponse
     */
    public function show(int $id): JsonResponse
    {
        try {
            if (!auth()->user()->can('discount.access')) {
                return $this->sendError('Action non autorisée', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $discount = Discount::where('business_id', $business_id)
                ->with(['brand', 'category', 'location'])
                ->findOrFail($id);

            return $this->sendResponse($discount, 'Remise récupérée avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Remise non trouvée', [], 404);
        }
    }

    /**
     * Get form data for editing discount
     *
     * @param int $id
     * @return JsonResponse
     */
    public function edit(int $id): JsonResponse
    {
        try {
            if (!auth()->user()->can('discount.access')) {
                return $this->sendError('Action non autorisée', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $discount = Discount::where('business_id', $business_id)->findOrFail($id);

            $categories = Category::where('business_id', $business_id)
                ->where('parent_id', 0)
                ->pluck('name', 'id');

            $brands = Brand::forDropdown($business_id);
            $locations = BusinessLocation::forDropdown($business_id);

            $data = [
                'discount' => $discount,
                'categories' => $categories,
                'brands' => $brands,
                'locations' => $locations
            ];

            return $this->sendResponse($data, 'Données de modification récupérées avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Remise non trouvée', [], 404);
        }
    }

    /**
     * Update the specified discount
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function update(Request $request, int $id): JsonResponse
    {
        try {
            if (!auth()->user()->can('discount.access')) {
                return $this->sendError('Action non autorisée', [], 403);
            }

            $input = $request->only([
                'name',
                'brand_id',
                'category_id',
                'location_id',
                'priority',
                'discount_type',
                'discount_amount'
            ]);

            $business_id = auth()->user()->business_id;

            $input['starts_at'] = $request->has('starts_at')
                ? $this->commonUtil->uf_date($request->input('starts_at'), true)
                : null;
            $input['ends_at'] = $request->has('ends_at')
                ? $this->commonUtil->uf_date($request->input('ends_at'), true)
                : null;

            $checkboxes = ['is_active', 'applicable_in_spg', 'applicable_in_cg'];
            foreach ($checkboxes as $checkbox) {
                $input[$checkbox] = $request->has($checkbox) ? 1 : 0;
            }

            $discount = Discount::where('business_id', $business_id)
                ->where('id', $id)
                ->firstOrFail();

            $discount->update($input);

            return $this->sendResponse($discount, 'Remise mise à jour avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Remove the specified discount
     *
     * @param int $id
     * @return JsonResponse
     */
    public function destroy(int $id): JsonResponse
    {
        try {
            if (!auth()->user()->can('discount.access')) {
                return $this->sendError('Action non autorisée', [], 403);
            }

            $business_id = request()->user()->business_id;
            $discount = Discount::where('business_id', $business_id)->findOrFail($id);
            $discount->delete();

            return $this->sendResponse(null, 'Remise supprimée avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Mass deactivate discounts
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function massDeactivate(Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('discount.access')) {
                return $this->sendError('Action non autorisée', [], 403);
            }

            if (!empty($request->input('selected_discounts'))) {
                $business_id = auth()->user()->business_id;
                $selected_discounts = is_array($request->input('selected_discounts'))
                    ? $request->input('selected_discounts')
                    : explode(',', $request->input('selected_discounts'));

                DB::beginTransaction();

                Discount::where('business_id', $business_id)
                    ->whereIn('id', $selected_discounts)
                    ->update(['is_active' => 0]);

                DB::commit();
            }

            return $this->sendResponse(null, 'Remises désactivées avec succès');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }

    /**
     * Activate the specified discount
     *
     * @param int $id
     * @return JsonResponse
     */
    public function activate(int $id): JsonResponse
    {
        try {
            if (!auth()->user()->can('discount.access')) {
                return $this->sendError('Action non autorisée', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            Discount::where('id', $id)
                ->where('business_id', $business_id)
                ->update(['is_active' => 1]);

            return $this->sendResponse(null, 'Remise activée avec succès');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Une erreur est survenue', [], 500);
        }
    }
}
