<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Contact;
use App\Models\CustomerGroup;
use App\Utils\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\DataTables;

class CustomerGroupController extends BaseController
{
    /**
     * All Utils instance.
     */
    protected $commonUtil;

    /**
     * Constructor
     *
     * @param Util $commonUtil
     * @return void
     */
    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of customer groups.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            if (!auth()->user()->can('customer.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $query = CustomerGroup::where('business_id', $business_id)
                ->select(['id', 'name', 'amount', 'created_at', 'updated_at']);

            // Apply search filter
            if ($request->filled('search')) {
                $search = $request->get('search');
                $query->where('name', 'like', "%{$search}%");
            }

            // Apply pagination
            $per_page = $request->get('per_page', 15);
            $customer_groups = $query->paginate($per_page);

            // Transform data
            $transformed_groups = $customer_groups->map(function ($group) {
                return [
                    'id' => $group->id,
                    'name' => $group->name,
                    'amount' => $group->amount,
                    'formatted_amount' => $this->commonUtil->num_f($group->amount, true),
                    'created_at' => $group->created_at,
                    'updated_at' => $group->updated_at,
                ];
            });

            $response_data = [
                'customer_groups' => $transformed_groups,
                'pagination' => [
                    'current_page' => $customer_groups->currentPage(),
                    'last_page' => $customer_groups->lastPage(),
                    'per_page' => $customer_groups->perPage(),
                    'total' => $customer_groups->total(),
                ]
            ];

            return $this->sendResponse($response_data, 'Customer groups retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Display a listing of customer groups for DataTables.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function datatable(Request $request)
    {
        try {
            if (!auth()->user()->can('customer.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $customer_groups = CustomerGroup::where('business_id', $business_id)
                ->select(['name', 'amount', 'id']);

            return DataTables::of($customer_groups)
                ->addColumn('action', function ($group) {
                    $actions = [];

                    if (auth()->user()->can('customer.update')) {
                        $actions[] = [
                            'type' => 'edit',
                            'url' => route('api.customer-groups.show', $group->id),
                            'label' => 'Edit'
                        ];
                    }

                    if (auth()->user()->can('customer.delete')) {
                        $actions[] = [
                            'type' => 'delete',
                            'url' => route('api.customer-groups.destroy', $group->id),
                            'label' => 'Delete'
                        ];
                    }

                    return $actions;
                })
                ->editColumn('amount', function ($group) {
                    return $this->commonUtil->num_f($group->amount, true);
                })
                ->removeColumn('id')
                ->make(true);
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Store a newly created customer group.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        //try {
            if (!auth()->user()->can('customer.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            // Validation rules
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'amount' => 'nullable|numeric|min:0'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $input = $request->only(['name', 'amount']);
            $input['business_id'] = auth()->user()->business_id;
            $input['created_by'] = $user_id = auth()->user()->id;
            $input['amount'] = !empty($input['amount']) ? $this->commonUtil->num_uf($input['amount']) : 0;

            $customer_group = CustomerGroup::create($input);

            $group_data = [
                'id' => $customer_group->id,
                'name' => $customer_group->name,
                'amount' => $customer_group->amount,
                'formatted_amount' => $this->commonUtil->num_f($customer_group->amount, true),
                'created_at' => $customer_group->created_at,
                'updated_at' => $customer_group->updated_at,
            ];

            return $this->sendResponse($group_data, 'Customer group created successfully.');
        //} catch (\Exception $e) {
        //    Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
        //    return $this->sendError('Something went wrong.', [], 500);
        //}
    }

    /**
     * Display the specified customer group.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id, Request $request)
    {
        try {
            if (!auth()->user()->can('customer.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $customer_group = CustomerGroup::where('business_id', $business_id)->find($id);

            if (!$customer_group) {
                return $this->sendError('Customer group not found.', [], 404);
            }

            $group_data = [
                'id' => $customer_group->id,
                'name' => $customer_group->name,
                'amount' => $customer_group->amount,
                'formatted_amount' => $this->commonUtil->num_f($customer_group->amount, true),
                'created_at' => $customer_group->created_at,
                'updated_at' => $customer_group->updated_at,
            ];

            return $this->sendResponse($group_data, 'Customer group retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Update the specified customer group.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('customer.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            // Validation rules
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'amount' => 'nullable|numeric|min:0'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $input = $request->only(['name', 'amount']);
            $business_id = auth()->user()->business_id;

            $input['amount'] = !empty($input['amount']) ? $this->commonUtil->num_uf($input['amount']) : 0;

            $customer_group = CustomerGroup::where('business_id', $business_id)->find($id);

            if (!$customer_group) {
                return $this->sendError('Customer group not found.', [], 404);
            }

            $customer_group->name = $input['name'];
            $customer_group->amount = $input['amount'];
            $customer_group->save();

            $group_data = [
                'id' => $customer_group->id,
                'name' => $customer_group->name,
                'amount' => $customer_group->amount,
                'formatted_amount' => $this->commonUtil->num_f($customer_group->amount, true),
                'created_at' => $customer_group->created_at,
                'updated_at' => $customer_group->updated_at,
            ];

            return $this->sendResponse($group_data, 'Customer group updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Remove the specified customer group.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id, Request $request)
    {
        try {
            if (!auth()->user()->can('customer.delete')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id;

            $customer_group = CustomerGroup::where('business_id', $business_id)->find($id);

            if (!$customer_group) {
                return $this->sendError('Customer group not found.', [], 404);
            }

            // Check if any contacts are using this customer group
            $contacts_count = Contact::where('business_id', $business_id)
                ->where('customer_group_id', $id)
                ->count();

            if ($contacts_count > 0) {
                return $this->sendError("Cannot delete this customer group. {$contacts_count} contact(s) are using this group.", [], 400);
            }

            $customer_group->delete();

            return $this->sendResponse([], 'Customer group deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Get customer groups for dropdown/select.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function dropdown(Request $request)
    {
        try {
            if (!auth()->user()->can('customer.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $customer_groups = CustomerGroup::where('business_id', $business_id)
                ->select(['id', 'name', 'amount'])
                ->orderBy('name')
                ->get()
                ->map(function ($group) {
                    return [
                        'id' => $group->id,
                        'name' => $group->name,
                        'amount' => $group->amount,
                        'formatted_amount' => $this->commonUtil->num_f($group->amount, true)
                    ];
                });

            return $this->sendResponse($customer_groups, 'Customer groups dropdown retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Get customer groups statistics.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function statistics(Request $request)
    {
        try {
            if (!auth()->user()->can('customer.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            $stats = CustomerGroup::where('customer_groups.business_id', $business_id)
                ->leftJoin('contacts', function ($join) use ($business_id) {
                    $join->on('customer_groups.id', '=', 'contacts.customer_group_id')
                        ->where('contacts.business_id', $business_id)
                        ->whereIn('contacts.type', ['customer', 'both']);
                })
                ->select([
                    'customer_groups.id',
                    'customer_groups.name',
                    'customer_groups.amount',
                    DB::raw('COUNT(contacts.id) as customer_count')
                ])
                ->groupBy('customer_groups.id', 'customer_groups.name', 'customer_groups.amount')
                ->orderBy('customer_groups.name')
                ->get()
                ->map(function ($group) {
                    return [
                        'id' => $group->id,
                        'name' => $group->name,
                        'amount' => $group->amount,
                        'formatted_amount' => $this->commonUtil->num_f($group->amount, true),
                        'customer_count' => (int) $group->customer_count
                    ];
                });

            $total_groups = $stats->count();
            $total_customers = $stats->sum('customer_count');

            return $this->sendResponse([
                'groups' => $stats,
                'summary' => [
                    'total_groups' => $total_groups,
                    'total_customers_in_groups' => $total_customers
                ]
            ], 'Customer groups statistics retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }
}
