<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Contact;
use App\Models\Transaction;
use App\Models\User;
use App\Utils\ContactUtil;
use App\Utils\ModuleUtil;
use App\Utils\NotificationUtil;
use App\Utils\TransactionUtil;
use App\Utils\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class ContactController extends BaseController
{
    protected $commonUtil;
    protected $contactUtil;
    protected $transactionUtil;
    protected $moduleUtil;
    protected $notificationUtil;

    /**
     * Constructor
     */
    public function __construct(
        Util $commonUtil,
        ModuleUtil $moduleUtil,
        TransactionUtil $transactionUtil,
        NotificationUtil $notificationUtil,
        ContactUtil $contactUtil
    ) {
        $this->commonUtil = $commonUtil;
        $this->contactUtil = $contactUtil;
        $this->moduleUtil = $moduleUtil;
        $this->transactionUtil = $transactionUtil;
        $this->notificationUtil = $notificationUtil;
    }

    /**
     * Display a listing of contacts.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        //try {
            $type = $request->get('type', 'customer');
            $types = ['supplier', 'customer', 'both'];

            if (!in_array($type, $types)) {
                return $this->sendError('Invalid contact type. Must be: supplier, customer, or both.', [], 422);
            }

            // Check permissions
            if ($type == 'supplier' && !auth()->user()->can('supplier.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }
            if ($type == 'customer' && !auth()->user()->can('customer.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $query = $this->contactUtil->getContactQuery($business_id, $type);

            // Apply search filters
            if ($request->filled('search')) {
                $search = $request->get('search');
                $query->where(function ($q) use ($search) {
                    $q->where('contacts.name', 'like', "%{$search}%")
                        ->orWhere('contacts.mobile', 'like', "%{$search}%")
                        ->orWhere('contacts.email', 'like', "%{$search}%")
                        ->orWhere('contacts.contact_id', 'like', "%{$search}%");
                });
            }

            // Apply pagination
            $per_page = $request->get('per_page', 15);
            $contacts = $query->paginate($per_page);

        // Transform data
        $transformed_contacts = $contacts->map(function ($contact) use ($type, $request) {
                $data = [
                    'id' => $contact->id,
                    'type' => $contact->type,
                    'name' => $contact->name,
                    'contact_id' => $contact->contact_id,
                    'email' => $contact->email,
                    'mobile' => $contact->mobile,
                    'address' => $this->formatAddress($contact),
                    'contact_status' => $contact->contact_status,
                    'opening_balance' => $contact->opening_balance ?? 0,
                    'balance' => $contact->balance ?? 0,
                    'customer_group' => $contact->customer_group,
                    'created_at' => $contact->created_at
                ];

                if (in_array($type, ['supplier', 'both'])) {
                    $data['total_purchase'] = $contact->total_purchase ?? 0;
                    $data['purchase_paid'] = $contact->purchase_paid ?? 0;
                    $data['purchase_due'] = ($contact->total_purchase ?? 0) - ($contact->purchase_paid ?? 0);
                    $data['total_purchase_return'] = $contact->total_purchase_return ?? 0;
                    $data['purchase_return_paid'] = $contact->purchase_return_paid ?? 0;
                    $data['purchase_return_due'] = ($contact->total_purchase_return ?? 0) - ($contact->purchase_return_paid ?? 0);
                }
                $business = auth()->user()->busines;
                $enable_rp = $business->enable_rp ?? 0;

                if (in_array($type, ['customer', 'both'])) {
                    $data['total_invoice'] = $contact->total_invoice ?? 0;
                    $data['invoice_received'] = $contact->invoice_received ?? 0;
                    $data['sale_due'] = ($contact->total_invoice ?? 0) - ($contact->invoice_received ?? 0);
                    $data['total_sell_return'] = $contact->total_sell_return ?? 0;
                    $data['sell_return_paid'] = $contact->sell_return_paid ?? 0;
                    $data['sell_return_due'] = ($contact->total_sell_return ?? 0) - ($contact->sell_return_paid ?? 0);
                    $data['credit_limit'] = $contact->credit_limit;
                    $data['pay_term_number'] = $contact->pay_term_number;
                    $data['pay_term_type'] = $contact->pay_term_type;

                    if ($enable_rp == 1) {
                        $data['total_rp'] = $contact->total_rp ?? 0;
                    }
                }

                return $data;
            });

            $response_data = [
                'contacts' => $transformed_contacts,
                'pagination' => [
                    'current_page' => $contacts->currentPage(),
                    'last_page' => $contacts->lastPage(),
                    'per_page' => $contacts->perPage(),
                    'total' => $contacts->total(),
                ]
            ];

            return $this->sendResponse($response_data, 'Contacts retrieved successfully.');
        //} catch (\Exception $e) {
        //    Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
        //    return $this->sendError('Something went wrong.', [], 500);
        //}
    }

    /**
     * Store a newly created contact.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $type = $request->get('type', 'customer');

            if ($type == 'supplier' && !auth()->user()->can('supplier.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }
            if ($type == 'customer' && !auth()->user()->can('customer.create')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 403);
            }

            // Validation rules
            $rules = [
                'type' => 'required|in:supplier,customer,both',
                'first_name' => 'required|string|max:255',
                'middle_name' => 'nullable|string|max:255',
                'last_name' => 'nullable|string|max:255',
                'prefix' => 'nullable|string|max:10',
                'email' => 'nullable|email|unique:contacts,email',
                'mobile' => 'required|string|max:20',
                'contact_id' => 'nullable|string|unique:contacts,contact_id',
                'tax_number' => 'nullable|string|max:50',
                'pay_term_number' => 'nullable|integer|min:0',
                'pay_term_type' => 'nullable|in:days,months',
                'opening_balance' => 'nullable|numeric',
                'credit_limit' => 'nullable|numeric|min:0',
                'customer_group_id' => 'nullable|exists:customer_groups,id',
                'dob' => 'nullable|date',
                'address_line_1' => 'nullable|string',
                'address_line_2' => 'nullable|string',
                'city' => 'nullable|string',
                'state' => 'nullable|string',
                'country' => 'nullable|string',
                'zip_code' => 'nullable|string|max:10',
                'supplier_business_name' => 'nullable|string|max:255',
            ];

            if ($type == 'supplier') {
                $rules['supplier_business_name'] = 'required|string|max:255';
                //$rules['pay_term_number'] = 'required|integer|min:0';
                //$rules['pay_term_type'] = 'required|in:days,months';
            }

            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $input = $request->only([
                'type',
                'supplier_business_name',
                'prefix',
                'first_name',
                'middle_name',
                'last_name',
                'tax_number',
                'pay_term_number',
                'pay_term_type',
                'mobile',
                'landline',
                'alternate_number',
                'city',
                'state',
                'country',
                'address_line_1',
                'address_line_2',
                'customer_group_id',
                'zip_code',
                'contact_id',
                'custom_field1',
                'custom_field2',
                'custom_field3',
                'custom_field4',
                'email',
                'shipping_address',
                'position',
                'dob'
            ]);

            $input['name'] = trim(implode(' ', array_filter([
                $input['prefix'] ?? '',
                $input['first_name'] ?? '',
                $input['middle_name'] ?? '',
                $input['last_name'] ?? ''
            ])));

            if (!empty($input['dob'])) {
                $input['dob'] = $this->commonUtil->uf_date($input['dob']);
            }

            $input['business_id'] = $business_id;
            $input['created_by'] = $user_id = auth()->user()->id;
            $input['credit_limit'] = $request->input('credit_limit') != '' ? $this->commonUtil->num_uf($request->input('credit_limit')) : null;
            $input['opening_balance'] = $this->commonUtil->num_uf($request->input('opening_balance', 0));

            $result = $this->contactUtil->createNewContact($input);

            if ($result['success']) {
                $contact_data = $this->formatContactResponse($result['data']);
                return $this->sendResponse($contact_data, 'Contact created successfully.');
            } else {
                return $this->sendError($result['msg'], [], 400);
            }
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Display the specified contact.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id, Request $request)
    {
        try {
            if (!auth()->user()->can('supplier.view') && !auth()->user()->can('customer.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $contact = $this->contactUtil->getContactInfo($business_id, $id);

            if (!$contact) {
                return $this->sendError('Contact not found.', [], 404);
            }

            $contact_data = $this->formatContactResponse($contact);

            // Add financial summary
            $contact_data['financial_summary'] = [
                'total_purchase' => $contact->total_purchase ?? 0,
                'purchase_paid' => $contact->purchase_paid ?? 0,
                'purchase_due' => ($contact->total_purchase ?? 0) - ($contact->purchase_paid ?? 0),
                'total_invoice' => $contact->total_invoice ?? 0,
                'invoice_received' => $contact->invoice_received ?? 0,
                'sale_due' => ($contact->total_invoice ?? 0) - ($contact->invoice_received ?? 0),
                'opening_balance' => $contact->opening_balance ?? 0,
                'opening_balance_paid' => $contact->opening_balance_paid ?? 0,
            ];

            return $this->sendResponse($contact_data, 'Contact retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Update the specified contact.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            if (!auth()->user()->can('supplier.update') && !auth()->user()->can('customer.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;

            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 403);
            }

            // Validation rules
            $rules = [
                'type' => 'required|in:supplier,customer,both',
                'first_name' => 'required|string|max:255',
                'middle_name' => 'nullable|string|max:255',
                'last_name' => 'nullable|string|max:255',
                'prefix' => 'nullable|string|max:10',
                'email' => 'nullable|email|unique:contacts,email,' . $id,
                'mobile' => 'required|string|max:20',
                'contact_id' => 'nullable|string|unique:contacts,contact_id,' . $id,
                'tax_number' => 'nullable|string|max:50',
                'pay_term_number' => 'nullable|integer|min:0',
                'pay_term_type' => 'nullable|in:days,months',
                'opening_balance' => 'nullable|numeric',
                'credit_limit' => 'nullable|numeric|min:0',
                'customer_group_id' => 'nullable|exists:customer_groups,id',
                'dob' => 'nullable|date',
                'address_line_1' => 'nullable|string',
                'address_line_2' => 'nullable|string',
                'city' => 'nullable|string',
                'state' => 'nullable|string',
                'country' => 'nullable|string',
                'zip_code' => 'nullable|string|max:10',
                'supplier_business_name' => 'nullable|string|max:255',
            ];

            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {
                return $this->sendError('Validation Error.', $validator->errors(), 422);
            }

            $input = $request->only([
                'type',
                'supplier_business_name',
                'prefix',
                'first_name',
                'middle_name',
                'last_name',
                'tax_number',
                'pay_term_number',
                'pay_term_type',
                'mobile',
                'address_line_1',
                'address_line_2',
                'zip_code',
                'dob',
                'alternate_number',
                'city',
                'state',
                'country',
                'landline',
                'customer_group_id',
                'contact_id',
                'custom_field1',
                'custom_field2',
                'custom_field3',
                'custom_field4',
                'email',
                'shipping_address',
                'position'
            ]);

            $input['name'] = trim(implode(' ', array_filter([
                $input['prefix'] ?? '',
                $input['first_name'] ?? '',
                $input['middle_name'] ?? '',
                $input['last_name'] ?? ''
            ])));

            if (!empty($input['dob'])) {
                $input['dob'] = $this->commonUtil->uf_date($input['dob']);
            }

            $input['credit_limit'] = $request->input('credit_limit') != '' ? $this->commonUtil->num_uf($request->input('credit_limit')) : null;
            $input['opening_balance'] = $this->commonUtil->num_uf($request->input('opening_balance', 0));

            $result = $this->contactUtil->updateContact($input, $id, $business_id);

            if ($result['success']) {
                $contact_data = $this->formatContactResponse($result['data']);
                return $this->sendResponse($contact_data, 'Contact updated successfully.');
            } else {
                return $this->sendError($result['msg'], [], 400);
            }
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Remove the specified contact.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id, Request $request)
    {
        try {
            if (!auth()->user()->can('supplier.delete') && !auth()->user()->can('customer.delete')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $request->user()->business_id;

            // Check if any transaction related to this contact exists
            $count = Transaction::where('business_id', $business_id)
                ->where('contact_id', $id)
                ->count();
            if ($count == 0) {
                $contact = Contact::where('business_id', $business_id)->findOrFail($id);
                if (!$contact->is_default) {
                    $contact->delete();
                    return $this->sendResponse([], 'Contact deleted successfully.');
                } else {
                    return $this->sendError('Cannot delete default contact.', [], 400);
                }
            } else {
                return $this->sendError('You cannot delete this contact because it has associated transactions.', [], 400);
            }
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Get customers for dropdown/autocomplete.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCustomers(Request $request)
    {
        try {
            $term = $request->input('q', '');
            $business_id = auth()->user()->business_id;
            $user_id = $user_id = auth()->user()->id;

            $contacts = Contact::where('business_id', $business_id)->active();

            $selected_contacts = User::isSelectedContacts($user_id);
            if ($selected_contacts) {
                $contacts->join('user_contact_access AS uca', 'contacts.id', 'uca.contact_id')
                    ->where('uca.user_id', $user_id);
            }

            if (!empty($term)) {
                $contacts->where(function ($query) use ($term) {
                    $query->where('name', 'like', '%' . $term . '%')
                        ->orWhere('supplier_business_name', 'like', '%' . $term . '%')
                        ->orWhere('mobile', 'like', '%' . $term . '%')
                        ->orWhere('contacts.contact_id', 'like', '%' . $term . '%');
                });
            }

            $contacts = $contacts->select([
                'contacts.id',
                DB::raw("IF(contacts.contact_id IS NULL OR contacts.contact_id='', name, CONCAT(name, ' (', contacts.contact_id, ')')) AS name"),
                'mobile',
                'address_line_1',
                'city',
                'state',
                'pay_term_number',
                'pay_term_type',
                'balance'
            ])->onlyCustomers();

            if ($request->session()->get('business.enable_rp') == 1) {
                $contacts->addSelect('total_rp');
            }

            $contacts = $contacts->limit(20)->get();

            return $this->sendResponse($contacts, 'Customers retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Update contact status (active/inactive).
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateStatus($id, Request $request)
    {
        try {
            if (!auth()->user()->can('supplier.update') && !auth()->user()->can('customer.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = auth()->user()->business_id;
            $contact = Contact::where('business_id', $business_id)->findOrFail($id);

            $old_status = $contact->contact_status;
            $contact->contact_status = $contact->contact_status == 'active' ? 'inactive' : 'active';
            $contact->save();

            return $this->sendResponse([
                'id' => $contact->id,
                'old_status' => $old_status,
                'new_status' => $contact->contact_status
            ], 'Contact status updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Get contact ledger.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getLedger(Request $request)
    {
        try {
            if (!auth()->user()->can('supplier.view') && !auth()->user()->can('customer.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $contact_id = $request->input('contact_id');
            $start_date = $request->start_date;
            $end_date = $request->end_date;

            if (!$contact_id) {
                return $this->sendError('Contact ID is required.', [], 422);
            }

            $contact = Contact::find($contact_id);
            if (!$contact) {
                return $this->sendError('Contact not found.', [], 404);
            }

            $ledger_details = $this->transactionUtil->getLedgerDetails($contact_id, $start_date, $end_date);

            return $this->sendResponse([
                'contact' => $this->formatContactResponse($contact),
                'ledger' => $ledger_details
            ], 'Ledger retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Check if contact ID already exists.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkContactId(Request $request)
    {
        try {
            $contact_id = $request->input('contact_id');
            $hidden_id = $request->input('hidden_id');
            $business_id = auth()->user()->business_id;

            $exists = false;
            if (!empty($contact_id)) {
                $query = Contact::where('business_id', $business_id)
                    ->where('contact_id', $contact_id);
                if (!empty($hidden_id)) {
                    $query->where('id', '!=', $hidden_id);
                }
                $exists = $query->exists();
            }

            return $this->sendResponse([
                'exists' => $exists,
                'available' => !$exists
            ], 'Contact ID availability checked.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Format contact response data.
     */
    private function formatContactResponse($contact)
    {
        return [
            'id' => $contact->id,
            'type' => $contact->type,
            'name' => $contact->name,
            'first_name' => $contact->first_name,
            'middle_name' => $contact->middle_name,
            'last_name' => $contact->last_name,
            'prefix' => $contact->prefix,
            'contact_id' => $contact->contact_id,
            'supplier_business_name' => $contact->supplier_business_name,
            'email' => $contact->email,
            'mobile' => $contact->mobile,
            'landline' => $contact->landline,
            'alternate_number' => $contact->alternate_number,
            'address_line_1' => $contact->address_line_1,
            'address_line_2' => $contact->address_line_2,
            'city' => $contact->city,
            'state' => $contact->state,
            'country' => $contact->country,
            'zip_code' => $contact->zip_code,
            'contact_status' => $contact->contact_status,
            'customer_group_id' => $contact->customer_group_id,
            'tax_number' => $contact->tax_number,
            'pay_term_number' => $contact->pay_term_number,
            'pay_term_type' => $contact->pay_term_type,
            'credit_limit' => $contact->credit_limit,
            'dob' => $contact->dob,
            'shipping_address' => $contact->shipping_address,
            'position' => $contact->position,
            'custom_field1' => $contact->custom_field1,
            'custom_field2' => $contact->custom_field2,
            'custom_field3' => $contact->custom_field3,
            'custom_field4' => $contact->custom_field4,
            'created_at' => $contact->created_at,
            'updated_at' => $contact->updated_at,
        ];
    }

    /**
     * Format address for display.
     */
    private function formatAddress($contact)
    {
        return implode(', ', array_filter([
            $contact->address_line_1,
            $contact->address_line_2,
            $contact->city,
            $contact->state,
            $contact->country
        ]));
    }
}
