<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\BusinessLocation;
use App\Models\TaxRate;
use App\Models\Transaction;
use App\Models\PurchaseLine;
use App\Utils\ProductUtil;
use App\Utils\ModuleUtil;
use App\Utils\TransactionUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CombinedPurchaseReturnController extends BaseController
{
    /**
     * All Utils instance.
     */
    protected $productUtil;
    protected $moduleUtil;
    protected $transactionUtil;

    /**
     * Constructor
     *
     * @param ProductUtil $productUtil
     * @param ModuleUtil $moduleUtil
     * @param TransactionUtil $transactionUtil
     * @return void
     */
    public function __construct(ProductUtil $productUtil, ModuleUtil $moduleUtil, TransactionUtil $transactionUtil)
    {
        $this->productUtil = $productUtil;
        $this->moduleUtil = $moduleUtil;
        $this->transactionUtil = $transactionUtil;
    }

    /**
     * Display a listing of the resource.
     * GET /api/purchase-returns
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            // Check if subscribed
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            }

            $purchase_returns = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->with(['contact:id,name,supplier_business_name', 'business_location:id,name'])
                ->orderBy('created_at', 'desc')
                ->get();

            return $this->sendResponse($purchase_returns, 'Purchase returns retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving purchase returns.', $e->getMessage(), 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     * GET /api/purchase-returns/create
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create()
    {
        if (!auth()->user()->can('purchase.update')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            // Check if subscribed
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            }

            $business_locations = BusinessLocation::forDropdown($business_id);
            $taxes = TaxRate::where('business_id', $business_id)->get();

            $data = [
                'business_locations' => $business_locations,
                'taxes' => $taxes,
                'fields' => [
                    'location_id' => 'required|exists:business_locations,id',
                    'contact_id' => 'required|exists:contacts,id',
                    'transaction_date' => 'required|date',
                    'ref_no' => 'nullable|string|unique:transactions,ref_no',
                    'tax_id' => 'nullable|exists:tax_rates,id',
                    'tax_amount' => 'required|numeric|min:0',
                    'final_total' => 'required|numeric|min:0',
                    'products' => 'required|array|min:1',
                    'products.*.product_id' => 'required|exists:products,id',
                    'products.*.variation_id' => 'required|exists:variations,id',
                    'products.*.quantity' => 'required|numeric|min:0.01',
                    'products.*.unit_price' => 'required|numeric|min:0'
                ]
            ];

            return $this->sendResponse($data, 'Purchase return creation form data.');
        } catch (\Exception $e) {
            return $this->sendError('Error loading creation form.', $e->getMessage(), 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     * POST /api/purchase-returns
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('purchase.update')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            DB::beginTransaction();

            $input_data = $request->only([
                'location_id',
                'transaction_date',
                'final_total',
                'ref_no',
                'tax_id',
                'tax_amount',
                'contact_id'
            ]);

            $business_id = auth()->user()->business_id;

            // Check if subscribed
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            }

            $user_id = $user_id = auth()->user()->id;

            $input_data['type'] = 'purchase_return';
            $input_data['business_id'] = $business_id;
            $input_data['created_by'] = $user_id;
            $input_data['transaction_date'] = $this->productUtil->uf_date($input_data['transaction_date'], true);
            $input_data['total_before_tax'] = $input_data['final_total'] - $input_data['tax_amount'];

            // Update reference count
            $ref_count = $this->productUtil->setAndGetReferenceCount('purchase_return');

            // Generate reference number
            if (empty($input_data['ref_no'])) {
                $input_data['ref_no'] = $this->productUtil->generateReferenceNumber('purchase_return', $ref_count);
            }

            // Upload document
            $input_data['document'] = $this->productUtil->uploadFile($request, 'document', 'documents');

            $products = $request->input('products');

            if (!empty($products)) {
                $product_data = [];

                foreach ($products as $product) {
                    $unit_price = $this->productUtil->num_uf($product['unit_price']);
                    $return_line = [
                        'product_id' => $product['product_id'],
                        'variation_id' => $product['variation_id'],
                        'quantity' => 0,
                        'purchase_price' => $unit_price,
                        'pp_without_discount' => $unit_price,
                        'purchase_price_inc_tax' => $unit_price,
                        'quantity_returned' => $this->productUtil->num_uf($product['quantity']),
                        'lot_number' => !empty($product['lot_number']) ? $product['lot_number'] : null,
                        'exp_date' => !empty($product['exp_date']) ?
                            $this->productUtil->uf_date($product['exp_date']) : null
                    ];

                    $product_data[] = $return_line;

                    // Decrease available quantity
                    $this->productUtil->decreaseProductQuantity(
                        $product['product_id'],
                        $product['variation_id'],
                        $input_data['location_id'],
                        $this->productUtil->num_uf($product['quantity'])
                    );
                }

                $purchase_return = Transaction::create($input_data);
                $purchase_return->purchase_lines()->createMany($product_data);

                // Update payment status
                $this->transactionUtil->updatePaymentStatus($purchase_return->id, $purchase_return->final_total);

                DB::commit();

                return $this->sendResponse($purchase_return, 'Purchase return added successfully.');
            } else {
                return $this->sendError('No products provided.', [], 422);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Display the specified resource.
     * GET /api/purchase-returns/{id}
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            if (!auth()->user()->can('purchase.view') && !auth()->user()->can('purchase.update')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $purchase_return = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->with(['contact', 'business_location', 'purchase_lines.product', 'purchase_lines.variation'])
                ->find($id);

            if (!$purchase_return) {
                return $this->sendError('Purchase return not found.', [], 404);
            }

            return $this->sendResponse($purchase_return, 'Purchase return retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving purchase return.', $e->getMessage(), 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     * GET /api/purchase-returns/{id}/edit
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        if (!auth()->user()->can('purchase.update')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $purchase_return = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->with(['contact'])
                ->find($id);

            if (!$purchase_return) {
                return $this->sendError('Purchase return not found.', [], 404);
            }

            $location_id = $purchase_return->location_id;
            $purchase_lines = PurchaseLine::join('products AS p', 'purchase_lines.product_id', '=', 'p.id')
                ->join('variations AS variations', 'purchase_lines.variation_id', '=', 'variations.id')
                ->join('product_variations AS pv', 'variations.product_variation_id', '=', 'pv.id')
                ->leftjoin('variation_location_details AS vld', function ($join) use ($location_id) {
                    $join->on('variations.id', '=', 'vld.variation_id')
                        ->where('vld.location_id', '=', $location_id);
                })
                ->leftjoin('units', 'units.id', '=', 'p.unit_id')
                ->where('purchase_lines.transaction_id', $id)
                ->select([
                    DB::raw("IF(pv.is_dummy = 0, CONCAT(p.name, 
                        ' (', pv.name, ':',variations.name, ')'), p.name) AS product_name"),
                    'p.id as product_id',
                    'p.enable_stock',
                    'pv.is_dummy as is_dummy',
                    'variations.sub_sku',
                    'vld.qty_available',
                    'variations.id as variation_id',
                    'units.short_name as unit',
                    'units.allow_decimal as unit_allow_decimal',
                    'purchase_lines.purchase_price',
                    'purchase_lines.id as purchase_line_id',
                    'purchase_lines.quantity_returned as quantity_returned',
                    'purchase_lines.lot_number',
                    'purchase_lines.exp_date'
                ])
                ->get();

            foreach ($purchase_lines as $key => $value) {
                $purchase_lines[$key]->qty_available += $value->quantity_returned;
                $purchase_lines[$key]->formatted_qty_available = $this->productUtil->num_f($purchase_lines[$key]->qty_available);
            }

            $business_locations = BusinessLocation::forDropdown($business_id);
            $taxes = TaxRate::where('business_id', $business_id)->get();

            $data = [
                'purchase_return' => $purchase_return,
                'purchase_lines' => $purchase_lines,
                'business_locations' => $business_locations,
                'taxes' => $taxes
            ];

            return $this->sendResponse($data, 'Purchase return edit data retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error loading edit form.', $e->getMessage(), 500);
        }
    }

    /**
     * Update the specified resource in storage.
     * PUT/PATCH /api/purchase-returns/{id}
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('purchase.update')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            DB::beginTransaction();

            $input_data = $request->only([
                'transaction_date',
                'final_total',
                'tax_id',
                'tax_amount',
                'contact_id'
            ]);

            $business_id = auth()->user()->business_id;

            if (!empty($request->input('ref_no'))) {
                $input_data['ref_no'] = $request->input('ref_no');
            }

            // Check if subscribed
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            }

            $input_data['transaction_date'] = $this->productUtil->uf_date($input_data['transaction_date'], true);
            $input_data['total_before_tax'] = $input_data['final_total'] - $input_data['tax_amount'];

            // Upload document
            $doc_name = $this->productUtil->uploadFile($request, 'document', 'documents');
            if (!empty($doc_name)) {
                $input_data['document'] = $doc_name;
            }

            $products = $request->input('products');
            $purchase_return = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->find($id);

            if (!$purchase_return) {
                return $this->sendError('Purchase return not found.', [], 404);
            }

            if (!empty($products)) {
                $product_data = [];
                $updated_purchase_lines = [];

                foreach ($products as $product) {
                    $unit_price = $this->productUtil->num_uf($product['unit_price']);

                    if (!empty($product['purchase_line_id'])) {
                        $return_line = PurchaseLine::find($product['purchase_line_id']);
                        $updated_purchase_lines[] = $return_line->id;

                        $this->productUtil->decreaseProductQuantity(
                            $product['product_id'],
                            $product['variation_id'],
                            $purchase_return->location_id,
                            $this->productUtil->num_uf($product['quantity']),
                            $return_line->quantity_returned
                        );
                    } else {
                        $return_line = new PurchaseLine([
                            'product_id' => $product['product_id'],
                            'variation_id' => $product['variation_id'],
                            'quantity' => 0
                        ]);

                        // Decrease available quantity
                        $this->productUtil->decreaseProductQuantity(
                            $product['product_id'],
                            $product['variation_id'],
                            $purchase_return->location_id,
                            $this->productUtil->num_uf($product['quantity'])
                        );
                    }

                    $return_line->purchase_price = $unit_price;
                    $return_line->pp_without_discount = $unit_price;
                    $return_line->purchase_price_inc_tax = $unit_price;
                    $return_line->quantity_returned = $this->productUtil->num_uf($product['quantity']);
                    $return_line->lot_number = !empty($product['lot_number']) ? $product['lot_number'] : null;
                    $return_line->exp_date = !empty($product['exp_date']) ?
                        $this->productUtil->uf_date($product['exp_date']) : null;

                    $product_data[] = $return_line;
                }

                $purchase_return->update($input_data);

                // If purchase line deleted add return quantity to stock
                $deleted_purchase_lines = PurchaseLine::where('transaction_id', $id)
                    ->whereNotIn('id', $updated_purchase_lines)
                    ->get();

                foreach ($deleted_purchase_lines as $dpl) {
                    $this->productUtil->updateProductQuantity(
                        $purchase_return->location_id,
                        $dpl->product_id,
                        $dpl->variation_id,
                        $dpl->quantity_returned,
                        0,
                        null,
                        false
                    );
                }

                PurchaseLine::where('transaction_id', $id)
                    ->whereNotIn('id', $updated_purchase_lines)
                    ->delete();

                $purchase_return->purchase_lines()->saveMany($product_data);

                // Update payment status
                $this->transactionUtil->updatePaymentStatus($purchase_return->id, $purchase_return->final_total);

                DB::commit();

                return $this->sendResponse($purchase_return, 'Purchase return updated successfully.');
            } else {
                return $this->sendError('No products provided.', [], 422);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     * DELETE /api/purchase-returns/{id}
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('purchase.delete')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            DB::beginTransaction();

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $purchase_return = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->with('purchase_lines')
                ->find($id);

            if (!$purchase_return) {
                return $this->sendError('Purchase return not found.', [], 404);
            }

            // Restore quantities for all returned products
            foreach ($purchase_return->purchase_lines as $line) {
                $this->productUtil->updateProductQuantity(
                    $purchase_return->location_id,
                    $line->product_id,
                    $line->variation_id,
                    $line->quantity_returned,
                    0,
                    null,
                    false
                );
            }

            // Delete purchase lines
            $purchase_return->purchase_lines()->delete();

            // Delete the transaction
            $purchase_return->delete();

            DB::commit();

            return $this->sendResponse([], 'Purchase return deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error deleting purchase return.', $e->getMessage(), 500);
        }
    }

    /**
     * Get product row for purchase return
     * POST /api/purchase-returns/get-product-row
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProductRow(Request $request)
    {
        try {
            $row_index = $request->input('row_index');
            $variation_id = $request->input('variation_id');
            $location_id = $request->input('location_id');

            if (empty($variation_id) || empty($location_id)) {
                return $this->sendError('Missing required parameters.', [], 422);
            }

            $business_id = auth()->user()->business_id;
            $product = $this->productUtil->getDetailsFromVariation($variation_id, $business_id, $location_id);

            if (!$product) {
                return $this->sendError('Product not found.', [], 404);
            }

            $product->formatted_qty_available = $this->productUtil->num_f($product->qty_available);

            $data = [
                'product' => $product,
                'row_index' => $row_index,
                'html' => view('purchase_return.partials.product_table_row')
                    ->with(compact('product', 'row_index'))->render()
            ];

            return $this->sendResponse($data, 'Product row retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving product row.', $e->getMessage(), 500);
        }
    }

    /**
     * Get products by location for purchase return
     * GET /api/purchase-returns/products/{location_id}
     *
     * @param int $location_id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProductsByLocation($location_id)
    {
        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $products = $this->productUtil->getProductsForPurchaseReturn($business_id, $location_id);

            return $this->sendResponse($products, 'Products retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving products.', $e->getMessage(), 500);
        }
    }

    /**
     * Get purchase return statistics
     * GET /api/purchase-returns/statistics
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getStatistics()
    {
        try {
            if (!auth()->user()->can('purchase.view')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $total_returns = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->count();

            $total_amount = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->sum('final_total');

            $monthly_returns = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->whereMonth('transaction_date', now()->month)
                ->whereYear('transaction_date', now()->year)
                ->count();

            $monthly_amount = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->whereMonth('transaction_date', now()->month)
                ->whereYear('transaction_date', now()->year)
                ->sum('final_total');

            $data = [
                'total_returns' => $total_returns,
                'total_amount' => $total_amount,
                'monthly_returns' => $monthly_returns,
                'monthly_amount' => $monthly_amount
            ];

            return $this->sendResponse($data, 'Purchase return statistics retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving statistics.', $e->getMessage(), 500);
        }
    }
}
