<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\BusinessLocation;
use App\Models\CashRegister;
use App\Utils\CashRegisterUtil;
use App\Utils\ModuleUtil;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class CashRegisterController extends BaseController
{
    /**
     * All Utils instance.
     */
    protected $cashRegisterUtil;
    protected $moduleUtil;

    /**
     * Constructor
     *
     * @param CashRegisterUtil $cashRegisterUtil
     * @param ModuleUtil $moduleUtil
     * @return void
     */
    public function __construct(CashRegisterUtil $cashRegisterUtil, ModuleUtil $moduleUtil)
    {
        $this->cashRegisterUtil = $cashRegisterUtil;
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of the resource.
     * GET /api/cash-registers
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $user_id = auth()->user()->id;

            // Get all cash registers for the business
            $cash_registers = CashRegister::where('business_id', $business_id)
                ->with(['user:id,first_name,last_name', 'business_location:id,name'])
                ->orderBy('created_at', 'desc')
                ->get();

            // Get current open register count
            $open_register_count = $this->cashRegisterUtil->countOpenedRegister();

            $data = [
                'cash_registers' => $cash_registers,
                'open_register_count' => $open_register_count,
                'has_open_register' => $open_register_count > 0
            ];

            return $this->sendResponse($data, 'Cash registers retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving cash registers.', $e->getMessage(), 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     * GET /api/cash-registers/create
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create()
    {
        try {
            $sub_type = request()->get('sub_type');

            // Check if there is an open register
            $open_register_count = $this->cashRegisterUtil->countOpenedRegister();

            if ($open_register_count != 0) {
                return $this->sendError('There is already an open cash register.', [
                    'redirect_to_pos' => true,
                    'sub_type' => $sub_type
                ], 409);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $business_locations = BusinessLocation::forDropdown($business_id);

            $data = [
                'business_locations' => $business_locations,
                'sub_type' => $sub_type,
                'fields' => [
                    'location_id' => 'required|exists:business_locations,id',
                    'amount' => 'nullable|numeric|min:0'
                ]
            ];

            return $this->sendResponse($data, 'Cash register creation form data.');
        } catch (\Exception $e) {
            return $this->sendError('Error loading creation form.', $e->getMessage(), 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     * POST /api/cash-registers
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $sub_type = request()->get('sub_type');

            $initial_amount = 0;
            if (!empty($request->input('amount'))) {
                $initial_amount = $this->cashRegisterUtil->num_uf($request->input('amount'));
            }

            $user_id = $user_id = auth()->user()->id;
            $business_id = auth()->user()->business_id;

            $register = CashRegister::create([
                'business_id' => $business_id,
                'user_id' => $user_id,
                'status' => 'open',
                'location_id' => $request->input('location_id'),
                'created_at' => Carbon::now()->format('Y-m-d H:i:00')
            ]);

            if (!empty($initial_amount)) {
                $register->cash_register_transactions()->create([
                    'amount' => $initial_amount,
                    'pay_method' => 'cash',
                    'type' => 'credit',
                    'transaction_type' => 'initial'
                ]);
            }

            $data = [
                'register' => $register,
                'initial_amount' => $initial_amount,
                'redirect_to_pos' => true,
                'sub_type' => $sub_type
            ];

            return $this->sendResponse($data, 'Cash register opened successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error creating cash register.', $e->getMessage(), 500);
        }
    }

    /**
     * Display the specified resource.
     * GET /api/cash-registers/{id}
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $register_details = $this->cashRegisterUtil->getRegisterDetails($id);

            if (!$register_details) {
                return $this->sendError('Cash register not found.', [], 404);
            }

            $user_id = $register_details->user_id;
            $open_time = $register_details['open_time'];
            $close_time = !empty($register_details['closed_at']) ?
                $register_details['closed_at'] : Carbon::now()->toDateTimeString();

            $details = $this->cashRegisterUtil->getRegisterTransactionDetails($user_id, $open_time, $close_time);
            $payment_types = $this->cashRegisterUtil->payment_types();

            $data = [
                'register_details' => $register_details,
                'transaction_details' => $details,
                'payment_types' => $payment_types,
                'close_time' => $close_time
            ];

            return $this->sendResponse($data, 'Cash register details retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving cash register details.', $e->getMessage(), 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     * GET /api/cash-registers/{id}/edit
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        try {
            $register = CashRegister::find($id);

            if (!$register) {
                return $this->sendError('Cash register not found.', [], 404);
            }

            if ($register->status == 'close') {
                return $this->sendError('Cannot edit closed cash register.', [], 400);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $business_locations = BusinessLocation::forDropdown($business_id);

            $data = [
                'register' => $register,
                'business_locations' => $business_locations,
                'fields' => [
                    'location_id' => 'required|exists:business_locations,id',
                    'closing_note' => 'nullable|string'
                ]
            ];

            return $this->sendResponse($data, 'Cash register edit data retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error loading edit form.', $e->getMessage(), 500);
        }
    }

    /**
     * Update the specified resource in storage.
     * PUT/PATCH /api/cash-registers/{id}
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $register = CashRegister::find($id);

            if (!$register) {
                return $this->sendError('Cash register not found.', [], 404);
            }

            if ($register->status == 'close') {
                return $this->sendError('Cannot update closed cash register.', [], 400);
            }

            $input = $request->only(['location_id', 'closing_note']);
            $register->update($input);

            return $this->sendResponse($register, 'Cash register updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error updating cash register.', $e->getMessage(), 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     * DELETE /api/cash-registers/{id}
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $register = CashRegister::find($id);

            if (!$register) {
                return $this->sendError('Cash register not found.', [], 404);
            }

            if ($register->status == 'open') {
                return $this->sendError('Cannot delete open cash register. Please close it first.', [], 400);
            }

            // Delete related transactions first
            $register->cash_register_transactions()->delete();
            $register->delete();

            return $this->sendResponse([], 'Cash register deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Error deleting cash register.', $e->getMessage(), 500);
        }
    }

    /**
     * Get current register details.
     * GET /api/cash-registers/current
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCurrentRegisterDetails()
    {
        try {
            $register_details = $this->cashRegisterUtil->getRegisterDetails();

            if (!$register_details) {
                return $this->sendError('No open cash register found.', [], 404);
            }

            $user_id = auth()->user()->id;
            $open_time = $register_details['open_time'];
            $close_time = Carbon::now()->toDateTimeString();

            $is_types_of_service_enabled = $this->moduleUtil->isModuleEnabled('types_of_service');
            $details = $this->cashRegisterUtil->getRegisterTransactionDetails(
                $user_id,
                $open_time,
                $close_time,
                $is_types_of_service_enabled
            );

            $payment_types = $this->cashRegisterUtil->payment_types($register_details->location_id);

            $data = [
                'register_details' => $register_details,
                'transaction_details' => $details,
                'payment_types' => $payment_types,
                'close_time' => $close_time,
                'is_types_of_service_enabled' => $is_types_of_service_enabled
            ];

            return $this->sendResponse($data, 'Current register details retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving current register details.', $e->getMessage(), 500);
        }
    }

    /**
     * Get close register form data.
     * GET /api/cash-registers/close-form
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCloseRegisterForm()
    {
        try {
            $register_details = $this->cashRegisterUtil->getRegisterDetails();

            if (!$register_details) {
                return $this->sendError('No open cash register found.', [], 404);
            }

            $user_id = auth()->user()->id;
            $open_time = $register_details['open_time'];
            $close_time = Carbon::now()->toDateTimeString();

            $is_types_of_service_enabled = $this->moduleUtil->isModuleEnabled('types_of_service');
            $details = $this->cashRegisterUtil->getRegisterTransactionDetails(
                $user_id,
                $open_time,
                $close_time,
                $is_types_of_service_enabled
            );

            $payment_types = $this->cashRegisterUtil->payment_types($register_details->location_id);

            $data = [
                'register_details' => $register_details,
                'transaction_details' => $details,
                'payment_types' => $payment_types,
                'fields' => [
                    'closing_amount' => 'required|numeric|min:0',
                    'total_card_slips' => 'nullable|integer|min:0',
                    'total_cheques' => 'nullable|integer|min:0',
                    'closing_note' => 'nullable|string'
                ]
            ];

            return $this->sendResponse($data, 'Close register form data retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error loading close register form.', $e->getMessage(), 500);
        }
    }

    /**
     * Close currently opened register.
     * POST /api/cash-registers/close
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function closeRegister(Request $request)
    {
        try {
            // Disable in demo
            if (config('app.env') == 'demo') {
                return $this->sendError('Feature disabled in demo!!', [], 403);
            }

            $input = $request->only(['closing_amount', 'total_card_slips', 'total_cheques', 'closing_note']);

            if (!isset($input['closing_amount']) || $input['closing_amount'] === null) {
                return $this->sendError('Closing amount is required.', [], 422);
            }

            $input['closing_amount'] = $this->cashRegisterUtil->num_uf($input['closing_amount']);
            $user_id = $user_id = auth()->user()->id;
            $input['closed_at'] = Carbon::now()->format('Y-m-d H:i:s');
            $input['status'] = 'close';

            $updated = CashRegister::where('user_id', $user_id)
                ->where('status', 'open')
                ->update($input);

            if (!$updated) {
                return $this->sendError('No open cash register found to close.', [], 404);
            }

            return $this->sendResponse([], 'Cash register closed successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Get register status.
     * GET /api/cash-registers/status
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getRegisterStatus()
    {
        try {
            $open_register_count = $this->cashRegisterUtil->countOpenedRegister();
            $user_id = auth()->user()->id;

            $current_open_register = CashRegister::where('user_id', $user_id)
                ->where('status', 'open')
                ->first();

            $data = [
                'has_open_register' => $open_register_count > 0,
                'open_register_count' => $open_register_count,
                'current_user_has_open_register' => $current_open_register !== null,
                'current_open_register' => $current_open_register
            ];

            return $this->sendResponse($data, 'Register status retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving register status.', $e->getMessage(), 500);
        }
    }

    /**
     * Get cash register statistics.
     * GET /api/cash-registers/statistics
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getStatistics()
    {
        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $total_registers = CashRegister::where('business_id', $business_id)->count();
            $open_registers = CashRegister::where('business_id', $business_id)
                ->where('status', 'open')->count();
            $closed_registers = CashRegister::where('business_id', $business_id)
                ->where('status', 'close')->count();

            // Get recent registers
            $recent_registers = CashRegister::where('business_id', $business_id)
                ->with(['user:id,first_name,last_name', 'business_location:id,name'])
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get();

            $data = [
                'total_registers' => $total_registers,
                'open_registers' => $open_registers,
                'closed_registers' => $closed_registers,
                'recent_registers' => $recent_registers
            ];

            return $this->sendResponse($data, 'Cash register statistics retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving statistics.', $e->getMessage(), 500);
        }
    }
}
