<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Account;
use App\Models\BusinessLocation;
use App\Models\InvoiceLayout;
use App\Models\InvoiceScheme;
use App\Models\SellingPriceGroup;
use App\Utils\ModuleUtil;
use App\Utils\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class BusinessLocationController extends BaseController
{
    protected $moduleUtil;
    protected $commonUtil;

    public function __construct(ModuleUtil $moduleUtil, Util $commonUtil)
    {
        $this->moduleUtil = $moduleUtil;
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of the resource.
     * GET /api/business-locations
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        try {
            if (!auth()->user()->can('business_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $locations = BusinessLocation::where('business_locations.business_id', $business_id)
                ->leftjoin('invoice_schemes as ic', 'business_locations.invoice_scheme_id', '=', 'ic.id')
                ->leftjoin('invoice_layouts as il', 'business_locations.invoice_layout_id', '=', 'il.id')
                ->leftjoin('selling_price_groups as spg', 'business_locations.selling_price_group_id', '=', 'spg.id')
                ->select([
                    'business_locations.name',
                    'location_id',
                    'landmark',
                    'city',
                    'zip_code',
                    'state',
                    'country',
                    'business_locations.id',
                    'spg.name as price_group',
                    'ic.name as invoice_scheme',
                    'il.name as invoice_layout',
                    'business_locations.is_active'
                ]);

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $locations->whereIn('business_locations.id', $permitted_locations);
            }

            $locations = $locations->get();

            return $this->sendResponse($locations, 'Business locations retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving business locations.', $e->getMessage(), 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     * GET /api/business-locations/create
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create()
    {
        if (!auth()->user()->can('business_settings.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            // Check if subscribed or not, then check for location quota
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            } elseif (!$this->moduleUtil->isQuotaAvailable('locations', $business_id)) {
                return $this->sendError('Location quota exceeded.', [], 402);
            }

            $invoice_layouts = InvoiceLayout::where('business_id', $business_id)->get();
            $invoice_schemes = InvoiceScheme::where('business_id', $business_id)->get();
            $price_groups = SellingPriceGroup::forDropdown($business_id);
            $payment_types = $this->commonUtil->payment_types();

            $accounts = [];
            if ($this->commonUtil->isModuleEnabled('account')) {
                $accounts = Account::forDropdown($business_id, true, false);
            }

            $data = [
                'invoice_layouts' => $invoice_layouts,
                'invoice_schemes' => $invoice_schemes,
                'price_groups' => $price_groups,
                'payment_types' => $payment_types,
                'accounts' => $accounts,
                'fields' => [
                    'name' => 'required|string',
                    'landmark' => 'nullable|string',
                    'city' => 'nullable|string',
                    'state' => 'nullable|string',
                    'country' => 'nullable|string',
                    'zip_code' => 'nullable|string',
                    'location_id' => 'nullable|string|unique:business_locations,location_id',
                    'mobile' => 'nullable|string',
                    'alternate_number' => 'nullable|string',
                    'email' => 'nullable|email',
                    'website' => 'nullable|url',
                    'invoice_scheme_id' => 'required|exists:invoice_schemes,id',
                    'invoice_layout_id' => 'required|exists:invoice_layouts,id',
                    'selling_price_group_id' => 'nullable|exists:selling_price_groups,id'
                ]
            ];

            return $this->sendResponse($data, 'Business location creation form data.');
        } catch (\Exception $e) {
            return $this->sendError('Error loading creation form.', $e->getMessage(), 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     * POST /api/business-locations
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('business_settings.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = auth()->user()->business_id;

            // Check if subscribed or not, then check for location quota
            if (!$this->moduleUtil->isSubscribed($business_id)) {
                return $this->sendError('Subscription expired.', [], 402);
            } elseif (!$this->moduleUtil->isQuotaAvailable('locations', $business_id)) {
                return $this->sendError('Location quota exceeded.', [], 402);
            }

            $input = $request->only([
                'name',
                'landmark',
                'city',
                'state',
                'country',
                'zip_code',
                'invoice_scheme_id',
                'invoice_layout_id',
                'mobile',
                'alternate_number',
                'email',
                'website',
                'custom_field1',
                'custom_field2',
                'custom_field3',
                'custom_field4',
                'location_id',
                'selling_price_group_id',
                'default_payment_accounts',
                'featured_products'
            ]);

            $business_id = auth()->user()->business_id;

            $input['default_payment_accounts'] = !empty($input['default_payment_accounts']) ?
                json_encode($input['default_payment_accounts']) : null;

            $input['featured_products'] = !empty($input['featured_products']) ?
                json_encode($input['featured_products']) : null;

            $location = BusinessLocation::where('business_id', $business_id)->where('id', $id)->first();

            if (!$location) {
                return $this->sendError('Business location not found.', [], 404);
            }

            $location->update($input);

            return $this->sendResponse($location, 'Business location updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     * DELETE /api/business-locations/{id}
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('business_settings.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $location = BusinessLocation::where('business_id', $business_id)->find($id);

            if (!$location) {
                return $this->sendError('Business location not found.', [], 404);
            }

            // Check if location can be deleted (you might want to add more business logic here)
            $location->delete();

            return $this->sendResponse([], 'Business location deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Checks if the given location id already exists for the current business.
     * POST /api/business-locations/check-location-id
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkLocationId(Request $request)
    {
        try {
            $location_id = $request->input('location_id');

            if (empty($location_id)) {
                return $this->sendResponse(['valid' => true], 'Location ID is valid.');
            }

            $business_id = auth()->user()->business_id;
            $hidden_id = $request->input('hidden_id');

            $query = BusinessLocation::where('business_id', $business_id)
                ->where('location_id', $location_id);

            if (!empty($hidden_id)) {
                $query->where('id', '!=', $hidden_id);
            }

            $count = $query->count();
            $valid = $count == 0;

            return $this->sendResponse(['valid' => $valid], 'Location ID validation completed.');
        } catch (\Exception $e) {
            return $this->sendError('Error validating location ID.', $e->getMessage(), 500);
        }
    }

    /**
     * Function to activate or deactivate a location.
     * POST /api/business-locations/{id}/toggle-status
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function activateDeactivateLocation($id)
    {
        if (!auth()->user()->can('business_settings.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $business_location = BusinessLocation::where('business_id', $business_id)->find($id);

            if (!$business_location) {
                return $this->sendError('Business location not found.', [], 404);
            }

            $business_location->is_active = !$business_location->is_active;
            $business_location->save();

            $msg = $business_location->is_active ?
                'Business location activated successfully' :
                'Business location deactivated successfully';

            return $this->sendResponse($business_location, $msg);
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }
}
