<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Business;
use App\Models\Currency;
use App\Models\Role;
use App\Models\System;
use App\Models\TaxRate;
use App\Models\Unit;
use App\Models\User;
use App\Utils\BusinessUtil;
use App\Utils\ModuleUtil;
use App\Utils\RestaurantUtil;
use Carbon\Carbon;
use DateTimeZone;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Spatie\Permission\Models\Permission;

class BusinessController extends BaseController
{
    /*
    |--------------------------------------------------------------------------
    | BusinessController
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new business/business as well as their
    | validation and creation.
    |
    */

    /**
     * All Utils instance.
     *
     */
    protected $businessUtil;
    protected $restaurantUtil;
    protected $moduleUtil;
    protected $mailDrivers;

    /**
     * Constructor
     *
     * @param ProductUtils $product
     * @return void
     */
    public function __construct(BusinessUtil $businessUtil, RestaurantUtil $restaurantUtil, ModuleUtil $moduleUtil)
    {
        $this->businessUtil = $businessUtil;
        $this->moduleUtil = $moduleUtil;

        $this->theme_colors = [
            'blue' => 'Blue',
            'black' => 'Black',
            'purple' => 'Purple',
            'green' => 'Green',
            'red' => 'Red',
            'yellow' => 'Yellow',
            'blue-light' => 'Blue Light',
            'black-light' => 'Black Light',
            'purple-light' => 'Purple Light',
            'green-light' => 'Green Light',
            'red-light' => 'Red Light',
        ];

        $this->mailDrivers = [
            'smtp' => 'SMTP',
            // 'sendmail' => 'Sendmail',
            // 'mailgun' => 'Mailgun',
            // 'mandrill' => 'Mandrill',
            // 'ses' => 'SES',
            // 'sparkpost' => 'Sparkpost'
        ];
    }

    /**
     * Affiche les données nécessaires à l'inscription (API).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getRegister(){
        if (!config('constants.allow_registration')) {
            return $this->sendError(__('messages.registration_disabled'));
        }

        $currencies = $this->businessUtil->allCurrencies();
        $timezone_list = $this->businessUtil->allTimeZones();

        $months = [];
        for ($i = 1; $i <= 12; $i++) {
            $months[$i] = __('business.months.' . $i);
        }

        $accounting_methods = $this->businessUtil->allAccountingMethods();
        $package_id = request()->package;

        $system_settings = System::getProperties(['superadmin_enable_register_tc', 'superadmin_register_tc'], true);

        $data = compact(
            'currencies',
            'timezone_list',
            'months',
            'accounting_methods',
            'package_id',
            'system_settings'
        );

        return $this->sendResponse($data, __('messages.registration_data_loaded'));
    }

    // Ajoutez cette méthode dans votre BusinessController

    private function getBusinessDefaults($business_details)
    {
        return array_merge($business_details, [
            // Produits et inventaire
            'stop_selling_before' => 0,
            'enable_product_expiry' => 0,
            'on_product_expiry' => 'keep_selling',
            'stop_selling_before_days' => 0,
            'stock_expiry_alert_days' => 30,
            'enable_brand' => 1,
            'enable_category' => 1,
            'enable_sub_category' => 1,
            'enable_lot_number' => 0,
            'enable_sub_units' => 0,
            'enable_racks' => 0,
            'enable_row' => 0,
            'enable_position' => 0,
            'enable_editing_product_from_purchase' => 1,
            'enable_editing_sp_from_purchase' => 1,
            'enable_secondary_unit' => 0,
            'default_unit' => null,

            // Interface et affichage
            'enable_tooltip' => 1,
            'item_addition_method' => 1,
            'enable_inline_tax' => 0,
            'currency_symbol_placement' => 'before',
            'date_format' => 'd/m/Y',
            'time_format' => '24',
            'currency_precision' => 2,
            'quantity_precision' => 2,

            // Devises et prix
            'purchase_in_diff_currency' => 0,
            'purchase_currency_id' => null,
            'p_exchange_rate' => 1.000,
            'enable_price_tax' => 1,
            'default_sales_discount' => 0.00,
            'default_sales_tax' => null,

            // Paramètres business
            'transaction_edit_days' => 30,
            'enable_purchase_status' => 1,
            'sales_cmsn_agnt' => null,
            'monthly_reminder' => null,
            'is_active' => 1,
            'enable_duplicate_invoice_number' => 0,
            'accounting_method' => request('accounting_method', 'fifo'),

            // Modules optionnels
            'enable_repair' => 0,
            'repair_settings' => null,
            'enable_booking' => 0,
            'booking_settings' => null,

            // Paramètres JSON
            'weighing_scale_setting' => json_encode(['enable_scale' => 0]),
            'common_settings' => json_encode([
                'enable_rp' => 0,
                'rp_name' => 'Point',
                'amount_for_unit_rp' => 1,
                'min_order_total_for_rp' => 1,
                'max_rp_per_order' => null,
                'redeem_amount_per_unit_rp' => 1,
                'min_order_total_for_redeem' => 1,
                'min_redeem_point' => null,
                'max_redeem_point' => null,
                'rp_expiry_period' => null,
                'rp_expiry_type' => 'year'
            ]),
            'enabled_features' => json_encode([])
        ]);
    }
    /**
     * Handles the registration of a new business and its owner (API).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function postRegister(Request $request)
    {
        /*if (!config('constants.allow_registration')) {
            return $this->sendError(__('messages.registration_disabled'));
        }*/

        //try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|max:255',
                'currency_id' => 'required|numeric',
                'country' => 'required|max:255',
                'state' => 'required|max:255',
                'city' => 'required|max:255',
                'zip_code' => 'required|max:255',
                'landmark' => 'required|max:255',
                'time_zone' => 'required|max:255',
                'surname' => 'max:10',
                'email' => 'sometimes|nullable|email|unique:users|max:255',
                'first_name' => 'required|max:255',
                'username' => 'required|min:4|max:255|unique:users',
                'password' => 'required|min:4|max:255',
                'fy_start_month' => 'required',
                'accounting_method' => 'required',
            ], [
                'name.required' => __('validation.required', ['attribute' => __('business.business_name')]),
                'name.currency_id' => __('validation.required', ['attribute' => __('business.currency')]),
                'country.required' => __('validation.required', ['attribute' => __('business.country')]),
                'state.required' => __('validation.required', ['attribute' => __('business.state')]),
                'city.required' => __('validation.required', ['attribute' => __('business.city')]),
                'zip_code.required' => __('validation.required', ['attribute' => __('business.zip_code')]),
                'landmark.required' => __('validation.required', ['attribute' => __('business.landmark')]),
                'time_zone.required' => __('validation.required', ['attribute' => __('business.time_zone')]),
                'email.email' => __('validation.email', ['attribute' => __('business.email')]),
                'email.unique' => __('validation.unique', ['attribute' => __('business.email')]),
                'first_name.required' => __('validation.required', ['attribute' => __('business.first_name')]),
                'username.required' => __('validation.required', ['attribute' => __('business.username')]),
                'username.min' => __('validation.min', ['attribute' => __('business.username')]),
                'password.required' => __('validation.required', ['attribute' => __('business.username')]),
                'password.min' => __('validation.min', ['attribute' => __('business.username')]),
                'fy_start_month.required' => __('validation.required', ['attribute' => __('business.fy_start_month')]),
                'accounting_method.required' => __('validation.required', ['attribute' => __('business.accounting_method')]),
            ]);

            if ($validator->fails()) {
                return $this->sendError(__('messages.validation_error'), $validator->errors(), 422);
            }

            DB::beginTransaction();

            // Création du propriétaire
            $owner_details = $request->only(['surname', 'first_name', 'last_name', 'username', 'email', 'password', 'language']);
            $owner_details['language'] = empty($owner_details['language']) ? config('app.locale') : $owner_details['language'];
            $user = User::create_user($owner_details);

            $business_details = $request->only(['name', 'start_date', 'currency_id', 'time_zone']);
            $business_details['fy_start_month'] = 1;

            $business_location = $request->only(['name', 'country', 'state', 'city', 'zip_code', 'landmark', 'website', 'mobile', 'alternate_number']);

            // Création de l'entreprise
            $business_details['owner_id'] = $user->id;
            if (!empty($business_details['start_date'])) {
                $business_details['start_date'] = Carbon::parse($business_details['start_date'])->toDateString();
            }

            // Upload du logo
            $logo_name = $this->businessUtil->uploadFile($request, 'business_logo', 'business_logos', 'image');
            if (!empty($logo_name)) {
                $business_details['logo'] = $logo_name;
            }

            // Modules activés par défaut
            $business_details['enabled_modules'] = ['purchases', 'add_sale', 'pos_sale', 'stock_transfers', 'stock_adjustment', 'expenses'];
            $business_details = $this->getBusinessDefaults($business_details);
            $business = $this->businessUtil->createNewBusiness($business_details);

            // Mise à jour de l'utilisateur avec l'id de l'entreprise
            $user->business_id = $business->id;
            $user->save();

            $this->businessUtil->newBusinessDefaultResources($business->id, $user->id);
            $new_location = $this->businessUtil->addLocation($business->id, $business_location);

            // Créer la permission pour la nouvelle localisation
            Permission::create(['name' => 'location.' . $new_location->id]);

            // Assigner le rôle Admin et configurer les permissions
            try {
                // 1. Assigner le rôle Admin à l'utilisateur
                $admin_role_name = 'Admin#' . $business->id;
                $user->assignRole($admin_role_name);

                // 2. Récupérer le rôle Admin
                $admin_role = Role::where('name', $admin_role_name)->first();

                if ($admin_role) {
                    // 3. Définir les permissions essentielles pour un Admin
                    $essential_permissions = [
                    'rofit_loss_report.view',
                    'irect_sell.access',
                    'roduct.opening_stock',
                    'rud_all_bookings',
                    'rud_own_bookings',
                    'ccess_default_selling_price',
                    'urchase.payments',
                    'ell.payments',
                    'dit_product_price_from_sale_screen',
                    'edit_product_discount_from_sale_screen',
                    'roles.view',
                    'roles.create',
                    'roles.update',
                    'roles.delete',
                    'account.access',
                    'discount.access',
                    'view_purchase_price',
                    'view_own_sell_only',
                    'edit_product_discount_from_pos_screen',
                    'edit_product_price_from_pos_screen',
                    'access_shipping',
                    'purchase.update_status',
                    'list_drafts',
                    'list_quotations',
                    'user.view',
                    'user.create',
                    'user.update',
                    'user.delete',
                    'supplier.view',
                    'supplier.create',
                    'supplier.update',
                    'supplier.delete',
                    'customer.view',
                    'customer.create',
                    'customer.update',
                    'customer.delete',
                    'product.view',
                    'product.create',
                    'product.update',
                    'product.delete',
                    'purchase.view',
                    'purchase.create',
                    'purchase.update',
                    'purchase.delete',
                    'sell.view',
                    'sell.create',
                    'sell.update',
                    'sell.delete',
                    'purchase_n_sell_report.view',
                    'contacts_report.view',
                    'stock_report.view',
                    'tax_report.view',
                    'trending_product_report.view ',
                    'register_report.view',
                    'sales_representative.view ',
                    'expense_report.view',
                    'business_settings.access',
                    'barcode_settings.access',
                    'invoice_settings.access',
                    'brand.view',
                    'brand.create',
                    'brand.update',
                    'brand.delete',
                    'tax_rate.view',
                    'tax_rate.create',
                    'tax_rate.update',
                    'tax_rate.delete',
                    'unit.view',
                    'unit.create',
                    'unit.update',
                    'unit.delete',
                    'category.view',
                    'category.create',
                    'category.update',
                    'category.delete',
                    'expense.access',
                    'access_all_locations',
                    'dashboard.data',
                    'view_own_purchase',
                    'view_product_stock_value',
                    'view_own_expense',
                    'access_printers',
                    'sell.cancelled',
                    'location.' . $new_location->id
                    ];

                    // 4. Créer et attribuer les permissions
                    $permissions_to_assign = [];
                    foreach ($essential_permissions as $permission_name) {
                        $permission = Permission::firstOrCreate(['name' => $permission_name]);
                        $permissions_to_assign[] = $permission;
                    }

                    // 5. Attribuer toutes les permissions au rôle Admin
                    $admin_role->syncPermissions($permissions_to_assign);

                    // 6. Optionnel : Attribuer TOUTES les permissions existantes (décommentez si nécessaire)
                    /*
            $all_permissions = Permission::all();
            $admin_role->syncPermissions($all_permissions);
            */
                }
            } catch (\Exception $e) {
                Log::warning("Erreur lors de l'attribution des permissions Admin: " . $e->getMessage());
                // Ne pas faire échouer l'inscription pour un problème de permissions
            }


            DB::commit();

            // Génère un token Sanctum pour l'utilisateur
            $token = $user->createToken('SIGH')->plainTextToken;

            return $this->sendResponse(
                ['token' => $token, 'user' => $user],
                __('business.business_created_succesfully')
            );
        /*} catch (\Exception $e) {
            DB::rollBack();
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            return $this->sendError(__('messages.something_went_wrong'), [], 500);
        }*/
    }

    /**
     * Handles the validation username
     *
     * @return \Illuminate\Http\Response
     */
    public function postCheckUsername(Request $request)
    {
        $username = $request->input('username');

        if (!empty($request->input('username_ext'))) {
            $username .= $request->input('username_ext');
        }

        $count = User::where('username', $username)->count();

        if ($count == 0) {
            echo "true";
            exit;
        } else {
            echo "false";
            exit;
        }
    }

    /**
     * Get business settings
     *
     * @return JsonResponse
     */
    public function getBusinessSettings(): JsonResponse
    {
        try {
            if (!auth()->user()->can('business_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = request()->user()->business_id ?? session('user.business_id');

            if (!$business_id) {
                return $this->sendError('Business ID not found', [], 400);
            }

            $business = Business::find($business_id);

            if (!$business) {
                return $this->sendError('Business not found', [], 404);
            }

            // Récupération des données
            $timezones = DateTimeZone::listIdentifiers(DateTimeZone::ALL);
            $timezone_list = array_combine($timezones, $timezones);

            $currencies = $this->businessUtil->allCurrencies();
            $tax_details = TaxRate::forBusinessDropdown($business_id);
            $tax_rates = $tax_details['tax_rates'];

            $months = [];
            for ($i = 1; $i <= 12; $i++) {
                $months[$i] = __('business.months.' . $i);
            }

            $accounting_methods = [
                'fifo' => __('business.fifo'),
                'lifo' => __('business.lifo')
            ];

            $commission_agent_dropdown = [
                '' => __('lang_v1.disable'),
                'logged_in_user' => __('lang_v1.logged_in_user'),
                'user' => __('lang_v1.select_from_users_list'),
                'cmsn_agnt' => __('lang_v1.select_from_commisssion_agents_list')
            ];

            $units_dropdown = Unit::forDropdown($business_id, true);
            $date_formats = Business::date_formats();

            $shortcuts = json_decode($business->keyboard_shortcuts, true) ?? [];
            $pos_settings = empty($business->pos_settings)
                ? $this->businessUtil->defaultPosSettings()
                : json_decode($business->pos_settings, true);

            $email_settings = empty($business->email_settings)
                ? $this->businessUtil->defaultEmailSettings()
                : $business->email_settings;

            $sms_settings = empty($business->sms_settings)
                ? $this->businessUtil->defaultSmsSettings()
                : $business->sms_settings;

            $modules = $this->moduleUtil->availableModules();
            $theme_colors = $this->theme_colors;
            $mail_drivers = $this->mailDrivers;
            $allow_superadmin_email_settings = System::getProperty('allow_email_settings_to_businesses');

            $custom_labels = !empty($business->custom_labels)
                ? json_decode($business->custom_labels, true)
                : [];

            $common_settings = !empty($business->common_settings)
                ? $business->common_settings
                : [];

            $weighing_scale_setting = !empty($business->weighing_scale_setting)
                ? $business->weighing_scale_setting
                : [];

            $data = [
                'business' => $business,
                'currencies' => $currencies,
                'tax_rates' => $tax_rates,
                'timezone_list' => $timezone_list,
                'months' => $months,
                'accounting_methods' => $accounting_methods,
                'commission_agent_dropdown' => $commission_agent_dropdown,
                'units_dropdown' => $units_dropdown,
                'date_formats' => $date_formats,
                'shortcuts' => $shortcuts,
                'pos_settings' => $pos_settings,
                'modules' => $modules,
                'theme_colors' => $theme_colors,
                'email_settings' => $email_settings,
                'sms_settings' => $sms_settings,
                'mail_drivers' => $mail_drivers,
                'allow_superadmin_email_settings' => $allow_superadmin_email_settings,
                'custom_labels' => $custom_labels,
                'common_settings' => $common_settings,
                'weighing_scale_setting' => $weighing_scale_setting
            ];

            return $this->sendResponse($data, 'Business settings retrieved successfully');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong', [], 500);
        }
    }

    /**
     * Update business settings
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function updateBusinessSettings(Request $request): JsonResponse
    {
        try {
            if (!auth()->user()->can('business_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $notAllowed = $this->businessUtil->notAllowedInDemo();
            if (!empty($notAllowed)) {
                return $this->sendError('Not allowed in demo mode', [], 403);
            }

            $business_id = $request->user()->business_id ?? session('user.business_id');

            if (!$business_id) {
                return $this->sendError('Business ID not found', [], 400);
            }

            $business = Business::find($business_id);

            if (!$business) {
                return $this->sendError('Business not found', [], 404);
            }

            $business_details = $request->only([
                'name',
                'start_date',
                'currency_id',
                'tax_label_1',
                'tax_number_1',
                'tax_label_2',
                'tax_number_2',
                'default_profit_percent',
                'default_sales_tax',
                'default_sales_discount',
                'sell_price_tax',
                'sku_prefix',
                'time_zone',
                'fy_start_month',
                'accounting_method',
                'transaction_edit_days',
                'sales_cmsn_agnt',
                'item_addition_method',
                'currency_symbol_placement',
                'on_product_expiry',
                'stop_selling_before',
                'default_unit',
                'expiry_type',
                'date_format',
                'time_format',
                'ref_no_prefixes',
                'theme_color',
                'email_settings',
                'sms_settings',
                'rp_name',
                'amount_for_unit_rp',
                'min_order_total_for_rp',
                'max_rp_per_order',
                'redeem_amount_per_unit_rp',
                'min_order_total_for_redeem',
                'min_redeem_point',
                'max_redeem_point',
                'rp_expiry_period',
                'rp_expiry_type',
                'custom_labels',
                'weighing_scale_setting'
            ]);

            // Traitement des récompenses
            $business_details['enable_rp'] = $request->input('enable_rp') == 1 ? 1 : 0;

            // Formatage des nombres
            $business_details['amount_for_unit_rp'] = !empty($business_details['amount_for_unit_rp'])
                ? $this->businessUtil->num_uf($business_details['amount_for_unit_rp']) : 1;
            $business_details['min_order_total_for_rp'] = !empty($business_details['min_order_total_for_rp'])
                ? $this->businessUtil->num_uf($business_details['min_order_total_for_rp']) : 1;
            $business_details['redeem_amount_per_unit_rp'] = !empty($business_details['redeem_amount_per_unit_rp'])
                ? $this->businessUtil->num_uf($business_details['redeem_amount_per_unit_rp']) : 1;
            $business_details['min_order_total_for_redeem'] = !empty($business_details['min_order_total_for_redeem'])
                ? $this->businessUtil->num_uf($business_details['min_order_total_for_redeem']) : 1;

            $business_details['default_profit_percent'] = !empty($business_details['default_profit_percent'])
                ? $this->businessUtil->num_uf($business_details['default_profit_percent']) : 0;

            $business_details['default_sales_discount'] = !empty($business_details['default_sales_discount'])
                ? $this->businessUtil->num_uf($business_details['default_sales_discount']) : 0;

            // Formatage de la date
            if (!empty($business_details['start_date'])) {
                $business_details['start_date'] = $this->businessUtil->uf_date($business_details['start_date']);
            }

            // Traitement des checkboxes
            $business_details['enable_tooltip'] = $request->input('enable_tooltip') == 1 ? 1 : 0;
            $business_details['enable_product_expiry'] = $request->input('enable_product_expiry') == 1 ? 1 : 0;

            if ($business_details['on_product_expiry'] == 'keep_selling') {
                $business_details['stop_selling_before'] = 0;
            }

            $business_details['stock_expiry_alert_days'] = !empty($request->input('stock_expiry_alert_days'))
                ? $request->input('stock_expiry_alert_days') : 30;

            // Devise d'achat différente
            if ($request->input('purchase_in_diff_currency') == 1) {
                $business_details['purchase_in_diff_currency'] = 1;
                $business_details['purchase_currency_id'] = $request->input('purchase_currency_id');
                $business_details['p_exchange_rate'] = $request->input('p_exchange_rate');
            } else {
                $business_details['purchase_in_diff_currency'] = 0;
                $business_details['purchase_currency_id'] = null;
                $business_details['p_exchange_rate'] = 1;
            }

            // Upload du logo
            $logo_name = $this->businessUtil->uploadFile($request, 'business_logo', 'business_logos', 'image');
            if (!empty($logo_name)) {
                $business_details['logo'] = $logo_name;
            }

            // Traitement des checkboxes multiples
            $checkboxes = [
                'enable_editing_product_from_purchase',
                'enable_inline_tax',
                'enable_brand',
                'enable_category',
                'enable_sub_category',
                'enable_price_tax',
                'enable_purchase_status',
                'enable_lot_number',
                'enable_racks',
                'enable_row',
                'enable_position',
                'enable_sub_units'
            ];

            foreach ($checkboxes as $value) {
                $business_details[$value] = $request->input($value) == 1 ? 1 : 0;
            }

            // Raccourcis clavier
            $shortcuts = $request->input('shortcuts', []);
            $business_details['keyboard_shortcuts'] = json_encode($shortcuts);

            // Paramètres POS
            $pos_settings = $request->input('pos_settings', []);
            $default_pos_settings = $this->businessUtil->defaultPosSettings();
            foreach ($default_pos_settings as $key => $value) {
                if (!isset($pos_settings[$key])) {
                    $pos_settings[$key] = $value;
                }
            }
            $business_details['pos_settings'] = json_encode($pos_settings);

            // Étiquettes personnalisées
            $business_details['custom_labels'] = json_encode($business_details['custom_labels'] ?? []);

            // Paramètres communs
            $business_details['common_settings'] = $request->input('common_settings', []);

            // Modules activés
            $enabled_modules = $request->input('enabled_modules');
            $business_details['enabled_modules'] = !empty($enabled_modules) ? $enabled_modules : null;

            // Mise à jour du logo si fourni
            if (!empty($business_details['logo'])) {
                $business->logo = $business_details['logo'];
            } else {
                unset($business_details['logo']);
            }

            // Sauvegarde
            $business->fill($business_details);
            $business->save();

            // Pour une API, nous pourrions vouloir retourner les données mises à jour
            // plutôt que de gérer les sessions
            $currency = Currency::find($business->currency_id);
            $financial_year = $this->businessUtil->getCurrentFinancialYear($business->id);

            $response_data = [
                'business' => $business->fresh(),
                'currency' => [
                    'id' => $currency->id,
                    'code' => $currency->code,
                    'symbol' => $currency->symbol,
                    'thousand_separator' => $currency->thousand_separator,
                    'decimal_separator' => $currency->decimal_separator,
                ],
                'financial_year' => $financial_year
            ];

            return $this->sendResponse($response_data, 'Business settings updated successfully');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . " Line:" . $e->getLine() . " Message:" . $e->getMessage());
            return $this->sendError('Something went wrong', [], 500);
        }
    }

    /**
     * Handles the validation email
     *
     * @return \Illuminate\Http\Response
     */
    public function postCheckEmail(Request $request)
    {
        $email = $request->input('email');

        $query = User::where('email', $email);

        if (!empty($request->input('user_id'))) {
            $user_id = $request->input('user_id');
            $query->where('id', '!=', $user_id);
        }

        $exists = $query->exists();
        if (!$exists) {
            echo "true";
            exit;
        } else {
            echo "false";
            exit;
        }
    }

    public function getEcomSettings()
    {
        try {
            $api_token = request()->header('API-TOKEN');
            $api_settings = $this->moduleUtil->getApiSettings($api_token);

            $settings = Business::where('id', $api_settings->business_id)
                ->value('ecom_settings');

            $settings_array = !empty($settings) ? json_decode($settings, true) : [];

            if (!empty($settings_array['slides'])) {
                foreach ($settings_array['slides'] as $key => $value) {
                    $settings_array['slides'][$key]['image_url'] = !empty($value['image']) ? url('uploads/img/' . $value['image']) : '';
                }
            }
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());

            return $this->respondWentWrong($e);
        }

        return $this->respond($settings_array);
    }

    /**
     * Handles the testing of email configuration
     *
     * @return \Illuminate\Http\Response
     */
    public function testEmailConfiguration(Request $request)
    {
        try {
            $email_settings = $request->input();

            $data['email_settings'] = $email_settings;
            \Notification::route('mail', $email_settings['mail_from_address'])
                ->notify(new TestEmailNotification($data));

            $output = [
                'success' => 1,
                'msg' => __('lang_v1.email_tested_successfully')
            ];
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            $output = [
                'success' => 0,
                'msg' => $e->getMessage()
            ];
        }

        return $output;
    }

    /**
     * Handles the testing of sms configuration
     *
     * @return \Illuminate\Http\Response
     */
    public function testSmsConfiguration(Request $request)
    {
        try {
            $sms_settings = $request->input();

            $data = [
                'sms_settings' => $sms_settings,
                'mobile_number' => $sms_settings['test_number'],
                'sms_body' => 'This is a test SMS',
            ];
            if (!empty($sms_settings['test_number'])) {
                $response = $this->businessUtil->sendSms($data);
            } else {
                $response = __('lang_v1.test_number_is_required');
            }

            $output = [
                'success' => 1,
                'msg' => $response
            ];
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            $output = [
                'success' => 0,
                'msg' => $e->getMessage()
            ];
        }

        return $output;
    }
}
