<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Brand;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class BrandController extends BaseController
{
    protected $moduleUtil;

    public function __construct(ModuleUtil $moduleUtil)
    {
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of brands
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $query = Brand::where('business_id', $business_id)
                ->select(['name', 'description', 'id']);

            // Apply search filter
            if ($request->has('search')) {
                $search = $request->get('search');
                $query->where('name', 'like', "%{$search}%");
            }

            $brands = $query->orderBy('name')
                ->paginate($request->get('per_page', 15));

            return $this->sendResponse($brands, 'Brands retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Brand index error: ' . $e->getMessage());
            return $this->sendError('Error retrieving brands', [], 500);
        }
    }

    /**
     * Store a newly created brand
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:brands,name,NULL,id,business_id,' . ($request->header('business-id') ?? $request->user()->business_id),
            'description' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $input = $request->only(['name', 'description']);
            $input['business_id'] = $business_id;
            $input['created_by'] = $request->user()->id;

            $brand = Brand::create($input);

            return $this->sendResponse($brand, 'Brand created successfully');
        } catch (\Exception $e) {
            Log::error('Brand store error: ' . $e->getMessage());
            return $this->sendError('Error creating brand', [], 500);
        }
    }

    /**
     * Display the specified brand
     * 
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $brand = Brand::where('business_id', $business_id)->find($id);

            if (!$brand) {
                return $this->sendError('Brand not found', [], 404);
            }

            return $this->sendResponse($brand, 'Brand retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Brand show error: ' . $e->getMessage());
            return $this->sendError('Error retrieving brand', [], 500);
        }
    }

    /**
     * Update the specified brand
     * 
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $business_id = $request->header('business-id') ?? $request->user()->business_id;

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255|unique:brands,name,' . $id . ',id,business_id,' . $business_id,
            'description' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error', $validator->errors(), 422);
        }

        try {
            $brand = Brand::where('business_id', $business_id)->find($id);

            if (!$brand) {
                return $this->sendError('Brand not found', [], 404);
            }

            $input = $request->only(['name', 'description']);
            $brand->update($input);

            return $this->sendResponse($brand, 'Brand updated successfully');
        } catch (\Exception $e) {
            Log::error('Brand update error: ' . $e->getMessage());
            return $this->sendError('Error updating brand', [], 500);
        }
    }

    /**
     * Remove the specified brand
     * 
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $business_id = request()->header('business-id') ?? request()->user()->business_id;

            $brand = Brand::where('business_id', $business_id)->find($id);

            if (!$brand) {
                return $this->sendError('Brand not found', [], 404);
            }

            // Check if brand has associated products
            $productsCount = $brand->products()->count();
            if ($productsCount > 0) {
                return $this->sendError('Cannot delete brand with associated products', [], 422);
            }

            $brand->delete();

            return $this->sendResponse([], 'Brand deleted successfully');
        } catch (\Exception $e) {
            Log::error('Brand destroy error: ' . $e->getMessage());
            return $this->sendError('Error deleting brand', [], 500);
        }
    }

    /**
     * Get brands for dropdown
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getForDropdown(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $brands = Brand::where('business_id', $business_id)
                ->select(['id', 'name'])
                ->orderBy('name')
                ->get();

            return $this->sendResponse($brands, 'Brands for dropdown retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Brand dropdown error: ' . $e->getMessage());
            return $this->sendError('Error retrieving brands', [], 500);
        }
    }

    /**
     * Get all brands (without pagination) for API consumption
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAllBrands(Request $request)
    {
        try {
            $business_id = $request->header('business-id') ?? $request->user()->business_id;

            $brands = Brand::where('business_id', $business_id)
                ->select(['id', 'name', 'description'])
                ->orderBy('name')
                ->get();

            return $this->sendResponse($brands, 'All brands retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Get all brands error: ' . $e->getMessage());
            return $this->sendError('Error retrieving brands', [], 500);
        }
    }
    
}
