<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Barcode;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class BarcodeController extends BaseController
{
    /**
     * Display a listing of the resource.
     * GET /api/barcodes
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        try {
            if (!auth()->user()->can('barcode_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            $barcodes = Barcode::where('business_id', $business_id)
                ->select(['name', 'description', 'id', 'is_default'])
                ->get();

            return $this->sendResponse($barcodes, 'Barcodes retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving barcodes.', $e->getMessage(), 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     * GET /api/barcodes/create
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create()
    {
        if (!auth()->user()->can('barcode_settings.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $data = [
            'fields' => [
                'name' => 'required|string',
                'description' => 'nullable|string',
                'width' => 'required|numeric',
                'height' => 'required|numeric',
                'top_margin' => 'required|numeric',
                'left_margin' => 'required|numeric',
                'row_distance' => 'required|numeric',
                'col_distance' => 'required|numeric',
                'stickers_in_one_row' => 'required|integer',
                'paper_width' => 'required|numeric',
                'paper_height' => 'required|numeric',
                'stickers_in_one_sheet' => 'required|integer',
                'is_default' => 'nullable|boolean',
                'is_continuous' => 'nullable|boolean'
            ]
        ];

        return $this->sendResponse($data, 'Barcode creation form data.');
    }

    /**
     * Store a newly created resource in storage.
     * POST /api/barcodes
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('barcode_settings.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $input = $request->only([
                'name',
                'description',
                'width',
                'height',
                'top_margin',
                'left_margin',
                'row_distance',
                'col_distance',
                'stickers_in_one_row',
                'paper_width'
            ]);

            $business_id = auth()->user()->business_id;
            $input['business_id'] = $business_id;

            if (!empty($request->input('is_default'))) {
                // Reset default
                Barcode::where('business_id', $business_id)
                    ->where('is_default', 1)
                    ->update(['is_default' => 0]);
                $input['is_default'] = 1;
            }

            if (!empty($request->input('is_continuous'))) {
                $input['is_continuous'] = 1;
                $input['stickers_in_one_sheet'] = 28;
            } else {
                $input['stickers_in_one_sheet'] = $request->input('stickers_in_one_sheet');
                $input['paper_height'] = $request->input('paper_height');
            }

            $barcode = Barcode::create($input);

            return $this->sendResponse($barcode, 'Barcode added successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Display the specified resource.
     * GET /api/barcodes/{id}
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            if (!auth()->user()->can('barcode_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $barcode = Barcode::where('business_id', $business_id)->find($id);

            if (!$barcode) {
                return $this->sendError('Barcode not found.', [], 404);
            }

            return $this->sendResponse($barcode, 'Barcode retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving barcode.', $e->getMessage(), 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     * GET /api/barcodes/{id}/edit
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($id)
    {
        try {
            if (!auth()->user()->can('barcode_settings.access')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $barcode = Barcode::where('business_id', $business_id)->find($id);

            if (!$barcode) {
                return $this->sendError('Barcode not found.', [], 404);
            }

            $data = [
                'barcode' => $barcode,
                'fields' => [
                    'name' => 'required|string',
                    'description' => 'nullable|string',
                    'width' => 'required|numeric',
                    'height' => 'required|numeric',
                    'top_margin' => 'required|numeric',
                    'left_margin' => 'required|numeric',
                    'row_distance' => 'required|numeric',
                    'col_distance' => 'required|numeric',
                    'stickers_in_one_row' => 'required|integer',
                    'paper_width' => 'required|numeric',
                    'paper_height' => 'required|numeric',
                    'stickers_in_one_sheet' => 'required|integer',
                    'is_continuous' => 'nullable|boolean'
                ]
            ];

            return $this->sendResponse($data, 'Barcode edit data retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving barcode for edit.', $e->getMessage(), 500);
        }
    }

    /**
     * Update the specified resource in storage.
     * PUT/PATCH /api/barcodes/{id}
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('barcode_settings.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $input = $request->only([
                'name',
                'description',
                'width',
                'height',
                'top_margin',
                'left_margin',
                'row_distance',
                'col_distance',
                'stickers_in_one_row',
                'paper_width'
            ]);

            if (!empty($request->input('is_continuous'))) {
                $input['is_continuous'] = 1;
                $input['stickers_in_one_sheet'] = 28;
                $input['paper_height'] = 0;
            } else {
                $input['is_continuous'] = 0;
                $input['stickers_in_one_sheet'] = $request->input('stickers_in_one_sheet');
                $input['paper_height'] = $request->input('paper_height');
            }

            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $barcode = Barcode::where('business_id', $business_id)->where('id', $id)->first();

            if (!$barcode) {
                return $this->sendError('Barcode not found.', [], 404);
            }

            $barcode->update($input);

            return $this->sendResponse($barcode, 'Barcode updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     * DELETE /api/barcodes/{id}
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('barcode_settings.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);
            $barcode = Barcode::where('business_id', $business_id)->find($id);

            if (!$barcode) {
                return $this->sendError('Barcode not found.', [], 404);
            }

            if ($barcode->is_default != 1) {
                $barcode->delete();
                return $this->sendResponse([], 'Barcode deleted successfully.');
            } else {
                return $this->sendError('Cannot delete default barcode.', [], 400);
            }
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }

    /**
     * Sets barcode setting as default
     * POST /api/barcodes/{id}/set-default
     *
     * @param  int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function setDefault($id)
    {
        if (!auth()->user()->can('barcode_settings.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $this->business_id ?? (auth()->user()->business_id ?? null);

            // Reset current default
            Barcode::where('business_id', $business_id)
                ->where('is_default', 1)
                ->update(['is_default' => 0]);

            $barcode = Barcode::where('business_id', $business_id)->find($id);

            if (!$barcode) {
                return $this->sendError('Barcode not found.', [], 404);
            }

            $barcode->is_default = 1;
            $barcode->save();

            return $this->sendResponse($barcode, 'Default barcode set successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', $e->getMessage(), 500);
        }
    }
}
