<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Utils\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Storage;

class BackUpController extends BaseController
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of the resource.
     * GET /api/backups
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        try {
            if (!auth()->user()->can('backup')) {
                return $this->sendError('Unauthorized action.', [], 403);
            }

            $disk = Storage::disk(config('backup.backup.destination.disks')[0]);
            $files = $disk->files(config('backup.backup.name'));

            $backups = [];
            foreach ($files as $k => $f) {
                if (substr($f, -4) == '.zip' && $disk->exists($f)) {
                    $backups[] = [
                        'file_path' => $f,
                        'file_name' => str_replace(config('backup.backup.name') . '/', '', $f),
                        'file_size' => $disk->size($f),
                        'last_modified' => $disk->lastModified($f),
                    ];
                }
            }
            $backups = array_reverse($backups);

            $cron_job_command = $this->commonUtil->getCronJobCommand();

            $data = [
                'backups' => $backups,
                'cron_job_command' => $cron_job_command
            ];

            return $this->sendResponse($data, 'Backups retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving backups.', $e->getMessage(), 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     * GET /api/backups/create
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function create()
    {
        if (!auth()->user()->can('backup')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $data = [
            'message' => 'Ready to create backup',
            'available_commands' => ['backup:run']
        ];

        return $this->sendResponse($data, 'Backup creation form data.');
    }

    /**
     * Store a newly created resource in storage.
     * POST /api/backups
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('backup')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            // Disable in demo
            $notAllowed = $this->commonUtil->notAllowedInDemo();
            if (!empty($notAllowed)) {
                return $this->sendError('Feature disabled in demo!!', [], 403);
            }

            // Start the backup process
            Artisan::call('backup:run');
            $output = Artisan::output();

            // Log the results
            Log::info("Backpack\BackupManager -- new backup started from API interface \r\n" . $output);

            return $this->sendResponse(['output' => $output], 'Backup created successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Backup creation failed.', $e->getMessage(), 500);
        }
    }

    /**
     * Display the specified resource.
     * GET /api/backups/{filename}
     *
     * @param  string  $filename
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($filename)
    {
        if (!auth()->user()->can('backup')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $file = config('backup.backup.name') . '/' . $filename;
            $disk = Storage::disk(config('backup.backup.destination.disks')[0]);

            if ($disk->exists($file)) {
                $data = [
                    'file_name' => $filename,
                    'file_path' => $file,
                    'file_size' => $disk->size($file),
                    'last_modified' => $disk->lastModified($file),
                    'download_url' => route('api.backups.download', $filename)
                ];

                return $this->sendResponse($data, 'Backup file details retrieved.');
            } else {
                return $this->sendError("The backup file doesn't exist.", [], 404);
            }
        } catch (\Exception $e) {
            return $this->sendError('Error retrieving backup details.', $e->getMessage(), 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     * GET /api/backups/{filename}/edit
     *
     * @param  string  $filename
     * @return \Illuminate\Http\JsonResponse
     */
    public function edit($filename)
    {
        if (!auth()->user()->can('backup')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        return $this->sendError('Backup files cannot be edited.', [], 405);
    }

    /**
     * Update the specified resource in storage.
     * PUT/PATCH /api/backups/{filename}
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  string  $filename
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $filename)
    {
        if (!auth()->user()->can('backup')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        return $this->sendError('Backup files cannot be updated.', [], 405);
    }

    /**
     * Remove the specified resource from storage.
     * DELETE /api/backups/{filename}
     *
     * @param  string  $filename
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($filename)
    {
        if (!auth()->user()->can('backup')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            // Disable in demo
            if (config('app.env') == 'demo') {
                return $this->sendError('Feature disabled in demo!!', [], 403);
            }

            $disk = Storage::disk(config('backup.backup.destination.disks')[0]);
            $filePath = config('backup.backup.name') . '/' . $filename;

            if ($disk->exists($filePath)) {
                $disk->delete($filePath);
                return $this->sendResponse([], 'Backup file deleted successfully.');
            } else {
                return $this->sendError("The backup file doesn't exist.", [], 404);
            }
        } catch (\Exception $e) {
            return $this->sendError('Error deleting backup file.', $e->getMessage(), 500);
        }
    }

    /**
     * Download a backup file.
     * GET /api/backups/{filename}/download
     *
     * @param  string  $filename
     * @return \Illuminate\Http\Response
     */
    public function download($filename)
    {
        if (!auth()->user()->can('backup')) {
            abort(403, 'Unauthorized action.');
        }

        // Disable in demo
        if (config('app.env') == 'demo') {
            return $this->sendError('Feature disabled in demo!!', [], 403);
        }

        $file = config('backup.backup.name') . '/' . $filename;
        $disk = Storage::disk(config('backup.backup.destination.disks')[0]);

        if ($disk->exists($file)) {
            $fs = Storage::disk(config('backup.backup.destination.disks')[0])->getDriver();
            $stream = $fs->readStream($file);

            return Response::stream(function () use ($stream) {
                fpassthru($stream);
            }, 200, [
                "Content-Type" => $fs->getMimetype($file),
                "Content-Length" => $fs->getSize($file),
                "Content-disposition" => "attachment; filename=\"" . basename($file) . "\"",
            ]);
        } else {
            return $this->sendError("The backup file doesn't exist.", [], 404);
        }
    }
}
