<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class AuthentificationController extends BaseController
{
    /**
     * Login API
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request): JsonResponse
    {
        try {
            // Validation des données d'entrée
            $validator = Validator::make($request->all(), [
                'username' => 'required|string|min:3|max:255',
                'password' => 'required|string|min:6',
            ]);

            // Si la validation échoue
            if ($validator->fails()) {
                return $this->sendError(
                    'Données de validation invalides',
                    $validator->errors()->toArray(),
                    422
                );
            }

            // Récupération de l'utilisateur
            $user = User::where('username', $request->username)->first();

            if (!$user) {
                return $this->sendError(
                    'Nom d\'utilisateur incorrect',
                    ['error' => 'Aucun compte trouvé avec ce nom d\'utilisateur.'],
                    401
                );
            }

            // Vérification du mot de passe avec Hash::check
            if (!Hash::check($request->password, $user->password)) {
                return $this->sendError(
                    'Mot de passe incorrect',
                    ['error' => 'Le mot de passe saisi est incorrect.'],
                    401
                );
            }

            // Authentifier l'utilisateur manuellement
            Auth::login($user);

            // Vérification si l'utilisateur est actif
            if (isset($user->is_active) && !$user->is_active) {
                return $this->sendError(
                    'Compte désactivé',
                    ['error' => 'Votre compte a été désactivé. Contactez l\'administrateur.'],
                    403
                );
            }

            // Vérification si l'utilisateur a un business_id
            if (empty($user->business_id)) {
                return $this->sendError(
                    'Configuration incomplète',
                    ['error' => 'Votre compte n\'est pas associé à une entreprise. Contactez l\'administrateur.'],
                    403
                );
            }

            // Suppression des anciens tokens
            $user->tokens()->delete();

            // Création du nouveau token avec Sanctum
            $tokenResult = $user->createToken('SIGH-API-Token', ['*']);
            $token = $tokenResult->plainTextToken;

            // Préparation des données de succès
            $success = [
                'token' => $token,
                'token_type' => 'Bearer',
                'expires_at' => null,
                'user' => [
                    'id' => $user->id,
                    'username' => $user->username,
                    'first_name' => $user->first_name ?? '',
                    'last_name' => $user->last_name ?? '',
                    'email' => $user->email ?? '',
                    'business_id' => $user->business_id,
                    'is_active' => $user->is_active ?? true,
                    'permissions' => $user->getAllPermissions()->pluck('name') ?? [],
                ]
            ];

            $message = __('messages.login_data_loaded') ?? 'Connexion réussie';

            return $this->sendResponse($success, $message);
        } catch (\Exception $e) {
            // Log de l'erreur
            Log::error('Erreur lors de la connexion: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'username' => $request->username ?? 'non fourni'
            ]);

            return $this->sendError(
                'Erreur de connexion',
                ['error' => 'Une erreur inattendue s\'est produite. Veuillez réessayer.'],
                500
            );
        }
    }

    /**
     * Alternative: Login avec email ou username
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function loginWithEmailOrUsername(Request $request): JsonResponse
    {
        try {
            // Validation flexible pour email ou username
            $validator = Validator::make($request->all(), [
                'login' => 'required|string|min:3|max:255', // peut être email ou username
                'password' => 'required|string|min:6',
            ]);

            if ($validator->fails()) {
                return $this->sendError(
                    'Données de validation invalides',
                    $validator->errors()->toArray(),
                    422
                );
            }

            $login = $request->login;
            $password = $request->password;

            // Déterminer si c'est un email ou un username
            $loginType = filter_var($login, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';

            // Tentative de connexion
            if (Auth::attempt([$loginType => $login, 'password' => $password])) {
                $user = Auth::user();

                // Vérifications supplémentaires (même logique que ci-dessus)
                if (isset($user->is_active) && !$user->is_active) {
                    Auth::logout();
                    return $this->sendError(
                        'Compte désactivé',
                        ['error' => 'Votre compte a été désactivé. Contactez l\'administrateur.'],
                        403
                    );
                }

                // Suppression des anciens tokens
                $user->tokens()->delete();

                // Création du nouveau token avec Sanctum
                $tokenResult = $user->createToken('SIGH-API-Token', ['*']);
                $token = $tokenResult->plainTextToken;

                $success = [
                    'token' => $token,
                    'token_type' => 'Bearer',
                    'expires_at' => null, // Sanctum n'a pas d'expiration par défaut
                    'user' => [
                        'id' => $user->id,
                        'username' => $user->username,
                        'first_name' => $user->first_name ?? '',
                        'last_name' => $user->last_name ?? '',
                        'email' => $user->email ?? '',
                        'business_id' => $user->business_id,
                    ]
                ];

                return $this->sendResponse($success, 'Connexion réussie');
            } else {
                return $this->sendError(
                    'Identifiants incorrects',
                    ['error' => 'Le nom d\'utilisateur/email ou le mot de passe est incorrect.'],
                    401
                );
            }
        } catch (\Exception $e) {
            \Log::error('Erreur lors de la connexion: ' . $e->getMessage());

            return $this->sendError(
                'Erreur de connexion',
                ['error' => 'Une erreur inattendue s\'est produite. Veuillez réessayer.'],
                500
            );
        }
    }

    /**
     * Logout API pour Sanctum
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout(Request $request): JsonResponse
    {
        try {
            // Supprimer le token actuel
            $request->user()->currentAccessToken()->delete();

            return $this->sendResponse([], 'Déconnexion réussie');
        } catch (\Exception $e) {
            Log::error('Erreur lors de la déconnexion: ' . $e->getMessage());

            return $this->sendError(
                'Erreur de déconnexion',
                ['error' => 'Une erreur s\'est produite lors de la déconnexion.'],
                500
            );
        }
    }

    /**
     * Logout de tous les appareils (supprimer tous les tokens)
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function logoutAll(Request $request): JsonResponse
    {
        try {
            // Supprimer tous les tokens de l'utilisateur
            $request->user()->tokens()->delete();

            return $this->sendResponse([], 'Déconnexion de tous les appareils réussie');
        } catch (\Exception $e) {
            Log::error('Erreur lors de la déconnexion complète: ' . $e->getMessage());

            return $this->sendError(
                'Erreur de déconnexion',
                ['error' => 'Une erreur s\'est produite lors de la déconnexion.'],
                500
            );
        }
    }

    /**
     * Vérification de l'état du token
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function me(Request $request): JsonResponse
    {
        try {
            $user = $request->user();
            $token = $request->user()->currentAccessToken();

            $data = [
                'user' => [
                    'id' => $user->id,
                    'username' => $user->username,
                    'first_name' => $user->first_name ?? '',
                    'last_name' => $user->last_name ?? '',
                    'email' => $user->email ?? '',
                    'business_id' => $user->business_id,
                    'is_active' => $user->is_active ?? true,
                ],
                'token_info' => [
                    'name' => $token->name,
                    'abilities' => $token->abilities,
                    'created_at' => $token->created_at,
                    'last_used_at' => $token->last_used_at,
                ]
            ];

            return $this->sendResponse($data, 'Informations utilisateur récupérées');
        } catch (\Exception $e) {
            return $this->sendError(
                'Erreur de récupération',
                ['error' => 'Impossible de récupérer les informations utilisateur.'],
                500
            );
        }
    }
}
