<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\AccountType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class AccountTypeController extends BaseController
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $account_types = AccountType::where('business_id', $business_id)
                ->with(['sub_types', 'parent_account_type'])
                ->get();

            return $this->sendResponse($account_types, 'Account types retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Show the form for creating a new resource (API version returns form data).
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $account_types = AccountType::where('business_id', $business_id)
                ->whereNull('parent_account_type_id')
                ->get();

            return $this->sendResponse(['parent_account_types' => $account_types], 'Account type creation form data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'parent_account_type_id' => 'nullable|exists:account_types,id'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error.', $validator->errors(), 422);
        }

        try {
            $input = $request->only(['name', 'parent_account_type_id']);
            $input['business_id'] = $request->user()->business_id;

            $account_type = AccountType::create($input);

            return $this->sendResponse($account_type, 'Account type created successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $account_type = AccountType::where('business_id', $business_id)
                ->with(['sub_types', 'parent_account_type'])
                ->findOrFail($id);

            return $this->sendResponse($account_type, 'Account type retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Account type not found.', [], 404);
        }
    }

    /**
     * Show the form for editing the specified resource (API version returns edit form data).
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $account_type = AccountType::where('business_id', $business_id)->findOrFail($id);

            $parent_account_types = AccountType::where('business_id', $business_id)
                ->whereNull('parent_account_type_id')
                ->where('id', '!=', $id) // Exclude current account type
                ->get();

            $data = [
                'account_type' => $account_type,
                'parent_account_types' => $parent_account_types
            ];

            return $this->sendResponse($data, 'Account type edit form data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Account type not found.', [], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'parent_account_type_id' => 'nullable|exists:account_types,id'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error.', $validator->errors(), 422);
        }

        try {
            $input = $request->only(['name', 'parent_account_type_id']);
            $business_id = $request->user()->business_id;

            $account_type = AccountType::where('business_id', $business_id)->findOrFail($id);

            //Account type is changed to subtype update all its sub type's parent type
            if (empty($account_type->parent_account_type_id) && !empty($input['parent_account_type_id'])) {
                AccountType::where('business_id', $business_id)
                    ->where('parent_account_type_id', $account_type->id)
                    ->update(['parent_account_type_id' => $input['parent_account_type_id']]);
            }

            $account_type->update($input);

            return $this->sendResponse($account_type, 'Account type updated successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $account_type = AccountType::where('business_id', $business_id)->findOrFail($id);

            // Check if account type has associated accounts
            $accounts_count = $account_type->accounts()->count();
            if ($accounts_count > 0) {
                return $this->sendError('Cannot delete account type. It has associated accounts.', [], 422);
            }

            $account_type->delete();

            //Update parent account if set
            AccountType::where('business_id', $business_id)
                ->where('parent_account_type_id', $id)
                ->update(['parent_account_type_id' => null]);

            return $this->sendResponse([], 'Account type deleted successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }
}
