<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Api\BaseController;
use App\Models\Account;
use App\Models\AccountTransaction;
use App\Models\TransactionPayment;
use App\Utils\TransactionUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class AccountReportsController extends BaseController
{
    /**
     * All Utils instance.
     *
     */
    protected $transactionUtil;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(TransactionUtil $transactionUtil)
    {
        $this->transactionUtil = $transactionUtil;
    }

    /**
     * Display balance sheet report.
     * @return Response
     */
    public function balanceSheet(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $end_date = !empty($request->input('end_date')) ? $this->transactionUtil->uf_date($request->input('end_date')) : Carbon::now()->format('Y-m-d');

            $purchase_details = $this->transactionUtil->getPurchaseTotals(
                $business_id,
                null,
                $end_date
            );
            $sell_details = $this->transactionUtil->getSellTotals(
                $business_id,
                null,
                $end_date
            );

            $transaction_types = ['sell_return'];

            $sell_return_details = $this->transactionUtil->getTransactionTotals(
                $business_id,
                $transaction_types,
                null,
                $end_date
            );

            $account_details = $this->getAccountBalance($business_id, $end_date);

            //Get Closing stock
            $closing_stock = $this->transactionUtil->getOpeningClosingStock(
                $business_id,
                $end_date,
                null
            );

            $output = [
                'supplier_due' => $purchase_details['purchase_due'],
                'customer_due' => $sell_details['invoice_due'] - $sell_return_details['total_sell_return_inc_tax'],
                'account_balances' => $account_details,
                'closing_stock' => $closing_stock,
                'capital_account_details' => null,
                'end_date' => $end_date
            ];

            return $this->sendResponse($output, 'Balance sheet retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Display trial balance report.
     * @return Response
     */
    public function trialBalance(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $end_date = !empty($request->input('end_date')) ? $this->transactionUtil->uf_date($request->input('end_date')) : Carbon::now()->format('Y-m-d');

            $purchase_details = $this->transactionUtil->getPurchaseTotals(
                $business_id,
                null,
                $end_date
            );
            $sell_details = $this->transactionUtil->getSellTotals(
                $business_id,
                null,
                $end_date
            );

            $account_details = $this->getAccountBalance($business_id, $end_date);

            $output = [
                'supplier_due' => $purchase_details['purchase_due'],
                'customer_due' => $sell_details['invoice_due'],
                'account_balances' => $account_details,
                'capital_account_details' => null,
                'end_date' => $end_date
            ];

            return $this->sendResponse($output, 'Trial balance retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Retrives account balances.
     * @return Obj
     */
    private function getAccountBalance($business_id, $end_date, $account_type = 'others')
    {
        $query = Account::leftjoin(
            'account_transactions as AT',
            'AT.account_id',
            '=',
            'accounts.id'
        )
            ->whereNull('AT.deleted_at')
            ->where('business_id', $business_id)
            ->whereDate('AT.operation_date', '<=', $end_date);

        $account_details = $query->select([
            'name',
            DB::raw("SUM( IF(AT.type='credit', amount, -1*amount) ) as balance")
        ])
            ->groupBy('accounts.id')
            ->get()
            ->pluck('balance', 'name');

        return $account_details;
    }

    /**
     * Displays payment account report.
     * @return Response
     */
    public function paymentAccountReport(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;

            $query = TransactionPayment::leftjoin(
                'transactions as T',
                'transaction_payments.transaction_id',
                '=',
                'T.id'
            )
                ->leftjoin('accounts as A', 'transaction_payments.account_id', '=', 'A.id')
                ->where('transaction_payments.business_id', $business_id)
                ->whereNull('transaction_payments.parent_id')
                ->where('transaction_payments.method', '!=', 'advance')
                ->select([
                    'paid_on',
                    'payment_ref_no',
                    'T.ref_no',
                    'T.invoice_no',
                    'T.type',
                    'T.id as transaction_id',
                    'A.name as account_name',
                    'A.account_number',
                    'transaction_payments.id as payment_id',
                    'transaction_payments.account_id',
                    'transaction_payments.amount'
                ]);

            $start_date = !empty($request->input('start_date')) ? $request->input('start_date') : '';
            $end_date = !empty($request->input('end_date')) ? $request->input('end_date') : '';

            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween(DB::raw('date(paid_on)'), [$start_date, $end_date]);
            }

            $account_id = !empty($request->input('account_id')) ? $request->input('account_id') : '';

            if ($account_id == 'none') {
                $query->whereNull('account_id');
            } elseif (!empty($account_id)) {
                $query->where('account_id', $account_id);
            }

            $payments = $query->get();
            $accounts = Account::forDropdown($business_id, false);

            $data = [
                'payments' => $payments,
                'accounts' => $accounts,
                'filters' => [
                    'start_date' => $start_date,
                    'end_date' => $end_date,
                    'account_id' => $account_id
                ]
            ];

            return $this->sendResponse($data, 'Payment account report retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }

    /**
     * Shows data for linking account with a payment.
     * @return Response
     */
    public function getLinkAccount(Request $request, $id)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        try {
            $business_id = $request->user()->business_id;
            $payment = TransactionPayment::where('business_id', $business_id)->findOrFail($id);
            $accounts = Account::forDropdown($business_id, false);

            $data = [
                'payment' => $payment,
                'accounts' => $accounts
            ];

            return $this->sendResponse($data, 'Link account data retrieved successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Payment not found.', [], 404);
        }
    }

    /**
     * Links account with a payment.
     * @param  Request $request
     * @return Response
     */
    public function postLinkAccount(Request $request)
    {
        if (!auth()->user()->can('account.access')) {
            return $this->sendError('Unauthorized action.', [], 403);
        }

        $validator = Validator::make($request->all(), [
            'transaction_payment_id' => 'required|exists:transaction_payments,id',
            'account_id' => 'required|exists:accounts,id'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error.', $validator->errors(), 422);
        }

        try {
            $business_id = $request->user()->business_id;
            $payment_id = $request->input('transaction_payment_id');
            $account_id = $request->input('account_id');

            $payment = TransactionPayment::with(['transaction'])->where('business_id', $business_id)->findOrFail($payment_id);
            $payment->account_id = $account_id;
            $payment->save();

            $payment_type = !empty($payment->transaction->type) ? $payment->transaction->type : null;
            if (empty($payment_type)) {
                $child_payment = TransactionPayment::where('parent_id', $payment->id)->first();
                $payment_type = !empty($child_payment->transaction->type) ? $child_payment->transaction->type : null;
            }

            AccountTransaction::updateAccountTransaction($payment, $payment_type);

            return $this->sendResponse($payment, 'Account linked successfully.');
        } catch (\Exception $e) {
            Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            return $this->sendError('Something went wrong.', [], 500);
        }
    }
}
